package com.ejianc.business.prjfinance.controller;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.ejianc.business.prjfinance.bean.PrjAdvancePaymentEntity;
import com.ejianc.business.prjfinance.service.IPrjAdvancePaymentService;
import com.ejianc.business.prjfinance.vo.PrjAdvancePaymentVO;
import com.ejianc.business.prjfinance.vo.RiskMarginVO;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.ExcelExport;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 项目代转款
 *
 * @author CJ
 * @Description:
 * @date 2021/7/21 19:31
 */
@RestController
@RequestMapping(value = "/prjAdvancePay/")
public class PrjAdvancePayController {

    @Autowired
    private IPrjAdvancePaymentService prjAdvancePaymentService;

    @Autowired
    private IOrgApi orgApi;

    private final String BILL_CODE_RULE = "PRJ_ADVANCE_PAYMENT_01";

    @Autowired
    private IBillCodeApi billCodeApi;

    @PostMapping(value = "pageList")
    public CommonResponse<JSONObject> pageList(@RequestBody QueryParam queryParam) {
        JSONObject resp = new JSONObject();
        queryParam.getFuzzyFields().add("projectName");
        queryParam.getFuzzyFields().add("applySubject");

        //查询本下
        CommonResponse<List<OrgVO>> childOrgResp = orgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId());
        if(!childOrgResp.isSuccess()) {
            throw new BusinessException("分页列表查询失败, 查询组织信息失败！");
        }
        List<Long> childIds = new ArrayList<>();
        childIds.addAll(childOrgResp.getData().stream().map(OrgVO::getId).collect(Collectors.toList()));
        queryParam.getParams().put("orgId", new Parameter(QueryParam.IN, childIds));

        IPage<PrjAdvancePaymentEntity> pageData = prjAdvancePaymentService.queryPage(queryParam, false);

        resp.put("total", pageData.getTotal());
        resp.put("current", pageData.getCurrent());
        resp.put("size", pageData.getSize());
        resp.put("pages", pageData.getPages());
        resp.put("records", pageData.getRecords());

        //查询项目带转款累计情况
        Map<String, BigDecimal> sumAmt = prjAdvancePaymentService.sumPrjAdvancePayInfo(queryParam);
        resp.put("totalAdvancePayAmt", sumAmt.get("totalAdvancePayAmt"));
        resp.put("totalBackAmt", sumAmt.get("totalBackAmt"));

        return CommonResponse.success(resp);
    }

    /**
     * 项目代转款保存/更新
     *
     * @param vo
     * @return
     */
    @PostMapping(value = "saveOrUpdate")
    public CommonResponse<PrjAdvancePaymentVO> saveOrUpdate(@RequestBody PrjAdvancePaymentVO vo) {
        if(RiskMarginVO.BILL_TYPE_BACK_CASH_DEPOSIT.equals(vo.getBillType())) {
            //退保证金，检查申请金额是否小于等于剩余可退金额
            Map<String, BigDecimal> info = prjAdvancePaymentService.getTotalPrjAdvancePayInfo(vo.getProjectId());
            if(null != vo.getId()) {
                if(info.get("remremainingRefundableAmt").compareTo(BigDecimal.ZERO) < 0) {
                    return CommonResponse.error("保存失败，申请金额大于剩余可退金额。");
                }
            } else if(vo.getAmount().compareTo(info.get("remremainingRefundableAmt")) > 0) {
                return CommonResponse.error("保存失败，申请金额大于剩余可退金额。");
            }
        }
        PrjAdvancePaymentEntity saveEntity = null;
        if(null == vo.getId()) {
            saveEntity = BeanMapper.map(vo, PrjAdvancePaymentEntity.class);
            CommonResponse<String> billCodeResp = billCodeApi.getCodeBatchByRuleCode(BILL_CODE_RULE, InvocationInfoProxy.getTenantid());
            if(!billCodeResp.isSuccess()) {
                return CommonResponse.error("保存失败，自动生成编码失败。");
            }
            saveEntity.setBillCode(billCodeResp.getData());
            saveEntity.setBillState(BillStateEnum.UNCOMMITED_STATE.getBillStateCode());
        } else {
            saveEntity = prjAdvancePaymentService.selectById(vo.getId());
            saveEntity.setProjectId(vo.getProjectId());
            saveEntity.setProjectName(vo.getProjectName());
            saveEntity.setAmount(vo.getAmount());
            saveEntity.setHandlingDate(vo.getHandlingDate());
            saveEntity.setSubject(vo.getSubject());
            saveEntity.setOperatorId(vo.getOperatorId());
            saveEntity.setOperatorName(vo.getOperatorName());
            saveEntity.setOrgId(vo.getOrgId());
            saveEntity.setOrgName(vo.getOrgName());
            saveEntity.setTransactionMode(vo.getTransactionMode());
            saveEntity.setTransactionModeName(vo.getTransactionModeName());
            saveEntity.setRecBankAcc(vo.getRecBankAcc());
            saveEntity.setRecBankAccName(vo.getRecBankAccName());
            saveEntity.setRecOpenAccBankName(vo.getRecOpenAccBankName());
        }

        prjAdvancePaymentService.saveOrUpdate(saveEntity, false);

        PrjAdvancePaymentVO resp = BeanMapper.map(saveEntity, PrjAdvancePaymentVO.class);
        //累计待转款
        resp.setTotalAdvancePayAmt(vo.getTotalAdvancePayAmt());
        //累计退款
        resp.setTotalBackAmt(vo.getTotalBackAmt());
        //剩余待转款
        resp.setRemanentAdvancePayAmt(vo.getRemanentAdvancePayAmt());
        //剩余可退金额
        resp.setRemremainingRefundableAmt(vo.getRemremainingRefundableAmt());
        return CommonResponse.success("保存成功！", resp);
    }

    /**
     * 查询指定的项目代转款详情
     *
     * @param id
     * @return
     */
    @GetMapping(value = "queryDetail")
    public CommonResponse<PrjAdvancePaymentVO> queryDetail(@RequestParam(value = "id") Long id) {
        PrjAdvancePaymentVO resp = null;
        PrjAdvancePaymentEntity entity = prjAdvancePaymentService.selectById(id);
        if(null != entity) {
            resp = BeanMapper.map(entity, PrjAdvancePaymentVO.class);
            Map<String, BigDecimal> info = prjAdvancePaymentService.getTotalPrjAdvancePayInfo(resp.getProjectId());

            //累计待转款
            resp.setTotalAdvancePayAmt(info.get("totalAdvancePayAmt"));
            //累计退款
            resp.setTotalBackAmt(info.get("totalBackAmt"));
            //剩余待转款
            resp.setRemanentAdvancePayAmt(info.get("remanentAdvancePayAmt"));
            //剩余可退金额
            resp.setRemremainingRefundableAmt(info.get("remremainingRefundableAmt"));
        }
        //查询项目风险保证金情况
        return CommonResponse.success(resp);
    }

    /**
     * 项目代转款删除
     *
     * @param vos
     * @return
     */
    @PostMapping(value = "delete")
    public CommonResponse<String> delete(@RequestBody List<PrjAdvancePaymentVO> vos) {
        if(CollectionUtils.isNotEmpty(vos)) {
            prjAdvancePaymentService.removeByIds(vos.stream().map(PrjAdvancePaymentVO::getId).collect(Collectors.toList()), false);
        }
        return CommonResponse.success("删除成功！");
    }


    /**
     * 查询项目代转款累计收款、退款、剩余等信息
     *
     * @param projectId
     * @return
     */
    @GetMapping(value = "getPrjAdvancePayInfo")
    public CommonResponse<Map<String, BigDecimal>> getProjectCashDepositInfo(@RequestParam(value = "projectId") Long projectId) {
        Map<String, BigDecimal> resp = prjAdvancePaymentService.getTotalPrjAdvancePayInfo(projectId);

        return CommonResponse.success(resp);
    }

    /**
     * 根据项目Id查询生效的项目代转款列表
     *
     * @param projectId
     * @return
     */
    @GetMapping(value = "queryList")
    public CommonResponse<JSONObject> queryList(@RequestParam(value = "projectId") Long projectId,
                                                @RequestParam(value = "billType", required = false) String billType) {
        JSONObject resp = new JSONObject();
        List<PrjAdvancePaymentVO> voList = new ArrayList<>();
        QueryParam queryParam = new QueryParam();
        queryParam.getParams().put("projectId", new Parameter(QueryParam.EQ, projectId));
        if(StringUtils.isNotBlank(billType)) {
            queryParam.getParams().put("billType", new Parameter(QueryParam.EQ, billType));
        }

        //查询生效的
        queryParam.getParams().put("billState", new Parameter(QueryParam.IN, Arrays.asList(new Integer[]{BillStateEnum.PASSED_STATE.getBillStateCode(), BillStateEnum.COMMITED_STATE.getBillStateCode()})));
        //按照申请日期倒序排列
        queryParam.getOrderMap().put("handlingDate", "DESC");

        List<PrjAdvancePaymentEntity> eList = prjAdvancePaymentService.queryList(queryParam);
        if(CollectionUtils.isNotEmpty(eList)) {
            voList.addAll(BeanMapper.mapList(eList, PrjAdvancePaymentVO.class));
        }
        Map<String, BigDecimal> info = prjAdvancePaymentService.getTotalPrjAdvancePayInfo(projectId);

        resp.put("records", voList);
        //累计待转款
        resp.put("totalAdvancePayAmt", info.get("totalAdvancePayAmt"));
        //累计退款
        resp.put("totalBackAmt", info.get("totalBackAmt"));
        //剩余待转款
        resp.put("remanentAdvancePayAmt", info.get("remanentAdvancePayAmt"));
        //剩余可退金额
        resp.put("remremainingRefundableAmt", info.get("remremainingRefundableAmt"));
        return CommonResponse.success(resp);
    }

    /**
     * 查询指定项目上一个单据中的收款账户信息
     *
     * @param projectId
     * @return
     */
    @GetMapping(value = "getLastBillAccInfo")
    public CommonResponse<JSONObject> getLastBillAccInfo(@RequestParam(value = "projectId") Long projectId) {
        JSONObject resp = null;

        PrjAdvancePaymentVO bill = prjAdvancePaymentService.getLastBillAccInfo(projectId, PrjAdvancePaymentVO.BILL_TYPE_BACK_PAYMENT);
        if(null != bill) {
            resp = new JSONObject();
            resp.put("recBankAccName", bill.getRecBankAccName());
            resp.put("recBankAcc", bill.getRecBankAcc());
            resp.put("recOpenAccBankName", bill.getRecOpenAccBankName());
        }

        return CommonResponse.success(resp);
    }

    /**
     * 项目代转款列表导出
     *
     * @param queryParam
     * @param response
     */
    @PostMapping(value = "excelExport")
    public void excelExport(@RequestBody QueryParam queryParam, HttpServletResponse response) {
        queryParam.getFuzzyFields().add("projectName");
        queryParam.getFuzzyFields().add("applySubject");
        queryParam.setPageIndex(1);
        queryParam.setPageSize(10000);

        //查询本下
        CommonResponse<List<OrgVO>> childOrgResp = orgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId());
        if(!childOrgResp.isSuccess()) {
            throw new BusinessException("分页列表查询失败, 查询组织信息失败！");
        }

        List<Long> childIds = new ArrayList<>();
        childIds.addAll(childOrgResp.getData().stream().map(OrgVO::getId).collect(Collectors.toList()));
        queryParam.getParams().put("orgId", new Parameter(QueryParam.IN, childIds));
        IPage<PrjAdvancePaymentEntity> pageData = prjAdvancePaymentService.queryPage(queryParam, false);
        List<PrjAdvancePaymentVO> voList = new ArrayList<>();
        if(CollectionUtils.isNotEmpty(pageData.getRecords())) {
            pageData.getRecords().forEach(risk -> {
                PrjAdvancePaymentVO r = BeanMapper.map(risk, PrjAdvancePaymentVO.class);
                r.setBillStateName(BillStateEnum.getEnumByStateCode(r.getBillState()).getDescription());
                r.setBillType(RiskMarginVO.BILL_TYPE_DEDUCT_CASH_DEPOSIT.equals(r.getBillType()) ? "收款登记" : "退款申请");
                voList.add(r);
            });
        }

        Map<String, Object> beans = new HashMap<String, Object>();
        beans.put("records", voList);
        ExcelExport.getInstance().export("prjAdvancePay-export.xlsx", beans, response);
    }


}
