package com.ejianc.business.prjfinance.controller;

import java.io.Serializable;

import com.alibaba.fastjson.JSONObject;
import com.ejianc.business.prjfinance.bean.ReturnMoneyEntity;
import com.ejianc.business.prjfinance.service.IReturnMoneyService;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.collection.ListUtil;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.util.ExcelExport;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.*;

import com.ejianc.framework.core.response.CommonResponse;
import org.springframework.beans.factory.annotation.Autowired;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;

import com.ejianc.business.prjfinance.bean.ProjectLoanEntity;
import com.ejianc.business.prjfinance.service.IProjectLoanService;
import com.ejianc.business.prjfinance.vo.ProjectLoanVO;

import javax.servlet.http.HttpServletResponse;

/**
 * 项目借款
 * 
 * @author generator
 * 
 */
@Controller
@RequestMapping("projectLoan")
public class ProjectLoanController implements Serializable {
	private static final long serialVersionUID = 1L;

    private static final String BILL_CODE = "prjfrance-loan-001";
    private Logger logger = LoggerFactory.getLogger(this.getClass());
    
    @Autowired
    private IProjectLoanService service;
    @Autowired
    private IReturnMoneyService iReturnMoneyService;
    @Autowired
    private IBillCodeApi billCodeApi;
    @Autowired
    private IOrgApi iOrgApi;
    
    /**
     * @Description saveOrUpdate 新增或者修改
     */
    @RequestMapping(value = "/saveOrUpdate", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<ProjectLoanVO> saveOrUpdate(@RequestBody ProjectLoanVO saveorUpdateVO) {
    	ProjectLoanEntity entity = BeanMapper.map(saveorUpdateVO, ProjectLoanEntity.class);
        if(entity.getId() == null || entity.getId() == 0){
            CommonResponse<String> billCode = billCodeApi.getCodeBatchByRuleCode(BILL_CODE, InvocationInfoProxy.getTenantid());
            if(billCode.isSuccess()) {
                entity.setCode(billCode.getData());
            }else{
                throw new BusinessException("网络异常， 编码生成失败， 请稍后再试");
            }
        }
    	service.saveOrUpdate(entity, false);
    	ProjectLoanVO vo = BeanMapper.map(entity, ProjectLoanVO.class);
    	return CommonResponse.success("保存或修改单据成功！",vo);
    }
    
    /**
     * @Description queryDetail 查询详情
     * @param id
     */
    @RequestMapping(value = "/queryDetail", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<ProjectLoanVO> queryDetail(Long id) {
    	ProjectLoanEntity entity = service.selectById(id);
    	ProjectLoanVO vo = BeanMapper.map(entity, ProjectLoanVO.class);
        return CommonResponse.success("查询详情数据成功！",vo);
    }
    
    /**
     * @Description delete 批量删除单据
     * @Param [ids]
     */
    @RequestMapping(value = "/delete", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<String> delete(@RequestBody List<ProjectLoanVO> vos) {
        service.removeByIds(vos.stream().map(ProjectLoanVO::getId).collect(Collectors.toList()),true);
        return CommonResponse.success("删除成功！");
    }
    
    /**
     * @Description queryList 查询列表
     * @param param
     * @Return com.ejianc.framework.core.response.CommonResponse<java.lang.String>
     */
    @RequestMapping(value = "/queryList", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<IPage<ProjectLoanVO>> queryList(@RequestBody QueryParam param) {
        /** 模糊搜索配置字段示例 */
        List<String> fuzzyFields = param.getFuzzyFields();
        fuzzyFields.add("agentName");
        fuzzyFields.add("projectName");
        fuzzyFields.add("name");

        CommonResponse<List<OrgVO>> childOrgResp = iOrgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId());
        if(!childOrgResp.isSuccess()) {
            throw new BusinessException("分页列表查询失败, 查询组织信息失败！");
        }
        List<Long> childIds = new ArrayList<>();
        childIds.addAll(childOrgResp.getData().stream().map(OrgVO::getId).collect(Collectors.toList()));
        param.getParams().put("orgId", new Parameter(QueryParam.IN, childIds));

        /** 租户隔离 */
        param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        
        IPage<ProjectLoanEntity> page = service.queryPage(param,false);
        IPage<ProjectLoanVO> pageData = new Page<>(page.getCurrent(), page.getSize(), page.getTotal());
  		pageData.setRecords(BeanMapper.mapList(page.getRecords(), ProjectLoanVO.class));
        
        return CommonResponse.success("查询列表数据成功！",pageData);
    }

    /**
     * @Author mrsir_wxp
     * @Date 2021/7/20 按根据项目id 查询该项目下已生效的借款累计金额，还款累计金额，累计借款结余等数据
     * @Description queryTotalMoneyData
     * @Param [param]
     * @Return totalLoanMoney 借款累计金额
     * @Return totalReturnMoney 还款累计金额
     * @Return totalLoanBalance 累计借款结余
     */
    @RequestMapping(value = "/queryTotalMoneyData", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<JSONObject> queryTotalMoneyData(@RequestParam("projectId") Long projectId) {
        QueryParam param = new QueryParam();
        /** 租户隔离 */
        param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        param.getParams().put("projectId", new Parameter(QueryParam.EQ, projectId));
        param.getParams().put("billState", new Parameter(QueryParam.IN, "1,3"));

        List<ProjectLoanEntity> loanEntityList = service.queryList(param);
        JSONObject res = new JSONObject();
        BigDecimal totalLoan = BigDecimal.ZERO;
        BigDecimal totalReturn = BigDecimal.ZERO;
        BigDecimal remainLoan = BigDecimal.ZERO;
        if(ListUtil.isNotEmpty(loanEntityList)){
            for (ProjectLoanEntity loanEntity : loanEntityList) {
                totalLoan = totalLoan.add(loanEntity.getLoanMoney());
            }
        }
        List<ReturnMoneyEntity> rme = iReturnMoneyService.queryList(param);
        if(ListUtil.isNotEmpty(rme)){
            for (ReturnMoneyEntity e : rme) {
                totalReturn = totalReturn.add(e.getCurrentTotalReturn());
            }
        }
        remainLoan = totalLoan.subtract(totalReturn);
        res.put("totalLoanMoney",totalLoan);
        res.put("totalReturnMoney",totalReturn);
        res.put("totalLoanBalance",remainLoan);
        return CommonResponse.success("查询列表数据成功！",res);
    }

    /**
     * @Author mrsir_wxp
     * @Date 2021/7/20 按根据项目id 查询该项目下已生效的借款单数据
     * @Description queryTotalMoneyData
     * @Param [param]
     * @Return loanBills 借款单
     * @Return totalLoanBalance 累计借款结余
     */
    @RequestMapping(value = "/queryLoanDataByProjectId", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<JSONObject> queryLoanDataByProjectId(@RequestParam("projectId") Long projectId) {
        QueryParam param = new QueryParam();
        /** 租户隔离 */
        param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        param.getParams().put("billState", new Parameter(QueryParam.IN, "1,3"));
        param.getParams().put("projectId", new Parameter(QueryParam.EQ, projectId));

        List<ProjectLoanEntity> loanEntityList = service.queryList(param);
        List<ProjectLoanVO> loanVOS = new ArrayList<>();
        JSONObject res = new JSONObject();
        BigDecimal remainLoan = BigDecimal.ZERO;
        if(ListUtil.isNotEmpty(loanEntityList)){
            loanVOS = BeanMapper.mapList(loanEntityList,ProjectLoanVO.class);
            for (ProjectLoanVO vo : loanVOS) {
                vo.setLoanMoneyLeft(vo.getReturnMoney()==null?vo.getLoanMoney():vo.getLoanMoney().subtract(vo.getReturnMoney()));
                remainLoan = remainLoan.add(vo.getLoanMoneyLeft() == null?BigDecimal.ZERO:vo.getLoanMoneyLeft());
            }
        }
        res.put("loanBills",loanVOS);
        res.put("totalLoanBalance",remainLoan);
        return CommonResponse.success("查询列表数据成功！",res);
    }

    /**
     * @Author mrsir_wxp
     * @Date 2021/7/20 按根据项目id 查询该项目下已生效的借款单数据 项目责任书联查使用
     * @Description queryLoanDataByProjectIdForDutyLetter
     * @Param [param]
     * @Return loanBills 借款单
     * @Return totalLoanBalance 累计借款结余
     */
    @RequestMapping(value = "/queryLoanDataByProjectIdForDutyLetter", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<JSONObject> queryLoanDataByProjectIdForDutyLetter(@RequestParam("projectId") Long projectId) {
        QueryParam param = new QueryParam();
        /** 租户隔离 */
        param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        param.getParams().put("billState", new Parameter(QueryParam.IN, "1,3"));
        param.getParams().put("projectId", new Parameter(QueryParam.EQ, projectId));

        List<ProjectLoanEntity> loanEntityList = service.queryList(param);
        List<ProjectLoanVO> loanVOS = new ArrayList<>();
        JSONObject res = new JSONObject();
        BigDecimal totalLoan = BigDecimal.ZERO;
        BigDecimal remainLoan = BigDecimal.ZERO;
        BigDecimal totalReturn = BigDecimal.ZERO;
        BigDecimal totalInterest = BigDecimal.ZERO;
        if(ListUtil.isNotEmpty(loanEntityList)){
            loanVOS = BeanMapper.mapList(loanEntityList,ProjectLoanVO.class);
            for (ProjectLoanVO vo : loanVOS) {
                vo.setLoanMoneyLeft(vo.getReturnMoney()==null?vo.getLoanMoney():vo.getLoanMoney().subtract(vo.getReturnMoney()));
                totalLoan = totalLoan.add(vo.getLoanMoney());
                remainLoan = remainLoan.add(vo.getLoanMoneyLeft() == null?BigDecimal.ZERO:vo.getLoanMoneyLeft());
                totalReturn = totalReturn.add(vo.getReturnMoney() == null?BigDecimal.ZERO:vo.getReturnMoney());
                totalInterest = totalInterest.add(vo.getInterest() == null?BigDecimal.ZERO:vo.getInterest());
            }
        }
        res.put("loanBills",loanVOS);
        res.put("totalLoan",totalLoan);
        res.put("remainLoan",remainLoan);
        res.put("totalReturn",totalReturn);
        res.put("totalInterest",totalInterest);
        return CommonResponse.success("查询列表数据成功！",res);
    }

    /**
     * @Author mrsir_wxp
     * @Date 2021/7/24 导出借款列表
     * @Description excelExport
     * @Param [queryParam, response]
     * @Return void
     */
    @PostMapping(value = "excelExport")
    public void excelExport(@RequestBody QueryParam queryParam, HttpServletResponse response) {
        queryParam.getFuzzyFields().add("agentName");
        queryParam.getFuzzyFields().add("projectName");
        queryParam.getFuzzyFields().add("name");
        queryParam.setPageSize(-1);
        queryParam.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        IPage<ProjectLoanEntity> page = service.queryPage(queryParam,false);
        List<ProjectLoanVO> voList = new ArrayList<>();
        if(page.getTotal()>0){
            voList = BeanMapper.mapList(page.getRecords(),ProjectLoanVO.class);
            voList.forEach(vo->vo.setBillStateName(BillStateEnum.getEnumByStateCode(vo.getBillState()).getDescription()));
        }
        Map<String, Object> beans = new HashMap<String, Object>();
        beans.put("records", voList);
        ExcelExport.getInstance().export("project-loan-export.xlsx", beans, response);
    }

    /**
     * 参照
     * @param param
     * @return
     */
    @RequestMapping(value = "/queryRefProjectLoanList", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<IPage<ProjectLoanVO>> queryRefProjectLoanList(@RequestParam(defaultValue = "1") int pageNumber, @RequestParam(defaultValue = "10") int pageSize,
                                                                        @RequestParam(required = false) String searchText, @RequestParam(required = false) String condition,@RequestParam(required = false) String searchObject) {
        /** 模糊搜索配置字段示例 */
        QueryParam param = new QueryParam();
        param.setPageIndex(pageNumber);
        param.setPageSize(pageSize);
        param.setSearchText(searchText);
        param.setSearchObject(searchObject);
        List<String> fuzzyFields = param.getFuzzyFields();
        fuzzyFields.add("agentName");
        fuzzyFields.add("projectName");
        fuzzyFields.add("name");
        Long projectId=0L;
        if(StringUtils.isNotBlank(condition)) {
            Map<String, Object> conditionMap = JSONObject.parseObject(condition, Map.class);
            if(null != conditionMap.get("projectId")) {
                projectId = Long.valueOf(conditionMap.get("projectId").toString());
            }else{
                return CommonResponse.error("projectId 不能为空");
            }
        }


        /** 租户隔离 */
        param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        param.getParams().put("projectId",new Parameter(QueryParam.EQ, projectId));
        param.getParams().put("billState", new Parameter(QueryParam.IN, "1,3"));
        IPage<ProjectLoanEntity> page = service.queryPage(param,false);
        IPage<ProjectLoanVO> pageData = new Page<>(page.getCurrent(), page.getSize(), page.getTotal());
        pageData.setRecords(BeanMapper.mapList(page.getRecords(), ProjectLoanVO.class));

        return CommonResponse.success("查询列表数据成功！",pageData);
    }

}
