package com.ejianc.business.prjfinance.controller.api;

import com.alibaba.fastjson.JSONObject;
import com.ejianc.business.prjfinance.bean.ProjectDutyLetterEntity;
import com.ejianc.business.prjfinance.bean.RiskMarginEntity;
import com.ejianc.business.prjfinance.service.IProjectDutyLetterService;
import com.ejianc.business.prjfinance.service.IRiskMarginService;
import com.ejianc.business.prjfinance.vo.RiskMarginTransVO;
import com.ejianc.business.prjfinance.vo.RiskMarginVO;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.List;
import java.util.Map;

/**
 * @author CJ
 * @Description:
 * @date 2021/7/29 14:01
 */
@RestController
@RequestMapping("/api/riskMargin/")
public class PrjRiskMarginApi {

    @Autowired
    private IRiskMarginService riskMarginService;

    private final String BILL_CODE_RULE = "RISK_MARGIN_01";

    @Autowired
    private IBillCodeApi billCodeApi;

    @Autowired
    private IProjectDutyLetterService projectDutyLetterService;

    /**
     * 根据收款记录生成对应的风险保证金
     *   生成条件：
     *  1、对应项目已经生成目标责任书且
     *  2、目标责任书才风险保证金收取方式为：按已收工程款比例
     *  3、且对应所有已生效保证金扣款单据金额累计小于目标责任书中的风险保证金金额上限
     *
     * @param saveVO
     * @return
     */
    @PostMapping(value = "generateByRecRecord")
    public CommonResponse<Object> generateByRecRecord(@RequestBody RiskMarginTransVO saveVO) {
        JSONObject resp = new JSONObject();
        RiskMarginEntity saveEntity = BeanMapper.map(saveVO, RiskMarginEntity.class);
        ProjectDutyLetterEntity dutyEntity = projectDutyLetterService.selectByPrjojectId(saveEntity.getProjectId());
        BigDecimal dutyRiskBanlance = null;
        if(null == dutyEntity) {
            resp.put("code", "ungenerated");
            resp.put("msg", "对应项目不存在目标责任书，无法生成对应的项目扣保证金记录。");
        } else {
            if("bybeceived".equals(dutyEntity.getRiskMarginCollectionMethod())) {
                //待生成的扣保证金金额校验
                Map<String, BigDecimal> info = riskMarginService.getProjectCashDepositInfo(saveEntity.getProjectId());
                dutyRiskBanlance = dutyEntity.getRiskGuaranteeAmount().subtract(info.get("totalDeductCashDeposit"));
                if(dutyRiskBanlance.compareTo(BigDecimal.ZERO) <= 0) {
                    resp.put("code", "ungenerated");
                    resp.put("msg", "对应项目已累计扣保证金金额已达到目标责任书设置风险保证金额上限, 无法生成对应的项目扣保证金记录。");
                } else {
                    SimpleDateFormat sdf = new SimpleDateFormat("yyyy年MM月");
                    CommonResponse<String> billCodeResp = billCodeApi.getCodeBatchByRuleCode(BILL_CODE_RULE, InvocationInfoProxy.getTenantid());
                    if(billCodeResp.isSuccess()) {
                        saveEntity.setBillCode(billCodeResp.getData());
                        //设置单据类型为扣保证金
                        saveEntity.setBillType(RiskMarginVO.BILL_TYPE_DEDUCT_CASH_DEPOSIT);
                        //单据来源设置为自动生成
                        saveEntity.setSourceType(RiskMarginVO.SOURCE_TYPE_AUTOG_ENERATION);
                        //单据状态设置为：已提交
                        saveEntity.setBillState(BillStateEnum.COMMITED_STATE.getBillStateCode());
                        //设置申请金额
                        //申请金额 = 收款单金额 * 单次收款比例
                        saveEntity.setApplyAmt(saveEntity.getApplyAmt().multiply(dutyEntity.getSingleDeductionRatio().divide(new BigDecimal("100"), 8, BigDecimal.ROUND_HALF_UP)));
                        saveEntity.setApplyAmt(dutyRiskBanlance.compareTo(saveEntity.getApplyAmt()) >= 0 ? saveEntity.getApplyAmt() : dutyRiskBanlance);
                        //设置主题
                        saveEntity.setApplySubject(sdf.format(saveEntity.getApplyDate())+"扣保证金");

                        riskMarginService.saveOrUpdate(saveEntity, false);
                        resp.put("code", "generatedSuc");
                        resp.put("msg", "生成对应扣保证金单据成功！");
                        resp.put("data", BeanMapper.map(saveEntity, RiskMarginVO.class));
                    } else {
                        resp.put("code", "ungenerated");
                        resp.put("msg", "生成扣保证金失败，自动生成单据编码失败。");
                        return CommonResponse.error("保存失败，自动生成编码失败。", resp);
                    }
                }
            } else {
                resp.put("code", "ungenerated");
                resp.put("msg", "对应项目目标责任书风险保证金收取方式不是: 按已收工程款比例, 无法生成对应的项目扣保证金记录。");
            }
        }

        return CommonResponse.success("服务调用成功", resp);
    }


    /**
     * 根据项目Id查询对应的
     *
     * @param projectId
     * @return
     */
    @GetMapping(value = "getPrjFinanceData")
    public CommonResponse<Map> getPrjFinanceData(@RequestParam Long projectId) {
        Map<String, BigDecimal> riskInfo = riskMarginService.getPrjFinanceData(projectId);
        return CommonResponse.success(riskInfo);
    }

    /**
     * 根据sourceId删除对应的保证金单据
     *
     * @param sourceIds
     * @return
     */
    @PostMapping(value = "delBySourceIds")
    public CommonResponse<JSONObject> delBySourceIds(@RequestBody List<Long> sourceIds) {
        JSONObject resp = new JSONObject();
        riskMarginService.delBySourceIds(sourceIds);

        return CommonResponse.success("删除对应扣保证金单据成功！", resp);
    }

}
