package com.ejianc.business.pay.controller;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.pay.bean.PayEntity;
import com.ejianc.business.pay.enums.PayEnum;
import com.ejianc.business.pay.service.IPayService;
import com.ejianc.business.pay.vo.PayVO;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.foundation.support.api.IBillTypeApi;
import com.ejianc.foundation.support.vo.BillCodeParam;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.collection.ListUtil;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.ExcelExport;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.io.Serializable;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 保证金缴纳实体
 *
 * @author generator
 */
@RestController
@RequestMapping("pay")
public class PayController implements Serializable {
	private static final long serialVersionUID = 1868296764004829202L;

	private final Logger logger = LoggerFactory.getLogger(this.getClass());
	private static final String RULE_CODE = "PROBID_PAY";
	private static final String BILL_CODE = "EJCBT202207000018";
	private final IBillTypeApi billTypeApi;
	private final IBillCodeApi billCodeApi;
	private final IOrgApi iOrgApi;
	private final IPayService service;
	private final SessionManager sessionManager;

	public PayController(IBillTypeApi billTypeApi, IBillCodeApi billCodeApi, IOrgApi iOrgApi, IPayService service, SessionManager sessionManager) {
		this.billTypeApi = billTypeApi;
		this.billCodeApi = billCodeApi;
		this.iOrgApi = iOrgApi;
		this.service = service;
		this.sessionManager = sessionManager;
	}

	/**
	 * @Description saveOrUpdate 新增或者修改
	 */
	@PostMapping(value = "/saveOrUpdate")
	public CommonResponse<PayVO> saveOrUpdate(@RequestBody PayVO saveOrUpdateVO) {
		PayEntity entity = BeanMapper.map(saveOrUpdateVO, PayEntity.class);
		if (entity.getId() == null || entity.getId() == 0) {
			BillCodeParam billCodeParam = BillCodeParam.build(RULE_CODE, InvocationInfoProxy.getTenantid(), saveOrUpdateVO);
			CommonResponse<String> billCode = billCodeApi.generateBillCode(billCodeParam);
			if (billCode.isSuccess()) {
				entity.setBillCode(billCode.getData());//此处需要根据实际修改 删除本行或者上一行
			} else {
				throw new BusinessException("网络异常， 编码生成失败， 请稍后再试");
			}
		}

		// 单据名称、单据类型
		entity.setBillName("保证金缴纳");
		entity.setBillType(BILL_CODE);
		entity.setBillStateName(BillStateEnum.UNCOMMITED_STATE.getDescription());
		entity.setBondModeName(PayEnum.getEnumByCode(entity.getBondMode()).getDesc());

		service.saveOrUpdate(entity, false);
		PayVO vo = BeanMapper.map(entity, PayVO.class);
		return CommonResponse.success("保存或修改单据成功！", vo);
	}

	/**
	 * @param id
	 *
	 * @Description queryDetail 查询详情
	 */
	@GetMapping(value = "/queryDetail")
	public CommonResponse<PayVO> queryDetail(Long id) {
		PayEntity entity = service.selectById(id);
		PayVO vo = BeanMapper.map(entity, PayVO.class);
		return CommonResponse.success("查询详情数据成功！", vo);
	}

	/**
	 * @Description delete 批量删除单据
	 * @Param [ids]
	 */
	@PostMapping(value = "/delete")
	public CommonResponse<String> delete(@RequestBody List<PayVO> vos) {
		if (ListUtil.isNotEmpty(vos)) {
			for (PayVO vo : vos) {
				// 参数是单据类型编码字符串 根据需求是否打开下面代码
				CommonResponse<String> resp = billTypeApi.checkQuote(BILL_CODE, vo.getId());
				if (!resp.isSuccess()) {
					return CommonResponse.error("删除失败！" + resp.getMsg());
				}
			}
		}
		service.removeByIds(vos.stream().map(PayVO::getId).collect(Collectors.toList()), true);
		return CommonResponse.success("删除成功！");
	}

	/**
	 * @param param
	 *
	 * @Description queryList 查询列表
	 * @Return com.ejianc.framework.core.response.CommonResponse<java.lang.String>
	 */
	@PostMapping(value = "/queryList")
	public CommonResponse<IPage<PayVO>> queryList(@RequestBody QueryParam param) {
		List<String> fuzzyFields = param.getFuzzyFields();
		fuzzyFields.add("billCode");
		fuzzyFields.add("projectName");
		fuzzyFields.add("bidUnitName");
		fuzzyFields.add("employeeName");

		param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
		param.getParams().put("bid_unit_id", new Parameter(QueryParam.IN, iOrgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));

		IPage<PayEntity> page = service.queryPage(param, false);
		IPage<PayVO> pageData = new Page<>(page.getCurrent(), page.getSize(), page.getTotal());
		pageData.setRecords(BeanMapper.mapList(page.getRecords(), PayVO.class));

		return CommonResponse.success("查询列表数据成功！", pageData);
	}


	/**
	 * @param param
	 *
	 * @Description 导出
	 * @Return void
	 */
	@PostMapping(value = "/excelExport")
	public void excelExport(@RequestBody QueryParam param, HttpServletResponse response) {
		List<String> fuzzyFields = param.getFuzzyFields();
		fuzzyFields.add("billCode");
		fuzzyFields.add("projectName");
		fuzzyFields.add("bidUnitName");
		fuzzyFields.add("employeeName");

		param.getParams().put("tenant_id", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
		param.setPageIndex(1);
		param.setPageSize(-1);

		param.getParams().put("bid_unit_id", new Parameter(QueryParam.IN, iOrgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));

		IPage<PayEntity> page = service.queryPage(param, false);
		List<PayVO> pays = new ArrayList<>();

		page.getRecords().forEach(e -> {
			PayVO pay = BeanMapper.map(e, PayVO.class);
			pay.setBillStateName(BillStateEnum.getEnumByStateCode(pay.getBillState()).getDescription());
			pays.add(pay);
		});

		Map<String, Object> beans = new HashMap<>();
		beans.put("records", pays);
		ExcelExport.getInstance().export("pay-export.xlsx", beans, response);
	}

	/**
	 * 根据投标单位过滤当前组织本下的审批通过且 保证金形式为 现金 的 保证金缴纳 信息
	 *
	 * @param pageNumber   页数
	 * @param pageSize     大小
	 * @param condition    参数
	 * @param searchObject 模糊查询具体关键字
	 * @param searchText   查询关键字
	 *
	 * @return 保证金缴纳分页参照
	 */
	@GetMapping(value = "/refPayData")
	public CommonResponse<IPage<PayVO>> refPayData(@RequestParam Integer pageNumber, @RequestParam Integer pageSize,
	                                               String condition,
	                                               String searchObject,
	                                               String searchText) {
		QueryParam param = new QueryParam();
		param.setPageSize(pageSize);
		param.setPageIndex(pageNumber);
		param.setSearchText(searchText);
		param.setSearchObject(searchObject);

		param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
		param.getParams().put("bid_unit_id", new Parameter(QueryParam.IN, iOrgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
		param.getParams().put("bill_state", new Parameter(QueryParam.IN, Arrays.asList(BillStateEnum.COMMITED_STATE.getBillStateCode(), BillStateEnum.PASSED_STATE.getBillStateCode())));
		param.getParams().put("bond_mode", new Parameter(QueryParam.EQ, PayEnum.CASH.getCode()));

		if (StringUtils.isNotEmpty(condition)) {
			/** 处理condition */
			JSONObject _con = JSONObject.parseObject(condition);
		}

		IPage<PayEntity> page = service.queryPage(param, false);
		IPage<PayVO> pageData = new Page<>(page.getCurrent(), page.getSize(), page.getTotal());
		pageData.setRecords(BeanMapper.mapList(page.getRecords(), PayVO.class));

		return CommonResponse.success("查询参照数据成功！", pageData);
	}


	/**
	 * 根据投标立项项目id查询保证金缴纳记录
	 *
	 * @param projectId 投标立项项目id
	 *
	 * @return 保证金缴纳记录
	 */
	@GetMapping(value = "/queryDetailByProjectId")
	public CommonResponse<PayVO> queryDetailByProjectId(@RequestParam(value = "projectId") Long projectId) {
		return CommonResponse.success("查询详情数据成功！", service.queryDetailByProjectId(projectId));
	}

}
