package com.ejianc.business.record.service.impl;

import com.ejianc.business.record.bean.RecordEntity;
import com.ejianc.business.record.service.IRecordService;
import com.ejianc.foundation.support.api.IBillTypeApi;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.auth.session.UserContext;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.skeleton.billState.service.ICommonBusinessService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import java.util.Date;

@Service("record")
public class RecordBpmServiceImpl implements ICommonBusinessService {
	private final IRecordService service;
	private final SessionManager sessionManager;
	private final IBillTypeApi billTypeApi;
	private final Logger logger = LoggerFactory.getLogger(this.getClass());

	public RecordBpmServiceImpl(IRecordService service, SessionManager sessionManager, IBillTypeApi billTypeApi) {
		this.service = service;
		this.sessionManager = sessionManager;
		this.billTypeApi = billTypeApi;
	}


	/**
	 * 终审审核完回调
	 *
	 * @param billId 单据id
	 * @param state  单据状态
	 *
	 * @return 响应信息
	 */
	@Override
	public CommonResponse<String> afterApprovalProcessor(Long billId, Integer state, String billTypeCode) {
		logger.info("终审审核完回调--start，billId={},state={},billTypeCode={}", billId, state, billTypeCode);

		// 审批通过/已提交
		RecordEntity entity = service.selectById(billId);
		if (entity == null) {
			throw new BusinessException("查询不到单据信息");
		}

		UserContext userContext = sessionManager.getUserContext();

		// 区分提交和审批
		if (state.equals(BillStateEnum.COMMITED_STATE.getBillStateCode())) {
			// 直审更新提交相关字段
			entity.setCommitDate(new Date());
			entity.setCommitUserCode(userContext.getUserCode());
			entity.setCommitUserName(userContext.getUserName());
		}
		entity.setBillStateName(BillStateEnum.getEnumByStateCode(state).getDescription());
		// 生效时间
		entity.setEffectiveDate(new Date());

		// 执行更新
		service.saveOrUpdate(entity, false);

		logger.info("终审审核完回调--end");
		return CommonResponse.success("终审审核完回调成功");
	}

	/**
	 * 有审批流的撤回前回调
	 *
	 * @param billId 单据id
	 * @param state  单据状态
	 *
	 * @return 响应信息
	 */
	@Override
	public CommonResponse<String> beforeHasBpmBack(Long billId, Integer state, String billTypeCode) {
		return bpmBackCheck(billId, state, billTypeCode);
	}

	/**
	 * 弃审前事件回调
	 *
	 * @param billId 单据id
	 * @param state  单据状态
	 *
	 * @return 响应信息
	 */
	@Override
	public CommonResponse<String> beforeAbstainingProcessor(Long billId, Integer state, String billTypeCode) {
		return bpmBackCheck(billId, state, billTypeCode);
	}

	private CommonResponse<String> bpmBackCheck(Long billId, Integer state, String billTypeCode) {
		/*
			单据撤回逻辑：
			1、单据操作包含：保存、返回列表、编辑、提交、删除，规则同工程云一致。
			2、单据撤回：当存在审批流程时且第一个人没有审批时，支持撤回；当不存在审批流程且没有下游业务时支持撤回。
			3、单据弃审：审批过程中，支持后台弃审；审批通过后，没有下游业务时支持弃审。
			下游业务包含：投标文件评审
		*/

		RecordEntity entity = service.selectById(billId);
		if (entity == null) {
			throw new BusinessException("查询不到单据信息");
		}

		// 是否被其他单据引用
		CommonResponse<String> res = billTypeApi.checkQuote(billTypeCode, billId);
		logger.info("平台返回查询被引用情况：{}，----{}", res.isSuccess(), res.getMsg());
		if (!res.isSuccess()) {
			return CommonResponse.error("当前单据已被下游业务引用，不能撤回/弃审！");
		}

		return CommonResponse.success("单据撤回成功");
	}

}
