package com.ejianc.business.probuilddiary.project.service.impl;

import com.ejianc.business.probuilddiary.ledger.bean.LedgerEntity;
import com.ejianc.business.probuilddiary.ledger.service.ILedgerService;
import com.ejianc.business.probuilddiary.project.bean.ProjectLogEntity;
import com.ejianc.business.probuilddiary.project.service.IProjectLogService;
import com.ejianc.foundation.support.api.IBillTypeApi;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.auth.session.UserContext;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.skeleton.billState.service.ICommonBusinessService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.Date;

@Service("projectLog")
public class ProjectLogBpmServiceImpl implements ICommonBusinessService {

	private final Logger logger = LoggerFactory.getLogger(this.getClass());
	private final SessionManager sessionManager;
	private final IBillTypeApi billTypeApi;
	private final IProjectLogService service;
	private final ILedgerService ledgerService;

	public ProjectLogBpmServiceImpl(SessionManager sessionManager, IBillTypeApi billTypeApi, IProjectLogService service, ILedgerService ledgerService) {
		this.sessionManager = sessionManager;
		this.billTypeApi = billTypeApi;
		this.service = service;
		this.ledgerService = ledgerService;
	}

	/**
	 * 终审审核完回调
	 *
	 * @param billId 单据id
	 * @param state  单据状态
	 *
	 * @return 响应信息
	 */
	@Override
	public CommonResponse<String> afterApprovalProcessor(Long billId, Integer state, String billTypeCode) {
		logger.info("终审审核完回调--start，billId={},state={},billTypeCode={}", billId, state, billTypeCode);

		// 审批通过/已提交
		ProjectLogEntity entity = service.selectById(billId);
		if (entity == null) {
			throw new BusinessException("查询不到单据信息");
		}

		// 区分提交和审批
		UserContext userContext = sessionManager.getUserContext();
		if (state.equals(BillStateEnum.COMMITED_STATE.getBillStateCode())) {
			// 直审更新提交相关字段
			entity.setCommitDate(new Date());
			entity.setCommitUserId(userContext.getUserId());
			entity.setCommitUserCode(userContext.getUserCode());
			entity.setCommitUserName(userContext.getUserName());
		}
		entity.setBillStateName(BillStateEnum.getEnumByStateCode(state).getDescription());
		// 生效时间
		entity.setEffectiveDate(new Date());

		// 执行更新
		service.saveOrUpdate(entity, false);

		// 项目日志（个人日志仅更新状态的数据）同步到日志台账
		syncLedger(entity);

		logger.info("终审审核完回调--end");
		return CommonResponse.success("终审审核完回调成功");
	}

	/**
	 * 项目日志同步到日志台账
	 *
	 * @param entity 项目日志实体
	 */
	private void syncLedger(ProjectLogEntity entity) {
		LedgerEntity ledgerEntity = ledgerService.projectLogConvertToLedger(entity);
		ledgerService.saveOrUpdate(ledgerEntity, false);
	}

	/**
	 * 有审批流的撤回前回调
	 *
	 * @param billId 单据id
	 * @param state  单据状态
	 *
	 * @return 响应信息
	 */
	@Override
	public CommonResponse<String> beforeHasBpmBack(Long billId, Integer state, String billTypeCode) {
		return bpmBackCheck(billId, state, billTypeCode);
	}

	/**
	 * 弃审前事件回调
	 *
	 * @param billId 单据id
	 * @param state  单据状态
	 *
	 * @return 响应信息
	 */
	@Override
	public CommonResponse<String> beforeAbstainingProcessor(Long billId, Integer state, String billTypeCode) {
		return bpmBackCheck(billId, state, billTypeCode);
	}

	private CommonResponse<String> bpmBackCheck(Long billId, Integer state, String billTypeCode) {
		ProjectLogEntity entity = service.selectById(billId);
		if (entity == null) {
			throw new BusinessException("查询不到单据信息");
		}

		// 生效时间7天后不支持撤回
		LocalDateTime effectiveDate = entity.getEffectiveDate().toInstant().atZone(ZoneId.systemDefault()).toLocalDateTime();
		LocalDateTime afterSevenDays = effectiveDate.plusDays(7);
		LocalDateTime now = LocalDateTime.now();
		if (now.isAfter(afterSevenDays)) {
			throw new BusinessException("超出审批通过后的7天后，不支持撤回或弃审！");
		}

		// 删除日志台账数据
		ledgerService.deleteAllDataByProjectLogId(entity.getId());

		return CommonResponse.success("单据撤回/弃审成功");
	}

}
