package com.ejianc.business.probuilddiary.person.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.ejianc.business.probuilddiary.person.bean.PersonEntity;
import com.ejianc.business.probuilddiary.person.service.IPersonService;
import com.ejianc.business.probuilddiary.project.bean.ProjectLogEntity;
import com.ejianc.business.probuilddiary.project.service.IProjectLogService;
import com.ejianc.foundation.support.api.IBillTypeApi;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.auth.session.UserContext;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.skeleton.billState.service.ICommonBusinessService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import java.util.Date;

@Service("person")
public class PersonBpmServiceImpl implements ICommonBusinessService {

	private final Logger logger = LoggerFactory.getLogger(this.getClass());
	private final SessionManager sessionManager;
	private final IBillTypeApi billTypeApi;
	private final IPersonService service;
	private final IProjectLogService projectLogService;

	public PersonBpmServiceImpl(SessionManager sessionManager, IBillTypeApi billTypeApi, IPersonService service, IProjectLogService projectLogService) {
		this.sessionManager = sessionManager;
		this.billTypeApi = billTypeApi;
		this.service = service;
		this.projectLogService = projectLogService;
	}


	/**
	 * 提交完回调
	 *
	 * @param billId       单据id
	 * @param state        单据状态
	 * @param billTypeCode 单据类型编码
	 *
	 * @return 响应信息
	 */
	@Override
	public CommonResponse<String> afterSubmitProcessor(Long billId, Integer state, String billTypeCode) {
		logger.info("提交完回调--start，billId={},state={},billTypeCode={}", billId, state, billTypeCode);

		PersonEntity entity = service.selectById(billId);
		if (entity == null) {
			throw new BusinessException("查询不到单据信息");
		}

		UserContext userContext = sessionManager.getUserContext();
		entity.setCommitDate(new Date());
		entity.setCommitUserId(userContext.getUserId());
		entity.setCommitUserCode(userContext.getUserCode());
		entity.setCommitUserName(userContext.getUserName());
		entity.setBillStateName(BillStateEnum.getEnumByStateCode(state).getDescription());
		service.saveOrUpdate(entity, false);

		logger.info("提交完回调--end");
		return CommonResponse.success();
	}

	/**
	 * 终审审核完回调
	 *
	 * @param billId 单据id
	 * @param state  单据状态
	 *
	 * @return 响应信息
	 */
	@Override
	public CommonResponse<String> afterApprovalProcessor(Long billId, Integer state, String billTypeCode) {
		logger.info("终审审核完回调--start，billId={},state={},billTypeCode={}", billId, state, billTypeCode);

		// 审批通过/已提交
		PersonEntity entity = service.selectById(billId);
		if (entity == null) {
			throw new BusinessException("查询不到单据信息");
		}

		if (state.equals(BillStateEnum.COMMITED_STATE.getBillStateCode())) {
			UserContext userContext = sessionManager.getUserContext();
			entity.setCommitDate(new Date());
			entity.setCommitUserId(userContext.getUserId());
			entity.setCommitUserCode(userContext.getUserCode());
			entity.setCommitUserName(userContext.getUserName());
		}

		entity.setBillStateName(BillStateEnum.getEnumByStateCode(state).getDescription());
		// 生效时间
		entity.setEffectiveDate(new Date());

		// 执行更新
		service.saveOrUpdate(entity, false);

		// 同步到项目日志
		syncProjectLog(entity);

		logger.info("终审审核完回调--end");
		return CommonResponse.success("终审审核完回调成功");
	}

	/**
	 * 同步到项目日志
	 *
	 * @param entity 个人日志主实体
	 */
	private void syncProjectLog(PersonEntity entity) {
		// 校验当天是否有项目日志
		boolean hasProjectLog = projectLogService.hasProjectLog(entity.getProjectId(), entity.getInformantDate());
		projectLogService.personLogConvertToProjectLog(entity, hasProjectLog);
	}

	/**
	 * 有审批流的撤回前回调
	 *
	 * @param billId 单据id
	 * @param state  单据状态
	 *
	 * @return 响应信息
	 */
	@Override
	public CommonResponse<String> beforeHasBpmBack(Long billId, Integer state, String billTypeCode) {
		return bpmBackCheck(billId, state, billTypeCode);
	}

	/**
	 * 弃审前事件回调
	 *
	 * @param billId 单据id
	 * @param state  单据状态
	 *
	 * @return 响应信息
	 */
	@Override
	public CommonResponse<String> beforeAbstainingProcessor(Long billId, Integer state, String billTypeCode) {
		return bpmBackCheck(billId, state, billTypeCode);
	}

	private CommonResponse<String> bpmBackCheck(Long billId, Integer state, String billTypeCode) {
		PersonEntity entity = service.selectById(billId);
		if (entity == null) {
			throw new BusinessException("查询不到单据信息");
		}
		// 是否被下游引用
		CommonResponse<String> res = billTypeApi.checkQuote(billTypeCode, billId);
		logger.info("平台返回查询被引用情况：{}，{}", res.isSuccess(), res.getMsg());


		if (!res.isSuccess()) {
			return CommonResponse.error("当前单据已被项目还款单引用，不能撤回/弃审！");
		}

		// 判断当前个人日志是否汇总，没汇总的直接撤回，
		// 汇总过的，需要根据项目id和填报日期判断项目日志状态是否是审批中、审批通过、已提交，是-不允许撤回
		if (Boolean.TRUE.equals(entity.getGatherFlag())) {
			LambdaQueryWrapper<ProjectLogEntity> lambdaQuery = Wrappers.lambdaQuery();
			lambdaQuery.eq(ProjectLogEntity::getProjectId, entity.getProjectId());
			lambdaQuery.eq(ProjectLogEntity::getFillDate, entity.getInformantDate());
			ProjectLogEntity projectLogEntity = projectLogService.getOne(lambdaQuery);

			// 项目日志被删除，则直接撤回
			if (null == projectLogEntity) {
				return CommonResponse.success("单据撤回/弃审成功");
			}

			if (BillStateEnum.APPROVING_HAS_STATE.getBillStateCode().equals(projectLogEntity.getBillState())
					|| BillStateEnum.APPROVING_UNEXAM_STATE.getBillStateCode().equals(projectLogEntity.getBillState())
					|| BillStateEnum.PASSED_STATE.getBillStateCode().equals(projectLogEntity.getBillState())
					|| BillStateEnum.COMMITED_STATE.getBillStateCode().equals(projectLogEntity.getBillState())) {
				throw new BusinessException("当前个人日志已汇总至项目日志，不允许撤回！");
			}
		}
		// 删除项目日志数据，有没有汇总都执行删除操作，有就删除没有就没有
		projectLogService.deleteAllDataByPersonId(entity.getId(), entity.getProjectId(), entity.getInformantDate());
		entity.setGatherFlag(Boolean.FALSE);
		service.saveOrUpdate(entity, false);

		// TODO：补充逻辑->个人日志撤回删除所有项目日志附件

		return CommonResponse.success("单据撤回/弃审成功");
	}


}
