package com.ejianc.business.probuilddiary.project.service.impl;

import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.ejianc.business.probuilddiary.ledger.bean.LedgerEntity;
import com.ejianc.business.probuilddiary.ledger.service.ILedgerService;
import com.ejianc.business.probuilddiary.project.bean.*;
import com.ejianc.business.probuilddiary.project.service.*;
import com.ejianc.foundation.support.api.IBillTypeApi;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.auth.session.UserContext;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.collection.ListUtil;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.skeleton.billState.service.ICommonBusinessService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.Date;
import java.util.List;

@Service("projectLog")
public class ProjectLogBpmServiceImpl implements ICommonBusinessService {

	private final Logger logger = LoggerFactory.getLogger(this.getClass());
	private final SessionManager sessionManager;
	private final IBillTypeApi billTypeApi;
	private final IProjectLogService service;
	private final ILedgerService ledgerService;

	@Autowired
	private IProjectLogDetailService detailService;
	@Autowired
	private IProjectLogScheduleService scheduleService;
	@Autowired
	private IProjectLogTypeService typeService;
	@Autowired
	private IProjectLogQualityService qualityService;
	@Autowired
	private IProjectLogSafetyService safetyService;
	@Autowired
	private IProjectLogMaterialService materialService;
	@Autowired
	private IProjectLogEquipmentService equipmentService;
	@Autowired
	private IProjectLogExamineService examineService;
	@Autowired
	private IProjectLogInformationService informationService;
	@Autowired
	private IProjectLogOtherService otherService;


	public ProjectLogBpmServiceImpl(SessionManager sessionManager, IBillTypeApi billTypeApi, IProjectLogService service, ILedgerService ledgerService) {
		this.sessionManager = sessionManager;
		this.billTypeApi = billTypeApi;
		this.service = service;
		this.ledgerService = ledgerService;
	}


	/**
	 * 提交完回调
	 *
	 * @param billId       单据id
	 * @param state        单据状态
	 * @param billTypeCode 单据类型编码
	 *
	 * @return 响应信息
	 */
	@Override
	public CommonResponse<String> afterSubmitProcessor(Long billId, Integer state, String billTypeCode) {
		logger.info("提交完回调--start，billId={},state={},billTypeCode={}", billId, state, billTypeCode);

		ProjectLogEntity entity = service.selectById(billId);
		if (entity == null) {
			throw new BusinessException("查询不到单据信息");
		}
		/**
		 * 子表中缺少信息不允许提交
		 */
		List<ProjectLogDetailEntity> detailEntities = detailService.list(Wrappers.<ProjectLogDetailEntity>lambdaQuery()
				.eq(ProjectLogDetailEntity::getProjectLogId, billId)
		);
		if(ListUtil.isEmpty(detailEntities)){
			return CommonResponse.error("施工内容（施工员）信息未填写，无法提交！");
		}
		List<ProjectLogScheduleEntity> scheduleEntities = scheduleService.list(Wrappers.<ProjectLogScheduleEntity>lambdaQuery()
				.eq(ProjectLogScheduleEntity::getProjectLogId, billId)
		);
		if(ListUtil.isEmpty(scheduleEntities)){
			return CommonResponse.error("进度情况（施工员）信息未填写，无法提交！");
		}
		List<ProjectLogTypeEntity> typeEntities = typeService.list(Wrappers.<ProjectLogTypeEntity>lambdaQuery()
				.eq(ProjectLogTypeEntity::getProjectLogId, billId)
		);
		if(ListUtil.isEmpty(typeEntities)){
			return CommonResponse.error("施工组织设计、方案、交底、变更情况（技术负责人）信息未填写，无法提交！");
		}
		List<ProjectLogQualityEntity> qualityEntities = qualityService.list(Wrappers.<ProjectLogQualityEntity>lambdaQuery()
				.eq(ProjectLogQualityEntity::getProjectLogId, billId)
		);
		if(ListUtil.isEmpty(qualityEntities)){
			return CommonResponse.error("工程质量检查、验收情况（质检员）信息未填写，无法提交！");
		}
		List<ProjectLogSafetyEntity> safetyEntities = safetyService.list(Wrappers.<ProjectLogSafetyEntity>lambdaQuery()
				.eq(ProjectLogSafetyEntity::getProjectLogId, billId)
		);
		if(ListUtil.isEmpty(safetyEntities)){
			return CommonResponse.error("安全活动情况（安全员）信息未填写，无法提交！");
		}
		List<ProjectLogMaterialEntity> materialEntities = materialService.list(Wrappers.<ProjectLogMaterialEntity>lambdaQuery()
				.eq(ProjectLogMaterialEntity::getProjectLogId, billId)
		);
		if(ListUtil.isEmpty(materialEntities)){
			return CommonResponse.error("项目日志-材料进场情况（材料员）信息未填写，无法提交！");
		}
		List<ProjectLogEquipmentEntity> equipmentEntities = equipmentService.list(Wrappers.<ProjectLogEquipmentEntity>lambdaQuery()
				.eq(ProjectLogEquipmentEntity::getProjectLogId, billId)
		);
		if(ListUtil.isEmpty(equipmentEntities)){
			return CommonResponse.error("设备进场情况（设备管理员）信息未填写，无法提交！");
		}
		List<ProjectLogExamineEntity> examineEntities = examineService.list(Wrappers.<ProjectLogExamineEntity>lambdaQuery()
				.eq(ProjectLogExamineEntity::getProjectLogId, billId)
		);
		if(ListUtil.isEmpty(examineEntities)){
			return CommonResponse.error("原材料检验、时间、试块及见证取样送检等情况（取样员）信息未填写，无法提交！");
		}
		List<ProjectLogInformationEntity> informationEntities = informationService.list(Wrappers.<ProjectLogInformationEntity>lambdaQuery()
				.eq(ProjectLogInformationEntity::getProjectLogId, billId)
		);
		if(ListUtil.isEmpty(informationEntities)){
			return CommonResponse.error("工程资料归档及会议情况（资料员）信息未填写，无法提交！");
		}
		List<ProjectLogOtherEntity> otherEntities = otherService.list(Wrappers.<ProjectLogOtherEntity>lambdaQuery()
				.eq(ProjectLogOtherEntity::getProjectLogId, billId)
		);
		if(ListUtil.isEmpty(otherEntities)){
			return CommonResponse.error("其他情况信息未填写，无法提交！");
		}
		UserContext userContext = sessionManager.getUserContext();
		entity.setCommitDate(new Date());
		entity.setCommitUserId(userContext.getUserId());
		entity.setCommitUserCode(userContext.getUserCode());
		entity.setCommitUserName(userContext.getUserName());
		entity.setBillStateName(BillStateEnum.getEnumByStateCode(state).getDescription());
		service.saveOrUpdate(entity, false);

		logger.info("提交完回调--end");
		return CommonResponse.success();
	}


	/**
	 * 终审审核完回调
	 *
	 * @param billId 单据id
	 * @param state  单据状态
	 *
	 * @return 响应信息
	 */
	@Override
	public CommonResponse<String> afterApprovalProcessor(Long billId, Integer state, String billTypeCode) {
		logger.info("终审审核完回调--start，billId={},state={},billTypeCode={}", billId, state, billTypeCode);

		// 审批通过/已提交
		ProjectLogEntity entity = service.selectById(billId);
		if (entity == null) {
			throw new BusinessException("查询不到单据信息");
		}

		if (state.equals(BillStateEnum.COMMITED_STATE.getBillStateCode())) {
			UserContext userContext = sessionManager.getUserContext();
			entity.setCommitDate(new Date());
			entity.setCommitUserId(userContext.getUserId());
			entity.setCommitUserCode(userContext.getUserCode());
			entity.setCommitUserName(userContext.getUserName());
		}

		entity.setBillStateName(BillStateEnum.getEnumByStateCode(state).getDescription());
		// 生效时间
		entity.setEffectiveDate(new Date());

		// 执行更新
		service.saveOrUpdate(entity, false);

		// 项目日志（个人日志仅更新状态的数据）同步到日志台账
		syncLedger(entity);

		logger.info("终审审核完回调--end");
		return CommonResponse.success("终审审核完回调成功");
	}

	/**
	 * 项目日志同步到日志台账
	 *
	 * @param entity 项目日志实体
	 */
	private void syncLedger(ProjectLogEntity entity) {
		LedgerEntity ledgerEntity = ledgerService.projectLogConvertToLedger(entity);
		ledgerService.saveOrUpdate(ledgerEntity, false);
	}

	/**
	 * 有审批流的撤回前回调
	 *
	 * @param billId 单据id
	 * @param state  单据状态
	 *
	 * @return 响应信息
	 */
	@Override
	public CommonResponse<String> beforeHasBpmBack(Long billId, Integer state, String billTypeCode) {
		return bpmBackCheck(billId, state, billTypeCode);
	}

	/**
	 * 弃审前事件回调
	 *
	 * @param billId 单据id
	 * @param state  单据状态
	 *
	 * @return 响应信息
	 */
	@Override
	public CommonResponse<String> beforeAbstainingProcessor(Long billId, Integer state, String billTypeCode) {
		return bpmBackCheck(billId, state, billTypeCode);
	}

	private CommonResponse<String> bpmBackCheck(Long billId, Integer state, String billTypeCode) {
		ProjectLogEntity entity = service.selectById(billId);
		if (entity == null) {
			throw new BusinessException("查询不到单据信息");
		}

		if (entity.getEffectiveDate() != null) {
			// 生效时间7天后不支持撤回
			LocalDateTime effectiveDate = entity.getEffectiveDate().toInstant().atZone(ZoneId.systemDefault()).toLocalDateTime();
			LocalDateTime afterSevenDays = effectiveDate.plusDays(7);
			LocalDateTime now = LocalDateTime.now();
			if (now.isAfter(afterSevenDays)) {
				throw new BusinessException("超出审批通过后的7天后，不支持撤回或弃审！");
			}

			// 删除日志台账数据
			ledgerService.deleteAllDataByProjectLogId(entity.getId());

			return CommonResponse.success("单据撤回/弃审成功");
		}

		return CommonResponse.success("单据撤回/弃审成功");
	}

}
