package com.ejianc.business.probuilddiary.project.controller;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.ejianc.business.probuilddiary.ledger.bean.LedgerDetailEntity;
import com.ejianc.business.probuilddiary.ledger.vo.LedgerDetailVO;
import com.ejianc.business.probuilddiary.project.bean.ProjectLogDetailEntity;
import com.ejianc.business.probuilddiary.project.bean.ProjectLogEntity;
import com.ejianc.business.probuilddiary.project.service.IProjectLogDetailService;
import com.ejianc.business.probuilddiary.project.service.IProjectLogService;
import com.ejianc.business.probuilddiary.project.vo.ProjectLogDetailVO;
import com.ejianc.business.probuilddiary.project.vo.ProjectLogVO;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.foundation.support.api.IBillTypeApi;
import com.ejianc.foundation.support.vo.BillCodeParam;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.collection.ListUtil;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.ExcelExport;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 项目日志
 *
 * @author baipengyan
 */
@RestController
@RequestMapping("projectLog")
public class ProjectLogController implements Serializable {
	private static final long serialVersionUID = -1763011100374980137L;
	private static final String RULE_CODE = "PRO-BUILD-DIARY-PROJECT-LOG";
	private static final String BILL_CODE = "EJCBT202209000019";
	private final Logger logger = LoggerFactory.getLogger(this.getClass());
	private final SessionManager sessionManager;
	private final IBillTypeApi billTypeApi;
	private final IBillCodeApi billCodeApi;
	private final IOrgApi iOrgApi;
	private final IProjectLogService service;
	@Autowired
	private IProjectLogDetailService detailService;

	public ProjectLogController(SessionManager sessionManager, IBillTypeApi billTypeApi, IBillCodeApi billCodeApi, IOrgApi iOrgApi, IProjectLogService service) {
		this.sessionManager = sessionManager;
		this.billTypeApi = billTypeApi;
		this.billCodeApi = billCodeApi;
		this.iOrgApi = iOrgApi;
		this.service = service;
	}


	/**
	 * 保存修改
	 *
	 * @param saveOrUpdateVO vo
	 *
	 * @return CommonResponse<ProjectLogVO>
	 */
	@PostMapping(value = "/saveOrUpdate")
	public CommonResponse<ProjectLogVO> saveOrUpdate(@RequestBody ProjectLogVO saveOrUpdateVO) {
		ProjectLogEntity entity = BeanMapper.map(saveOrUpdateVO, ProjectLogEntity.class);
		if (null == entity.getId() || 0 == entity.getId()) {
			BillCodeParam billCodeParam = BillCodeParam.build(RULE_CODE, InvocationInfoProxy.getTenantid(), saveOrUpdateVO);
			CommonResponse<String> billCode = billCodeApi.generateBillCode(billCodeParam);
			if (billCode.isSuccess()) {
				entity.setBillCode(billCode.getData());//此处需要根据实际修改 删除本行或者上一行
			} else {
				throw new BusinessException("网络异常， 编码生成失败， 请稍后再试");
			}
		}
		service.checkPersonBillState(entity);
		entity.setBillName("项目日志");
		entity.setBillStateName(BillStateEnum.UNCOMMITED_STATE.getDescription());
		service.saveOrUpdate(entity, false);
		return CommonResponse.success("保存或修改单据成功！", BeanMapper.map(service.selectById(entity.getId()), ProjectLogVO.class));
	}

	/**
	 * 查询详情
	 *
	 * @param id 主键
	 *
	 * @return CommonResponse<ProjectLogVO>
	 */
	@GetMapping(value = "/queryDetail")
	public CommonResponse<ProjectLogVO> queryDetail(@RequestParam(value = "id", required = true) Long id) {
		ProjectLogEntity entity = service.selectById(id);
		ProjectLogVO vo = BeanMapper.map(entity, ProjectLogVO.class);
		return CommonResponse.success("查询详情数据成功！", vo);
	}

	/**
	 * 删除
	 *
	 * @param vos 待删除的vos
	 *
	 * @return CommonResponse<String>
	 */
	@PostMapping(value = "/delete")
	public CommonResponse<String> delete(@RequestBody List<ProjectLogVO> vos) {
		if (ListUtil.isNotEmpty(vos)) {
			for (ProjectLogVO vo : vos) {
				CommonResponse<String> resp = billTypeApi.checkQuote(BILL_CODE, vo.getId());
				if (!resp.isSuccess()) {
					return CommonResponse.error("删除失败！" + resp.getMsg());
				}
			}
		}
		service.removeByIds(vos.stream().map(ProjectLogVO::getId).collect(Collectors.toList()), true);
		return CommonResponse.success("删除成功！");
	}

	/**
	 * 分页查询
	 *
	 * @param param 请求参数
	 *
	 * @return CommonResponse<JSONObject>
	 */
	@PostMapping(value = "/queryList")
	public CommonResponse<JSONObject> queryList(@RequestBody QueryParam param) {
		JSONObject resp = new JSONObject();

		List<String> fuzzyFields = param.getFuzzyFields();
		fuzzyFields.add("billCode");
		fuzzyFields.add("projectName");

		param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
		//若当前上下文为项目部，则根据项目部Id来进行查询
		if (OrgVO.ORG_TYPE_DEPARTMENT.equals(Integer.valueOf(InvocationInfoProxy.getOrgType()))) {
			param.getParams().put("orgId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getOrgId()));
		} else {
			CommonResponse<List<OrgVO>> orgResp = iOrgApi.findChildrenByParentIdWithoutProjectDept(InvocationInfoProxy.getOrgId());
			if (!orgResp.isSuccess()) {
				logger.error("分页查询失败，获取当前本下组织信息失败, {}", orgResp.getMsg());
				return CommonResponse.error("查询失败，获取组织信息失败！");
			}
			param.getParams().put("parentOrgId", new Parameter(QueryParam.IN,
					orgResp.getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
		}

		IPage<ProjectLogEntity> page = service.queryPage(param, false);
		List<ProjectLogVO> projectLogVOS = BeanMapper.mapList(page.getRecords(), ProjectLogVO.class);
		resp.put("current", page.getCurrent());
		resp.put("size", page.getSize());
		resp.put("pages", page.getPages());
		resp.put("total", page.getTotal());
		resp.put("records", projectLogVOS);
		return CommonResponse.success("查询列表数据成功！", resp);
	}


	/**
	 * 导出
	 *
	 * @param param    查询参数
	 * @param response 响应头
	 */
	@PostMapping(value = "/excelExport")
	public void excelExport(@RequestBody QueryParam param, HttpServletResponse response) {
		param.setPageIndex(1);
		param.setPageSize(-1);

		List<String> fuzzyFields = param.getFuzzyFields();
		fuzzyFields.add("billCode");
		fuzzyFields.add("parentOrgName");
		fuzzyFields.add("projectName");
		fuzzyFields.add("memo");

		param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
		//若当前上下文为项目部，则根据项目部Id来进行查询
		if (OrgVO.ORG_TYPE_DEPARTMENT.equals(Integer.valueOf(InvocationInfoProxy.getOrgType()))) {
			param.getParams().put("orgId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getOrgId()));
		} else {
			CommonResponse<List<OrgVO>> orgResp = iOrgApi.findChildrenByParentIdWithoutProjectDept(InvocationInfoProxy.getOrgId());
			if (!orgResp.isSuccess()) {
				logger.error("分页查询失败，获取当前本下组织信息失败, {}", orgResp.getMsg());
			}
			param.getParams().put("parentOrgId", new Parameter(QueryParam.IN,
					orgResp.getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
		}

		IPage<ProjectLogEntity> page = service.queryPage(param, false);
		List<ProjectLogVO> projectLogVOS = new ArrayList<>();
		page.getRecords().forEach(e -> {
			ProjectLogVO projectLogVO = BeanMapper.map(e, ProjectLogVO.class);
			projectLogVO.setBillStateName(BillStateEnum.getEnumByStateCode(projectLogVO.getBillState()).getDescription());
			projectLogVOS.add(projectLogVO);
		});
		Map<String, Object> beans = new HashMap<>();
		beans.put("records", projectLogVOS);
		ExcelExport.getInstance().export("projectLog-export.xlsx", beans, response);
	}

	/**
	 * 是否有项目日志
	 *
	 * @param jsonObject 参数
	 *
	 * @return CommonResponse<Boolean>
	 */
	@PostMapping(value = "/hasProjectLog")
	public CommonResponse<Boolean> hasProjectLog(@RequestBody JSONObject jsonObject) {
		if (!jsonObject.containsKey("projectId")) {
			throw new BusinessException("项目id不能为空！");
		}
		if (!jsonObject.containsKey("fillDate")) {
			throw new BusinessException("填报日期不能为空！");
		}
		return CommonResponse.success("查询成功！", service.hasProjectLog(jsonObject.getLong("projectId"), jsonObject.getDate("fillDate")));
	}

	/**
	 * 查询验收单据数量
	 *
	 * @param jsonObject 参数
	 *
	 * @return CommonResponse<Map < String, Integer>>
	 */
	@PostMapping(value = "/countBillNum")
	public CommonResponse<Map<String, Integer>> countBillNum(@RequestBody JSONObject jsonObject) {
		return CommonResponse.success("查询成功！", service.countBillNum(jsonObject));
	}


	/**
	 * 项目日志预警
	 *
	 * @param request 请求信息
	 *
	 * @return CommonResponse<String>
	 */
	@PostMapping(value = "/warn")
	public CommonResponse<String> warn(HttpServletRequest request) {
		logger.info("项目日志预警--------start");
		service.warn(request);
		logger.info("项目日志预警--------end");
		return CommonResponse.success();
	}
	/**
	 * 查询其他子表数据
	 *
	 * @param id 主键
	 *
	 * @return LedgerSubVO
	 */
	@GetMapping(value = "/fetchOtherSubDetail")
	public CommonResponse<List<JSONObject>> fetchOtherSubDetail(@RequestParam(value = "id") Long id) {
		return CommonResponse.success("查询成功！", service.fetchOtherSubDetail(id));
	}

	/**
	 * 日志台账-施工内容（施工员）分页查询
	 *
	 * @param id        主键
	 * @param pageIndex 当前页
	 * @param pageSize  当前分页总页数
	 *
	 * @return CommonResponse<IPage < LedgerDetailVO>>
	 */
	@GetMapping(value = "/ledgerDetailQueryPage")
	public CommonResponse<JSONObject> ledgerDetailQueryPage(@RequestParam(value = "id") Long id,
															@RequestParam(value = "pageIndex", defaultValue = "1") Integer pageIndex,
															@RequestParam(value = "pageSize", defaultValue = "10") Integer pageSize) {
		QueryParam param = new QueryParam();
		param.setPageIndex(pageIndex);
		param.setPageSize(pageSize);
		param.getParams().put("project_log_id", new Parameter(QueryParam.EQ, id));
		IPage<ProjectLogDetailEntity> page = detailService.queryPage(param);
		List<ProjectLogDetailVO> ledgerVOS = BeanMapper.mapList(page.getRecords(), ProjectLogDetailVO.class);

		JSONObject resp = new JSONObject();
		resp.put("current", page.getCurrent());
		resp.put("size", page.getSize());
		resp.put("pages", page.getPages());
		resp.put("total", page.getTotal());
		resp.put("records", ledgerVOS);
		return CommonResponse.success("查询分页成功！", resp);
	}

	/**
	 * 更新位置、天气信息
	 *
	 * @param saveOrUpdateVO vo
	 *
	 */
	@PostMapping(value = "/updateWeather")
	public CommonResponse<ProjectLogVO> updateWeather(@RequestBody ProjectLogVO saveOrUpdateVO) {
		if(saveOrUpdateVO.getId() == null){
			return CommonResponse.error("更新位置、天气信息失败！");
		}
		ProjectLogEntity entity = service.selectById(saveOrUpdateVO.getId());
		entity.setWeather(saveOrUpdateVO.getWeather());
		entity.setWindPower(saveOrUpdateVO.getWindPower());
		entity.setTemperature(saveOrUpdateVO.getTemperature());
		entity.setHumidity(saveOrUpdateVO.getHumidity());
		entity.setAirQuality(saveOrUpdateVO.getAirQuality());
		entity.setProvinceName(saveOrUpdateVO.getProvinceName());
		entity.setCityName(saveOrUpdateVO.getCityName());
		entity.setDistrictName(saveOrUpdateVO.getDistrictName());
		service.saveOrUpdate(entity);
		return CommonResponse.success("更新位置、天气信息成功！", BeanMapper.map(service.selectById(entity.getId()), ProjectLogVO.class));
	}
}
