package com.ejianc.business.profinance.plan.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.IdWorker;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.ejianc.business.contractbase.pool.settlepool.api.ISettlePoolApi;
import com.ejianc.business.profinance.plan.bean.*;
import com.ejianc.business.profinance.plan.enums.CloseStateEnum;
import com.ejianc.business.profinance.plan.mapper.PlanMapper;
import com.ejianc.business.profinance.plan.service.*;
import com.ejianc.business.receipt.bean.ReceiptRegisterEntity;
import com.ejianc.business.receipt.service.IReceiptRegisterService;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.skeleton.template.BaseServiceImpl;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.*;

/**
 * 项目资金计划
 *
 * @author generator
 */
@Service("planService")
public class PlanServiceImpl extends BaseServiceImpl<PlanMapper, PlanEntity> implements IPlanService {

	private final IReceiptRegisterService receiptRegisterService;
	private final ISettlePoolApi settlePoolApi;
	private final IPlanContractService planContractService;
	private final IPlanMaterialService planMaterialService;
	private final IPlanTemporaryService planTemporaryService;
	private final IPlanFeeService planFeeService;

	public PlanServiceImpl(IReceiptRegisterService receiptRegisterService,
	                       ISettlePoolApi settlePoolApi,
	                       IPlanContractService planContractService,
	                       IPlanMaterialService planMaterialService,
	                       IPlanTemporaryService planTemporaryService,
	                       IPlanFeeService planFeeService) {
		this.receiptRegisterService = receiptRegisterService;
		this.settlePoolApi = settlePoolApi;
		this.planContractService = planContractService;
		this.planMaterialService = planMaterialService;
		this.planTemporaryService = planTemporaryService;
		this.planFeeService = planFeeService;
	}

	/**
	 * 设置子表默认值
	 *
	 * @param entity 项目资金计划实体
	 */
	@Override
	public void setSubDefaultValue(PlanEntity entity) {
		List<PlanContractEntity> planContractList = entity.getPlanContractList();
		List<PlanMaterialEntity> planMaterialList = entity.getPlanMaterialList();
		List<PlanTemporaryEntity> planTemporaryList = entity.getPlanTemporaryList();
		List<PlanFeeEntity> planFeeList = entity.getPlanFeeList();

		resetPid(planContractList);

		if (CollectionUtils.isNotEmpty(planMaterialList)) {
			for (PlanMaterialEntity planMaterialEntity : planMaterialList) {
				planMaterialEntity.setCloseState(CloseStateEnum.NOT_CLOSED.getName());
			}
		}

		if (CollectionUtils.isNotEmpty(planTemporaryList)) {
			for (PlanTemporaryEntity planTemporaryEntity : planTemporaryList) {
				planTemporaryEntity.setCloseState(CloseStateEnum.NOT_CLOSED.getName());
			}
		}

		if (CollectionUtils.isNotEmpty(planFeeList)) {
			for (PlanFeeEntity planFeeEntity : planFeeList) {
				planFeeEntity.setCloseState(CloseStateEnum.NOT_CLOSED.getName());
			}
		}

	}

	/**
	 * 重置id和父id
	 *
	 * @param planContractList 支出合同
	 */
	private void resetPid(List<PlanContractEntity> planContractList) {
		if (CollectionUtils.isNotEmpty(planContractList)) {
			Map<String, Long> idMap = new HashMap<>();
			for (PlanContractEntity cdEntity : planContractList) {
				if (!("del").equals(cdEntity.getRowState())) {
					if (cdEntity.getId() == null) {
						cdEntity.setId(IdWorker.getId());
					}
					idMap.put(cdEntity.getTid(), cdEntity.getId());
					cdEntity.setParentId(null);
				}
			}
			for (PlanContractEntity cdEntity : planContractList) {
				if (!("del").equals(cdEntity.getRowState())) {
					if (StringUtils.isNotEmpty(cdEntity.getTpid())) {
						cdEntity.setCloseState(CloseStateEnum.NOT_CLOSED.getName());
						cdEntity.setParentId(idMap.get(cdEntity.getTpid()));
					}
				}
			}
		}
	}

	/**
	 * 根据项目id查询最新的项目资金计划信息
	 *
	 * @param projectId 项目id
	 *
	 * @return PlanEntity
	 */
	@Override
	public PlanEntity fetchLastPlan(Long projectId) {
		LambdaQueryWrapper<PlanEntity> queryWrapper = Wrappers.lambdaQuery();
		queryWrapper.in(PlanEntity::getBillState, Arrays.asList(BillStateEnum.COMMITED_STATE.getBillStateCode(), BillStateEnum.PASSED_STATE.getBillStateCode()));
		queryWrapper.eq(PlanEntity::getProjectId, projectId);
		queryWrapper.orderByDesc(PlanEntity::getPlanDate);
		List<PlanEntity> plans = super.list(queryWrapper);
		if (CollectionUtils.isNotEmpty(plans)) {
			return plans.get(0);
		}
		return null;
	}

	/**
	 * 根据项目id查询已审批通过的【收款登记】“本次实收金额”之和
	 *
	 * @param projectId 项目id
	 *
	 * @return 累计工程收款(含本次)
	 */
	@Override
	public BigDecimal fetchTotalReceiptRegister(Long projectId) {
		QueryWrapper<ReceiptRegisterEntity> queryWrapper = Wrappers.query();
		queryWrapper.in("bill_state", Arrays.asList(BillStateEnum.COMMITED_STATE.getBillStateCode(), BillStateEnum.PASSED_STATE.getBillStateCode()));
		queryWrapper.eq("project_id", projectId);
		queryWrapper.select("ifnull(sum(sum_received_mny), 0) as total");
		Map<String, Object> resultMap = receiptRegisterService.getMap(queryWrapper);
		return new BigDecimal(resultMap.get("total").toString());
	}

	/**
	 * 根据项目id查询结算池中结算类型为产值报量“本期结算金额(含税)”之和
	 *
	 * @param projectId 项目id
	 *
	 * @return 累计产值金额
	 */
	@Override
	public BigDecimal fetchTotalProductValue(Long projectId) {
		CommonResponse<BigDecimal> res = settlePoolApi.fetchTotalProductValueTaxMny(projectId);
		if (!res.isSuccess() || res.getData() == null) {
			throw new BusinessException("根据项目id：" + projectId + "查询结算池中结算类型为产值报量的累计产值金额，调用结算池接口异常！");
		}
		return res.getData();
	}

	/**
	 * 根据项目id获取上期计划日期、累计工程收款(不含本次)、累计产值金额
	 *
	 * @param projectId 项目id
	 *
	 * @return 上期计划日期、累计工程收款(不含本次)、累计产值金额
	 */
	@Override
	public Map<String, Object> fetchValue(Long projectId) {
		HashMap<String, Object> resultMap = new HashMap<>();

		// 获取上期计划日期
		LambdaQueryWrapper<PlanEntity> queryWrapper = Wrappers.lambdaQuery();
		queryWrapper.in(PlanEntity::getBillState, Arrays.asList(BillStateEnum.COMMITED_STATE.getBillStateCode(), BillStateEnum.PASSED_STATE.getBillStateCode()));
		queryWrapper.eq(PlanEntity::getProjectId, projectId);
		queryWrapper.orderByDesc(PlanEntity::getPlanDate);
		List<PlanEntity> plans = super.list(queryWrapper);
		if (CollectionUtils.isNotEmpty(plans)) {
			resultMap.put("lastPlanDate", plans.get(0).getPlanDate());
		}

		// 获取累计工程收款(不含本次)
		QueryWrapper<ReceiptRegisterEntity> queryWrapper1 = Wrappers.query();
		queryWrapper1.in("bill_state", Arrays.asList(BillStateEnum.COMMITED_STATE.getBillStateCode(), BillStateEnum.PASSED_STATE.getBillStateCode()));
		queryWrapper1.eq("project_id", projectId);
		queryWrapper1.select("ifnull(sum(sum_received_mny), 0) as total");
		Map<String, Object> resultMap1 = receiptRegisterService.getMap(queryWrapper1);
		if (resultMap1.isEmpty()) {
			resultMap.put("lastTotalProjectReceipt", BigDecimal.ZERO);
		} else {
			resultMap.put("lastTotalProjectReceipt", new BigDecimal(resultMap1.get("total").toString()));
		}

		// 获取累计产值金额
		CommonResponse<BigDecimal> res = settlePoolApi.fetchTotalProductValueTaxMny(projectId);
		if (!res.isSuccess()) {
			throw new BusinessException("根据项目id：" + projectId + "查询结算池中结算类型为产值报量的累计产值金额，调用结算池接口异常！");
		}
		resultMap.put("lastTotalOutputValueTaxMny", res.getData());

		return resultMap;
	}

	/**
	 * 校验引用的参照数据
	 *
	 * @param referMap 参照数据
	 */
	@Override
	public void checkReferQuote(JSONObject referMap) {
		if (referMap.isEmpty()) {
			throw new BusinessException("参照数据为空！");
		}

		if (referMap.containsKey("contract")) {
			Collection<Long> contractIds = (Collection<Long>) referMap.get("contract");
			if (CollectionUtils.isEmpty(contractIds)) {
				throw new BusinessException("参照的合同数据为空！");
			}
			LambdaQueryWrapper<PlanContractEntity> queryWrapper = Wrappers.lambdaQuery();
			queryWrapper.notIn(PlanContractEntity::getBillState, Arrays.asList(BillStateEnum.COMMITED_STATE.getBillStateCode(), BillStateEnum.PASSED_STATE.getBillStateCode()));
			queryWrapper.in(PlanContractEntity::getContractId, contractIds);
			if (referMap.containsKey("excludePlanId")) {
				queryWrapper.ne(PlanContractEntity::getPlanId, referMap.get("excludePlanId"));
			}
			List<PlanContractEntity> entityList = planContractService.list(queryWrapper);
			if (CollectionUtils.isNotEmpty(entityList)) {
				throw new BusinessException("操作失败！编号【" + entityList.get(0).getContractCode() + "】的数据已被未生效的【项目资金计划】引用，暂不允许使用该数据。");
			}
		}

		if (referMap.containsKey("material")) {
			Collection<Long> settleIds = (Collection<Long>) referMap.get("material");
			if (CollectionUtils.isEmpty(settleIds)) {
				throw new BusinessException("参照的零星材料采购数据为空！");
			}
			LambdaQueryWrapper<PlanMaterialEntity> queryWrapper = Wrappers.lambdaQuery();
			queryWrapper.notIn(PlanMaterialEntity::getBillState, Arrays.asList(BillStateEnum.COMMITED_STATE.getBillStateCode(), BillStateEnum.PASSED_STATE.getBillStateCode()));
			queryWrapper.in(PlanMaterialEntity::getMaterialSettleId, settleIds);
			if (referMap.containsKey("excludePlanId")) {
				queryWrapper.ne(PlanMaterialEntity::getPlanId, referMap.get("excludePlanId"));
			}
			List<PlanMaterialEntity> entityList = planMaterialService.list(queryWrapper);
			if (CollectionUtils.isNotEmpty(entityList)) {
				throw new BusinessException("操作失败！编号【" + entityList.get(0).getMaterialSettleCode() + "】的数据已被未生效的【项目资金计划】引用，暂不允许使用该数据。");
			}
		}

		if (referMap.containsKey("temporary")) {
			Collection<Long> settleIds = (Collection<Long>) referMap.get("temporary");
			if (CollectionUtils.isEmpty(settleIds)) {
				throw new BusinessException("参照的临时机械租赁数据为空！");
			}
			LambdaQueryWrapper<PlanTemporaryEntity> queryWrapper = Wrappers.lambdaQuery();
			queryWrapper.notIn(PlanTemporaryEntity::getBillState, Arrays.asList(BillStateEnum.COMMITED_STATE.getBillStateCode(), BillStateEnum.PASSED_STATE.getBillStateCode()));
			queryWrapper.in(PlanTemporaryEntity::getTemporarySettleId, settleIds);
			if (referMap.containsKey("excludePlanId")) {
				queryWrapper.ne(PlanTemporaryEntity::getPlanId, referMap.get("excludePlanId"));
			}
			List<PlanTemporaryEntity> entityList = planTemporaryService.list(queryWrapper);
			if (CollectionUtils.isNotEmpty(entityList)) {
				throw new BusinessException("操作失败！编号【" + entityList.get(0).getTemporarySettleCode() + "】的数据已被未生效的【项目资金计划】引用，暂不允许使用该数据。");
			}
		}

		if (referMap.containsKey("fee")) {
			Collection<Long> feeIds = (Collection<Long>) referMap.get("fee");
			if (CollectionUtils.isEmpty(feeIds)) {
				throw new BusinessException("参照的其他零星费用数据为空！");
			}
			LambdaQueryWrapper<PlanFeeEntity> queryWrapper = Wrappers.lambdaQuery();
			queryWrapper.notIn(PlanFeeEntity::getBillState, Arrays.asList(BillStateEnum.COMMITED_STATE.getBillStateCode(), BillStateEnum.PASSED_STATE.getBillStateCode()));
			queryWrapper.in(PlanFeeEntity::getFeeId, feeIds);
			if (referMap.containsKey("excludePlanId")) {
				queryWrapper.ne(PlanFeeEntity::getPlanId, referMap.get("excludePlanId"));
			}
			List<PlanFeeEntity> entityList = planFeeService.list(queryWrapper);
			if (CollectionUtils.isNotEmpty(entityList)) {
				throw new BusinessException("操作失败！编号【" + entityList.get(0).getFeeCode() + "】的数据已被未生效的【项目资金计划】引用，暂不允许使用该数据。");
			}
		}
	}

	/**
	 * 校验只能有一个未生效的单据
	 *
	 * @param projectId 项目id
	 */
	@Override
	public void checkOnlyOne(Long projectId) {
		QueryWrapper<PlanEntity> queryWrapper = new QueryWrapper<>();
		queryWrapper.eq("project_id", projectId);
		queryWrapper.notIn("bill_state", Arrays.asList(BillStateEnum.COMMITED_STATE.getBillStateCode(), BillStateEnum.PASSED_STATE.getBillStateCode()));
		List<PlanEntity> entities = super.list(queryWrapper);
		if (CollectionUtils.isNotEmpty(entities)) {
			throw new BusinessException("当前项目已存在未生效的单据！");
		}
	}
}
