package com.ejianc.business.profinance.projectloan.controller;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.ejianc.business.profinance.projectloan.bean.ProjectRepayDetailEntity;
import com.ejianc.business.profinance.projectloan.bean.ProjectRepayEntity;
import com.ejianc.business.profinance.projectloan.service.IProjectLoanService;
import com.ejianc.business.profinance.projectloan.service.IProjectRepayService;
import com.ejianc.business.profinance.projectloan.vo.ProjectRepayDetailVO;
import com.ejianc.business.profinance.projectloan.vo.ProjectRepayVO;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.foundation.support.api.IBillTypeApi;
import com.ejianc.foundation.support.vo.BillCodeParam;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.collection.ListUtil;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.ExcelExport;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.io.Serializable;
import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 借款管理-项目还款实体
 *
 * @author baipengyan
 */
@RestController
@RequestMapping("projectRepay")
public class ProjectRepayController implements Serializable {
	private static final long serialVersionUID = -7949596054926672689L;

	private Logger logger = LoggerFactory.getLogger(this.getClass());
	private static final String RULE_CODE = "PROJECT_REPAY";
	private static final String BILL_CODE = "EJCBT202208000002";
	private final IBillTypeApi billTypeApi;
	private final IBillCodeApi billCodeApi;
	private final IOrgApi iOrgApi;
	private final IProjectRepayService service;
	private final SessionManager sessionManager;
	private final IProjectLoanService loanService;

	public ProjectRepayController(IBillTypeApi billTypeApi, IBillCodeApi billCodeApi, IOrgApi iOrgApi,
	                              IProjectRepayService service, SessionManager sessionManager,
	                              IProjectLoanService loanService) {
		this.billTypeApi = billTypeApi;
		this.billCodeApi = billCodeApi;
		this.iOrgApi = iOrgApi;
		this.service = service;
		this.sessionManager = sessionManager;
		this.loanService = loanService;
	}

	/**
	 * 新增或者修改
	 *
	 * @param saveOrUpdateVO 实体VO
	 *
	 * @return 实体VO
	 */
	@PostMapping(value = "/saveOrUpdate")
	public CommonResponse<ProjectRepayVO> saveOrUpdate(@RequestBody ProjectRepayVO saveOrUpdateVO) {
		ProjectRepayEntity entity = BeanMapper.map(saveOrUpdateVO, ProjectRepayEntity.class);
		if (entity.getId() == null || entity.getId() == 0) {
			BillCodeParam billCodeParam = BillCodeParam.build(RULE_CODE, InvocationInfoProxy.getTenantid(), saveOrUpdateVO);
			CommonResponse<String> billCode = billCodeApi.generateBillCode(billCodeParam);
			if (billCode.isSuccess()) {
				entity.setBillCode(billCode.getData());
			} else {
				throw new BusinessException("网络异常， 编码生成失败， 请稍后再试");
			}
		}

		entity.setBillName("项目还款");
		entity.setBillStateName(BillStateEnum.UNCOMMITED_STATE.getDescription());
		List<ProjectRepayDetailVO> detailList = saveOrUpdateVO.getProjectRepayDetailList();
		List<Long> delIds = new ArrayList<>();
		List<Long> addIds = new ArrayList<>();
		for (ProjectRepayDetailVO vo : detailList) {
			if (vo.getRowState() != null && vo.getRowState().equals("del")) {
				Long sourceLoanId = vo.getSourceId();

				delIds.add(sourceLoanId);
			}
			if (vo.getRowState() != null && vo.getRowState().equals("add")) {
				addIds.add(vo.getSourceId());
			}
		}
		if (!delIds.isEmpty()) {
			loanService.updateColumnValue(delIds, "ref_status", "未引用");
		}
		if (!addIds.isEmpty()) {
			loanService.updateColumnValue(addIds, "ref_status", "引用");
		}
		service.saveOrUpdate(entity, false);
//		List<ProjectRepayDetailEntity> detailList = entity.getProjectRepayDetailList();
//		List<Long> ids = detailList.stream().map(ProjectRepayDetailEntity::getSourceId).collect(Collectors.toList());
//		loanService.updateColumnValue(ids,"ref_status","引用");
		ProjectRepayVO vo = BeanMapper.map(entity, ProjectRepayVO.class);
		return CommonResponse.success("保存或修改单据成功！", vo);
	}

	/**
	 * 查询详情
	 *
	 * @param id 主键
	 *
	 * @return 查询结果
	 */
	@GetMapping(value = "/queryDetail")
	public CommonResponse<ProjectRepayVO> queryDetail(Long id) {
		ProjectRepayEntity entity = service.selectById(id);
		ProjectRepayVO vo = BeanMapper.map(entity, ProjectRepayVO.class);
		return CommonResponse.success("查询详情数据成功！", vo);
	}

	/**
	 * 批量删除单据
	 *
	 * @param vos 要删除的单据
	 *
	 * @return 删除结果
	 */
	@PostMapping(value = "/delete")
	public CommonResponse<String> delete(@RequestBody List<ProjectRepayVO> vos) {
		if (ListUtil.isNotEmpty(vos)) {
			for (ProjectRepayVO vo : vos) {
				CommonResponse<String> resp = billTypeApi.checkQuote(BILL_CODE, vo.getId());
				if (!resp.isSuccess()) {
					return CommonResponse.error("删除失败！" + resp.getMsg());
				}
				ProjectRepayEntity repayEntity = service.selectById(vo.getId());
				List<ProjectRepayDetailEntity> detailList = repayEntity.getProjectRepayDetailList();
				List<Long> sourceLoanIds = detailList.stream().map(ProjectRepayDetailEntity::getSourceId).collect(Collectors.toList());
				loanService.updateColumnValue(sourceLoanIds, "ref_status", "未引用");
			}
		}
		service.removeByIds(vos.stream().map(ProjectRepayVO::getId).collect(Collectors.toList()), true);
		return CommonResponse.success("删除成功！");
	}

	/**
	 * 分页查询
	 *
	 * @param param 参数
	 *
	 * @return IPage<ProjectRepayVO>
	 */
	@PostMapping(value = "/queryList")
	public CommonResponse<JSONObject> queryList(@RequestBody QueryParam param) {
		JSONObject resp = new JSONObject();

		List<String> fuzzyFields = param.getFuzzyFields();
		fuzzyFields.add("billCode");
		fuzzyFields.add("repayUnitName");
		fuzzyFields.add("employeeName");

		param.getParams().put("tenant_id", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
		param.getParams().put("repay_unit_id", new Parameter(QueryParam.IN, iOrgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));

		// 金额合计
		Map<String, BigDecimal> map = service.fetchTotalMny(param);

		param.getOrderMap().put("createTime", QueryParam.DESC);

		IPage<ProjectRepayEntity> page = service.queryPage(param, false);
		List<ProjectRepayVO> projectRepayVOS = BeanMapper.mapList(page.getRecords(), ProjectRepayVO.class);

		resp.put("current", page.getCurrent());
		resp.put("size", page.getSize());
		resp.put("pages", page.getPages());
		resp.put("total", page.getTotal());
		resp.put("records", projectRepayVOS);
		resp.put("totalCurTotalRepayMny", map.get("totalCurTotalRepayMny"));

		return CommonResponse.success("查询列表数据成功！", resp);
	}


	/**
	 * 导出
	 *
	 * @param param    查询参数
	 * @param response 文件流
	 */
	@PostMapping(value = "/excelExport")
	public void excelExport(@RequestBody QueryParam param, HttpServletResponse response) {
		param.setPageIndex(1);
		param.setPageSize(-1);

		List<String> fuzzyFields = param.getFuzzyFields();
		fuzzyFields.add("billCode");
		fuzzyFields.add("repayUnitName");
		fuzzyFields.add("employeeName");

		param.getParams().put("tenant_id", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
		param.getParams().put("repay_unit_id", new Parameter(QueryParam.IN, iOrgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));

		param.getOrderMap().put("createTime", QueryParam.DESC);

		IPage<ProjectRepayEntity> page = service.queryPage(param, false);
		List<ProjectRepayVO> projectRepayVOS = new ArrayList<>();

		page.getRecords().forEach(e -> {
			ProjectRepayVO projectRepayVO = BeanMapper.map(e, ProjectRepayVO.class);
			projectRepayVO.setBillStateName(BillStateEnum.getEnumByStateCode(projectRepayVO.getBillState()).getDescription());
			projectRepayVOS.add(projectRepayVO);
		});

		Map<String, Object> beans = new HashMap<>();
		beans.put("records", projectRepayVOS);
		ExcelExport.getInstance().export("projectRepay-export.xlsx", beans, response);
	}

}
