package com.ejianc.business.profinance.projectloan.service.impl;

import com.ejianc.business.profinance.person.bean.PersonRepayDetailEntity;
import com.ejianc.business.profinance.person.bean.PersonRepayEntity;
import com.ejianc.business.profinance.projectloan.bean.ProjectLoanEntity;
import com.ejianc.business.profinance.projectloan.bean.ProjectRepayDetailEntity;
import com.ejianc.business.profinance.projectloan.bean.ProjectRepayEntity;
import com.ejianc.business.profinance.projectloan.service.IProjectLoanService;
import com.ejianc.business.profinance.projectloan.service.IProjectRepayService;
import com.ejianc.foundation.support.api.IBillTypeApi;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.auth.session.UserContext;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.skeleton.billState.service.ICommonBusinessService;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import java.math.RoundingMode;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

@Service("projectRepay")
public class ProjectRepayBpmServiceImpl implements ICommonBusinessService {
	private final IProjectRepayService service;
	private final IProjectLoanService projectLoanService;
	private final SessionManager sessionManager;
	private final IBillTypeApi billTypeApi;
	private final Logger logger = LoggerFactory.getLogger(this.getClass());

	public ProjectRepayBpmServiceImpl(IProjectRepayService service, IProjectLoanService projectLoanService, SessionManager sessionManager, IBillTypeApi billTypeApi) {
		this.service = service;
		this.projectLoanService = projectLoanService;
		this.sessionManager = sessionManager;
		this.billTypeApi = billTypeApi;
	}


	/**
	 * 终审审核完回调
	 *
	 * @param billId 单据id
	 * @param state  单据状态
	 *
	 * @return 响应信息
	 */
	@Override
	public CommonResponse<String> afterApprovalProcessor(Long billId, Integer state, String billTypeCode) {
		logger.info("终审审核完回调--start，billId={},state={},billTypeCode={}", billId, state, billTypeCode);

		// 审批通过/已提交
		ProjectRepayEntity entity = service.selectById(billId);
		if (entity == null) {
			throw new BusinessException("查询不到单据信息");
		}

		UserContext userContext = sessionManager.getUserContext();

		// 区分提交和审批
		if (state.equals(BillStateEnum.COMMITED_STATE.getBillStateCode())) {
			// 直审更新提交相关字段
			entity.setCommitDate(new Date());
			entity.setCommitUserCode(userContext.getUserCode());
			entity.setCommitUserName(userContext.getUserName());
		}
		entity.setBillStateName(BillStateEnum.getEnumByStateCode(state).getDescription());
		// 生效时间
		entity.setEffectiveDate(new Date());

		// 执行更新
		service.saveOrUpdate(entity, false);

		writeBackProjectLoan(entity, true);
		if(CollectionUtils.isNotEmpty(entity.getProjectRepayDetailList())){
			List<Long> sourceLoanIds = entity.getProjectRepayDetailList().stream().map(ProjectRepayDetailEntity::getSourceId).collect(Collectors.toList());
			projectLoanService.updateColumnValue(sourceLoanIds,"ref_status","未引用");
		}
		logger.info("终审审核完回调--end");
		return CommonResponse.success("终审审核完回调成功");
	}

	/**
	 * 有审批流的撤回前回调
	 *
	 * @param billId 单据id
	 * @param state  单据状态
	 *
	 * @return 响应信息
	 */
	@Override
	public CommonResponse<String> beforeHasBpmBack(Long billId, Integer state, String billTypeCode) {
		return bpmBackCheck(billId, state, billTypeCode);
	}

	/**
	 * 弃审前事件回调
	 *
	 * @param billId 单据id
	 * @param state  单据状态
	 *
	 * @return 响应信息
	 */
	@Override
	public CommonResponse<String> beforeAbstainingProcessor(Long billId, Integer state, String billTypeCode) {
		return bpmBackCheck(billId, state, billTypeCode);
	}

	private CommonResponse<String> bpmBackCheck(Long billId, Integer state, String billTypeCode) {
		ProjectRepayEntity entity = service.selectById(billId);
		if (entity == null) {
			throw new BusinessException("查询不到单据信息");
		}
		//判断当前是否最新单据
		QueryParam queryParam = new QueryParam();
		queryParam.getParams().put("repay_unit_id",new Parameter(QueryParam.EQ,entity.getRepayUnitId()));
		LinkedHashMap<String, String> orderMap = new LinkedHashMap<>();
		orderMap.put("effective_date", QueryParam.DESC);
		queryParam.setOrderMap(orderMap);
		List<ProjectRepayEntity> personRepayEntities = service.queryList(queryParam);
		if (!personRepayEntities.get(0).getId().equals(entity.getId())){
			return CommonResponse.error("当前单据不是最新单据，不能撤回或弃审！");
		}
		writeBackProjectLoan(entity, false);
		if(CollectionUtils.isNotEmpty(entity.getProjectRepayDetailList())){
			List<Long> sourceLoanIds = entity.getProjectRepayDetailList().stream().map(ProjectRepayDetailEntity::getSourceId).collect(Collectors.toList());
			projectLoanService.updateColumnValue(sourceLoanIds,"ref_status","引用");
		}
		return CommonResponse.success("单据撤回成功");
	}

	/**
	 * 回写对应借款单并重新计算借款单的借款余额
	 *
	 * @param entity 项目还款记录
	 * @param flag   是否生效
	 */
	private void writeBackProjectLoan(ProjectRepayEntity entity, boolean flag) {
		List<ProjectRepayDetailEntity> detailList = entity.getProjectRepayDetailList();
		if (CollectionUtils.isNotEmpty(detailList)) {
			Map<Long, ProjectRepayDetailEntity> repayDetailEntityMap = detailList.stream().collect(Collectors.toMap(ProjectRepayDetailEntity::getSourceId, Function.identity()));
			Set<Long> projectLoanIds = repayDetailEntityMap.keySet();
			Collection<ProjectLoanEntity> projectLoanList = projectLoanService.listByIds(projectLoanIds);

			for (ProjectLoanEntity projectLoan : projectLoanList) {
				ProjectRepayDetailEntity repayDetail = repayDetailEntityMap.get(projectLoan.getId());
				if (flag) {
					projectLoan.setRepaidLoanMny(projectLoan.getRepaidLoanMny().add(repayDetail.getCurRepayMny()).setScale(2, RoundingMode.HALF_UP));
				} else {
					projectLoan.setRepaidLoanMny(projectLoan.getRepaidLoanMny().subtract(repayDetail.getCurRepayMny()).setScale(2, RoundingMode.HALF_UP));
				}
				projectLoan.setLeftLoanMny(projectLoan.getLoanMny().subtract(projectLoan.getRepaidLoanMny()).setScale(2, RoundingMode.HALF_UP));
			}

			projectLoanService.saveOrUpdateBatch(new ArrayList<>(projectLoanList), 10, false);

		}
	}

}
