package com.ejianc.business.profinance.projectloan.controller;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.profinance.projectloan.bean.ProjectLoanEntity;
import com.ejianc.business.profinance.projectloan.service.IProjectLoanService;
import com.ejianc.business.profinance.projectloan.vo.ProjectLoanVO;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.foundation.support.api.IBillTypeApi;
import com.ejianc.foundation.support.vo.BillCodeParam;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.collection.ListUtil;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.ExcelExport;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.redisson.api.RList;
import org.redisson.api.RedissonClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.io.Serializable;
import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.ZoneId;
import java.time.temporal.ChronoUnit;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * 借款管理-项目借款实体
 *
 * @author baipengyan
 */
@RestController
@RequestMapping("projectLoan")
public class ProjectLoanController implements Serializable {
	private static final long serialVersionUID = -618353060912763817L;

	private Logger logger = LoggerFactory.getLogger(this.getClass());

	private static final String RULE_CODE = "PROJECT_LOAN";
	private static final String BILL_CODE = "EJCBT202208000001";
	private final IBillTypeApi billTypeApi;
	private final IBillCodeApi billCodeApi;
	private final IOrgApi iOrgApi;
	private final IProjectLoanService service;
	private final SessionManager sessionManager;
	private final RedissonClient redissonClient;

	public ProjectLoanController(IBillTypeApi billTypeApi, IBillCodeApi billCodeApi, IOrgApi iOrgApi, IProjectLoanService service, SessionManager sessionManager, RedissonClient redissonClient) {
		this.billTypeApi = billTypeApi;
		this.billCodeApi = billCodeApi;
		this.iOrgApi = iOrgApi;
		this.service = service;
		this.sessionManager = sessionManager;
		this.redissonClient = redissonClient;
	}

	/**
	 * 新增或者修改
	 *
	 * @param saveOrUpdateVO 实体VO
	 *
	 * @return 实体VO
	 */
	@PostMapping(value = "/saveOrUpdate")
	public CommonResponse<ProjectLoanVO> saveOrUpdate(@RequestBody ProjectLoanVO saveOrUpdateVO) {
		ProjectLoanEntity entity = BeanMapper.map(saveOrUpdateVO, ProjectLoanEntity.class);
		if (entity.getId() == null || entity.getId() == 0) {
			BillCodeParam billCodeParam = BillCodeParam.build(RULE_CODE, InvocationInfoProxy.getTenantid(), saveOrUpdateVO);
			CommonResponse<String> billCode = billCodeApi.generateBillCode(billCodeParam);
			if (billCode.isSuccess()) {
				entity.setBillCode(billCode.getData());
				entity.setRefStatus("未引用");
			} else {
				throw new BusinessException("网络异常， 编码生成失败， 请稍后再试");
			}
		}

		entity.setBillName("项目借款");
		entity.setBillStateName(BillStateEnum.UNCOMMITED_STATE.getDescription());

		service.saveOrUpdate(entity, false);
		ProjectLoanVO vo = BeanMapper.map(entity, ProjectLoanVO.class);
		return CommonResponse.success("保存或修改单据成功！", vo);
	}

	/**
	 * 查询详情
	 *
	 * @param id 主键
	 *
	 * @return 查询结果
	 */
	@GetMapping(value = "/queryDetail")
	public CommonResponse<ProjectLoanVO> queryDetail(Long id) {
		ProjectLoanEntity entity = service.selectById(id);
		ProjectLoanVO vo = BeanMapper.map(entity, ProjectLoanVO.class);
		return CommonResponse.success("查询详情数据成功！", vo);
	}

	/**
	 * 批量删除单据
	 *
	 * @param vos 要删除的单据
	 *
	 * @return 删除结果
	 */
	@PostMapping(value = "/delete")
	public CommonResponse<String> delete(@RequestBody List<ProjectLoanVO> vos) {
		if (ListUtil.isNotEmpty(vos)) {
			for (ProjectLoanVO vo : vos) {
				CommonResponse<String> resp = billTypeApi.checkQuote(BILL_CODE, vo.getId());
				if (!resp.isSuccess()) {
					return CommonResponse.error("删除失败！" + resp.getMsg());
				}
			}
		}
		service.removeByIds(vos.stream().map(ProjectLoanVO::getId).collect(Collectors.toList()), true);
		return CommonResponse.success("删除成功！");
	}

	/**
	 * 分页查询
	 *
	 * @param param 参数
	 *
	 * @return IPage<ProjectLoanVO>
	 */
	@PostMapping(value = "/queryList")
	public CommonResponse<JSONObject> queryList(@RequestBody QueryParam param) {
		JSONObject resp = new JSONObject();

		List<String> fuzzyFields = param.getFuzzyFields();
		fuzzyFields.add("billCode");
		fuzzyFields.add("borrowingUnitName");
		fuzzyFields.add("loanUnitName");
		fuzzyFields.add("memo");
		fuzzyFields.add("employeeName");

		param.getParams().put("tenant_id", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
		param.getParams().put("borrowing_unit_id", new Parameter(QueryParam.IN, iOrgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));

		// 金额合计
		Map<String, BigDecimal> map = service.fetchTotalMny(param);

		param.getOrderMap().put("createTime", QueryParam.DESC);

		IPage<ProjectLoanEntity> page = service.queryPage(param, false);
		List<ProjectLoanVO> projectLoanVOS = BeanMapper.mapList(page.getRecords(), ProjectLoanVO.class);

		// 待还款计算预计还款天数
		if (param.getParams().containsKey("leftLoanMny")) {
			LocalDate nowDate = LocalDate.now();
			projectLoanVOS.forEach(e -> {
				if (e.getExpectRepayDate() != null) {
					e.setExpectRepayDays(ChronoUnit.DAYS.between(nowDate, e.getExpectRepayDate().toInstant().atZone(ZoneId.systemDefault()).toLocalDate()));
				}
			});
		}

		resp.put("current", page.getCurrent());
		resp.put("size", page.getSize());
		resp.put("pages", page.getPages());
		resp.put("total", page.getTotal());
		resp.put("records", projectLoanVOS);
		resp.put("totalLoanMny", map.get("totalLoanMny"));
		resp.put("totalRepaidLoanMny", map.get("totalRepaidLoanMny"));
		resp.put("totalLeftLoanMny", map.get("totalLeftLoanMny"));

		return CommonResponse.success("查询列表数据成功！", resp);
	}


	/**
	 * 导出
	 *
	 * @param param    查询参数
	 * @param response 文件流
	 */
	@PostMapping(value = "/excelExport")
	public void excelExport(@RequestBody QueryParam param, HttpServletResponse response) {
		param.setPageIndex(1);
		param.setPageSize(-1);

		List<String> fuzzyFields = param.getFuzzyFields();
		fuzzyFields.add("billCode");
		fuzzyFields.add("borrowingUnitName");
		fuzzyFields.add("loanUnitName");
		fuzzyFields.add("memo");
		fuzzyFields.add("employeeName");

		param.getParams().put("tenant_id", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
		param.getParams().put("borrowing_unit_id", new Parameter(QueryParam.IN, iOrgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));

		param.getOrderMap().put("createTime", QueryParam.DESC);

		IPage<ProjectLoanEntity> page = service.queryPage(param, false);
		List<ProjectLoanVO> projectLoanVOS1 = new ArrayList<>();
		List<ProjectLoanVO> projectLoanVOS = BeanMapper.mapList(page.getRecords(), ProjectLoanVO.class);

		projectLoanVOS.forEach(e -> {
			LocalDate nowDate = LocalDate.now();
			if (e.getExpectRepayDate() != null) {
				e.setExpectRepayDays(ChronoUnit.DAYS.between(nowDate, e.getExpectRepayDate().toInstant().atZone(ZoneId.systemDefault()).toLocalDate()));
			}
			e.setBillStateName(BillStateEnum.getEnumByStateCode(e.getBillState()).getDescription());
			projectLoanVOS1.add(e);
		});

		Map<String, Object> beans = new HashMap<>();
		beans.put("records", projectLoanVOS1);
		if (param.getParams().containsKey("leftLoanMny")) {
			ExcelExport.getInstance().export("projectLoan-to-be-pay-export.xlsx", beans, response);
		} else {
			ExcelExport.getInstance().export("projectLoan-export.xlsx", beans, response);
		}
	}

	/**
	 * 借款单参照
	 *
	 * @param pageNumber   页数
	 * @param pageSize     大小
	 * @param condition    自定义参数
	 * @param searchObject 模糊查询具体关键字
	 * @param searchText   查询关键字
	 *
	 * @return
	 */
	@GetMapping(value = "/refProjectLoanData")
	public CommonResponse<IPage<ProjectLoanVO>> refProjectLoanData(@RequestParam Integer pageNumber, @RequestParam Integer pageSize,
	                                                               String condition,
	                                                               String searchObject,
	                                                               String searchText) {
		QueryParam param = new QueryParam();
		param.setPageSize(pageSize);
		param.setPageIndex(pageNumber);
		param.setSearchText(searchText);
		param.setSearchObject(searchObject);

		List<String> fuzzyFields = param.getFuzzyFields();
		fuzzyFields.add("billCode");
		fuzzyFields.add("loanUnitName");
		fuzzyFields.add("memo");

		param.getParams().put("tenant_id", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
		param.getParams().put("bill_state", new Parameter(QueryParam.IN, Arrays.asList(BillStateEnum.COMMITED_STATE.getBillStateCode(), BillStateEnum.PASSED_STATE.getBillStateCode())));
		param.getParams().put("left_loan_mny", new Parameter(QueryParam.GT, BigDecimal.ZERO));

		if (StringUtils.isNotEmpty(condition)) {
			JSONObject _con = JSONObject.parseObject(condition);
			if (_con.containsKey("repayUnitId")) {
				param.getParams().put("borrowing_unit_id", new Parameter(QueryParam.EQ, _con.get("repayUnitId")));
			}
			if (_con.containsKey("excludeIds") && CollectionUtils.isNotEmpty(_con.getJSONArray("excludeIds"))) {
				param.getParams().put("id", new Parameter(QueryParam.NOT_IN, _con.getJSONArray("excludeIds")));
			}
		}

		IPage<ProjectLoanEntity> page = service.queryPage(param, false);
		IPage<ProjectLoanVO> pageData = new Page<>(page.getCurrent(), page.getSize(), page.getTotal());
		pageData.setRecords(BeanMapper.mapList(page.getRecords(), ProjectLoanVO.class));

		return CommonResponse.success("查询参照数据成功！", pageData);
	}


	/**
	 * 获取redis的数据
	 *
	 * @param jsonObject key、ids
	 *
	 * @return Map<String, List < ProjectLoanVO>>
	 */
	@PostMapping(value = "/fetchProjectLoanData")
	public CommonResponse<Map<String, List<ProjectLoanVO>>> fetchProjectLoanData(@RequestBody JSONObject jsonObject) {

		Map<String, List<ProjectLoanVO>> data = new HashMap<>();
		if (!jsonObject.containsKey("key")) {
			throw new BusinessException("请求参数不能为空！");
		}
		String key = jsonObject.getString("key");
		RList<ProjectLoanVO> projectLoanList = redissonClient.getList(key);
		if (CollectionUtils.isEmpty(projectLoanList) && jsonObject.containsKey("ids")) {
			List<Long> ids = jsonObject.getJSONArray("ids").toJavaList(Long.class);
			Collection<ProjectLoanEntity> pls = service.listByIds(ids);
			checkRefStatus(pls);
			projectLoanList.addAll(BeanMapper.mapList(pls, ProjectLoanVO.class));
			projectLoanList.expire(Long.parseLong("5"), TimeUnit.MINUTES);
		} else {
			checkRefStatus(BeanMapper.mapList(projectLoanList, ProjectLoanEntity.class));
		}
		data.put(key, BeanMapper.mapList(redissonClient.getList(key), ProjectLoanVO.class));
		return CommonResponse.success("查询成功！", data);
	}

	private static void checkRefStatus(Collection<ProjectLoanEntity> pls) {
		for (ProjectLoanEntity pl : pls) {
			if ("引用".equals(pl.getRefStatus())) {
				throw new BusinessException("操作失败！借款单号[" + pl.getBillCode() + "]存在未生效的[项目还款单]，待生效后才允许继续还款");
			}
		}
	}

}
