package com.ejianc.business.profinance.person.service.impl;

import com.ejianc.business.profinance.person.bean.PersonLoanEntity;
import com.ejianc.business.profinance.person.service.IPersonLoanService;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.support.api.IBillTypeApi;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.auth.session.UserContext;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.skeleton.billState.service.ICommonBusinessService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import java.util.Date;

@Service("personLoan")
public class PersonLoanBpmServiceImpl implements ICommonBusinessService {

    private final Logger logger = LoggerFactory.getLogger(this.getClass());
    private final IOrgApi iOrgApi;
    private final SessionManager sessionManager;
    private final IPersonLoanService service;
    private final IBillTypeApi billTypeApi;

    public PersonLoanBpmServiceImpl(IOrgApi iOrgApi, SessionManager sessionManager, IPersonLoanService service, IBillTypeApi billTypeApi) {
        this.iOrgApi = iOrgApi;
        this.sessionManager = sessionManager;
        this.service = service;
        this.billTypeApi = billTypeApi;

    }

    /**
     * 提交前回调
     *
     * @param billId
     * @param state
     * @return
     */
    @Override
    public CommonResponse<String> beforeSubmitProcessor(Long billId, Integer state, String billTypeCode) {
        return CommonResponse.success("个人借款， 提交前回调成功");
    }

    ;

    /**
     * 提交完回调
     *
     * @param
     * @return
     */
    @Override
    public CommonResponse<String> afterSubmitProcessor(Long billId, Integer state, String billTypeCode) {
        PersonLoanEntity entity = service.selectById(billId);
        UserContext userContext = sessionManager.getUserContext();
        entity.setCommitDate(new Date());
        entity.setCommitUserCode(userContext.getUserCode());
        entity.setCommitUserName(userContext.getUserName());
        service.saveOrUpdate(entity, false);
        return CommonResponse.success("个人借款 审批回调处理成功");
    }

    /**
     * 有审批流的撤回前回调
     *
     * @param billId
     * @param state
     * @return
     */
    @Override
    public CommonResponse<String> beforeHasBpmBack(Long billId, Integer state, String billTypeCode) {
        return CommonResponse.success();
    }

    ;

    /**
     * 有审批流的撤回后回调
     *
     * @param
     * @return
     */
    @Override
    public CommonResponse<String> afterHasBpmBack(Long billId, Integer state, String billTypeCode) {
        return bpmBackCheck(billId, state, billTypeCode);
    }

    ;

    /**
     * 审批节点审批中时节点审批前回调
     *
     * @param billId
     * @param state
     * @return
     */
    @Override
    public CommonResponse<String> beforeInApprovalBack(Long billId, Integer state, String billTypeCode, String sign) {
        return CommonResponse.success();
    }

    ;

    /**
     * 终审审核前回调
     *
     * @param billId
     * @param state
     * @return
     */
    @Override
    public CommonResponse<String> beforeApprovalProcessor(Long billId, Integer state, String billTypeCode) {
        //TODO
        return CommonResponse.success();
    }

    /**
     * 终审审核完回调
     *
     * @param
     * @return
     */
    @Override
    public CommonResponse<String> afterApprovalProcessor(Long billId, Integer state, String billTypeCode) {
        PersonLoanEntity entity = service.selectById(billId);
        if (entity != null) {
            entity.setEffectiveDate(new Date());
        }
        // 区分提交和审批
        if (state.equals(BillStateEnum.COMMITED_STATE.getBillStateCode())) {
            //直审时添加单据提交人等信息
            UserContext userContext = sessionManager.getUserContext();
            entity.setCommitDate(new Date());
            entity.setCommitUserCode(userContext.getUserCode());
            entity.setCommitUserName(userContext.getUserName());
        }
        service.saveOrUpdate(entity, false);
        logger.info("终审审核完回调--end");
        return CommonResponse.success("个人借款 终审回调成功");
    }

    /**
     * 弃审前事件回调
     *
     * @param billId
     * @param state
     * @return
     */
    @Override
    public CommonResponse<String> beforeAbstainingProcessor(Long billId, Integer state, String billTypeCode) {
        return bpmBackCheck(billId, state, billTypeCode);
    }

    /**
     * 弃审后事件回调
     *
     * @param billId
     * @param state
     * @return
     */
    @Override
    public CommonResponse<String> afterAbstainingProcessor(Long billId, Integer state, String billTypeCode) {
        return bpmBackCheck(billId, state, billTypeCode);
    }

    private CommonResponse<String> bpmBackCheck(Long billId, Integer state, String billTypeCode) {
        PersonLoanEntity entity = service.selectById(billId);
        //合同是否被其他单据引用
        CommonResponse<String> res = billTypeApi.checkQuote(billTypeCode, billId);
        logger.info("平台返回查询被引用情况" + res.isSuccess() + "----" + res.getMsg());
        if (!res.isSuccess()) {
            return CommonResponse.error("当前单据已被下游业务引用，不能弃审！");
        }
        return CommonResponse.success("弃审/撤回前校验回调成功！");
    }
}
