package com.ejianc.business.profinance.plan.controller;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.ejianc.business.profinance.plan.bean.PlanEntity;
import com.ejianc.business.profinance.plan.service.*;
import com.ejianc.business.profinance.plan.vo.PlanContractVO;
import com.ejianc.business.profinance.plan.vo.PlanVO;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.share.utils.TreeNodeBUtil;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.foundation.support.api.IBillTypeApi;
import com.ejianc.foundation.support.vo.BillCodeParam;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.collection.ListUtil;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.ExcelExport;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.io.Serializable;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 项目资金计划
 *
 * @author baipengyan
 */
@RestController
@RequestMapping("plan")
public class PlanController implements Serializable {
	private static final long serialVersionUID = -617427815571056809L;

	private Logger logger = LoggerFactory.getLogger(this.getClass());
	private static final String RULE_CODE = "PROJECT_FUNDS_PLAN";
	private static final String BILL_CODE = "EJCBT202208000018";
	private final IBillTypeApi billTypeApi;
	private final IBillCodeApi billCodeApi;
	private final IOrgApi iOrgApi;
	private final SessionManager sessionManager;
	private final IPlanService service;
	private final IPlanContractService planContractService;
	private final IPlanMaterialService planMaterialService;
	private final IPlanTemporaryService planTemporaryService;
	private final IPlanFeeService planFeeService;

	public PlanController(IBillTypeApi billTypeApi,
	                      IBillCodeApi billCodeApi,
	                      IOrgApi iOrgApi,
	                      SessionManager sessionManager,
	                      IPlanService service,
	                      IPlanContractService planContractService,
	                      IPlanMaterialService planMaterialService,
	                      IPlanTemporaryService planTemporaryService,
	                      IPlanFeeService planFeeService) {
		this.billTypeApi = billTypeApi;
		this.billCodeApi = billCodeApi;
		this.iOrgApi = iOrgApi;
		this.sessionManager = sessionManager;
		this.service = service;
		this.planContractService = planContractService;
		this.planMaterialService = planMaterialService;
		this.planTemporaryService = planTemporaryService;
		this.planFeeService = planFeeService;
	}

	/**
	 * 新增或者修改
	 *
	 * @param saveOrUpdateVO 实体VO
	 *
	 * @return 实体VO
	 */
	@PostMapping(value = "/saveOrUpdate")
	public CommonResponse<PlanVO> saveOrUpdate(@RequestBody PlanVO saveOrUpdateVO) {
		PlanEntity entity = BeanMapper.map(saveOrUpdateVO, PlanEntity.class);
		if (entity.getId() == null || entity.getId() == 0) {

			// 校验只能有一个未生效的单据
			service.checkOnlyOne(entity.getProjectId());

			BillCodeParam billCodeParam = BillCodeParam.build(RULE_CODE, InvocationInfoProxy.getTenantid(), saveOrUpdateVO);
			CommonResponse<String> billCode = billCodeApi.generateBillCode(billCodeParam);
			if (billCode.isSuccess()) {
				entity.setBillCode(billCode.getData());
			} else {
				throw new BusinessException("网络异常， 编码生成失败， 请稍后再试");
			}
		}

		entity.setBillName("项目资金计划");
		entity.setBillStateName(BillStateEnum.UNCOMMITED_STATE.getDescription());

		service.setSubDefaultValue(entity);

		service.saveOrUpdate(entity, false);
		PlanVO vo = BeanMapper.map(entity, PlanVO.class);
		vo.setPlanContractList(TreeNodeBUtil.buildTree(vo.getPlanContractList()));
		return CommonResponse.success("保存或修改单据成功！", vo);
	}

	/**
	 * 查询详情
	 *
	 * @param id 主键
	 *
	 * @return 查询结果
	 */
	@GetMapping(value = "/queryDetail")
	public CommonResponse<PlanVO> queryDetail(Long id) {
		PlanEntity entity = service.selectById(id);
		PlanVO vo = BeanMapper.map(entity, PlanVO.class);

		//子表排序为树形
		if (CollectionUtils.isNotEmpty(vo.getPlanContractList())) {
			for (PlanContractVO planContractVO : vo.getPlanContractList()) {
				planContractVO.setTid(planContractVO.getId().toString());
				planContractVO.setTpid(planContractVO.getParentId() != null ? planContractVO.getParentId().toString() : null);
			}
			vo.setPlanContractList(TreeNodeBUtil.buildTree(vo.getPlanContractList()));
		}


		return CommonResponse.success("查询详情数据成功！", vo);
	}

	/**
	 * 批量删除单据
	 *
	 * @param vos 要删除的单据
	 *
	 * @return 删除结果
	 */
	@PostMapping(value = "/delete")
	public CommonResponse<String> delete(@RequestBody List<PlanVO> vos) {
		if (ListUtil.isNotEmpty(vos)) {
			for (PlanVO vo : vos) {
				CommonResponse<String> resp = billTypeApi.checkQuote(BILL_CODE, vo.getId());
				if (!resp.isSuccess()) {
					return CommonResponse.error("删除失败！" + resp.getMsg());
				}
			}
		}
		service.removeByIds(vos.stream().map(PlanVO::getId).collect(Collectors.toList()), true);
		return CommonResponse.success("删除成功！");
	}

	/**
	 * 分页查询
	 *
	 * @param param 参数
	 *
	 * @return IPage<PlanVO>
	 */
	@PostMapping(value = "/queryList")
	public CommonResponse<JSONObject> queryList(@RequestBody QueryParam param) {
		JSONObject resp = new JSONObject();

		List<String> fuzzyFields = param.getFuzzyFields();
		fuzzyFields.add("billCode");
		fuzzyFields.add("parentOrgName");
		fuzzyFields.add("projectName");
		fuzzyFields.add("employeeName");

		param.getParams().put("tenant_id", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));

		//若当前上下文为项目部，则根据项目部Id来进行查询
		if (OrgVO.ORG_TYPE_DEPARTMENT.equals(Integer.valueOf(InvocationInfoProxy.getOrgType()))) {
			param.getParams().put("orgId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getOrgId()));
		} else {
			CommonResponse<List<OrgVO>> orgResp = iOrgApi.findChildrenByParentIdWithoutProjectDept(InvocationInfoProxy.getOrgId());
			if (!orgResp.isSuccess()) {
				logger.error("分页查询失败，获取当前本下组织信息失败, {}", orgResp.getMsg());
				return CommonResponse.error("查询失败，获取组织信息失败！");
			}
			param.getParams().put("parentOrgId", new Parameter(QueryParam.IN,
					orgResp.getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
		}

		param.getOrderMap().put("createTime", QueryParam.DESC);

		IPage<PlanEntity> page = service.queryPage(param, false);
		List<PlanVO> plans = BeanMapper.mapList(page.getRecords(), PlanVO.class);

		resp.put("current", page.getCurrent());
		resp.put("size", page.getSize());
		resp.put("pages", page.getPages());
		resp.put("total", page.getTotal());
		resp.put("records", plans);

		return CommonResponse.success("查询列表数据成功！", resp);
	}


	/**
	 * 导出
	 *
	 * @param param    查询参数
	 * @param response 文件流
	 */
	@PostMapping(value = "/excelExport")
	public void excelExport(@RequestBody QueryParam param, HttpServletResponse response) {
		param.setPageIndex(1);
		param.setPageSize(-1);

		List<String> fuzzyFields = param.getFuzzyFields();
		fuzzyFields.add("billCode");
		fuzzyFields.add("parentOrgName");
		fuzzyFields.add("projectName");
		fuzzyFields.add("employeeName");

		param.getParams().put("tenant_id", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));

		//若当前上下文为项目部，则根据项目部Id来进行查询
		if (OrgVO.ORG_TYPE_DEPARTMENT.equals(Integer.valueOf(InvocationInfoProxy.getOrgType()))) {
			param.getParams().put("orgId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getOrgId()));
		} else {
			CommonResponse<List<OrgVO>> orgResp = iOrgApi.findChildrenByParentIdWithoutProjectDept(InvocationInfoProxy.getOrgId());
			if (!orgResp.isSuccess()) {
				logger.error("分页查询失败，获取当前本下组织信息失败, {}", orgResp.getMsg());
			}
			param.getParams().put("parentOrgId", new Parameter(QueryParam.IN,
					orgResp.getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
		}

		param.getOrderMap().put("createTime", QueryParam.DESC);

		IPage<PlanEntity> page = service.queryPage(param, false);
		List<PlanVO> plans = new ArrayList<>();

		page.getRecords().forEach(e -> {
			PlanVO PlanVO = BeanMapper.map(e, PlanVO.class);
			PlanVO.setBillStateName(BillStateEnum.getEnumByStateCode(PlanVO.getBillState()).getDescription());
			plans.add(PlanVO);
		});

		Map<String, Object> beans = new HashMap<>();
		beans.put("records", plans);
		ExcelExport.getInstance().export("plan-export.xlsx", beans, response);
	}

	/**
	 * 校验引用的参照数据
	 *
	 * @param referMap 参照数据
	 *
	 * @return 校验结果
	 */
	@PostMapping(value = "/checkReferQuote")
	public CommonResponse<String> checkReferQuote(@RequestBody JSONObject referMap) {
		service.checkReferQuote(referMap);
		return CommonResponse.success("查询详情数据成功！");
	}

	/**
	 * 根据项目id获取上期计划日期、累计工程收款(不含本次)、累计产值金额
	 *
	 * @param projectId 项目id
	 *
	 * @return CommonResponse<Map < String, BigDecimal>>
	 */
	@GetMapping(value = "/fetchValue")
	public CommonResponse<Map<String, Object>> fetchValue(@RequestParam(value = "projectId", required = true) Long projectId) {
		return CommonResponse.success("查询详情数据成功！", service.fetchValue(projectId));
	}

	/**
	 * 根据合同ids查询累计结算金额、累计申请预付款(不含本次)、累计已扣减预付款(不含本次)、累计申请进度款(不含本次)、累计已申请付款(不含本次)
	 *
	 * @param contractIds 合同id集合
	 *
	 * @return 累计结算金额、累计申请预付款(不含本次)、累计已扣减预付款(不含本次)、累计申请进度款(不含本次)、累计已申请付款(不含本次)
	 */
	@PostMapping(value = "/fetchContractValue")
	public CommonResponse<Map<String, Object>> fetchContractValue(@RequestBody Collection<Long> contractIds) {
		return CommonResponse.success("查询详情数据成功！", planContractService.fetchContractValue(contractIds));
	}

	/**
	 * 已申请金额(不含本次)=根据结算单ids查询所有审批通过或已提交的【项目资金计划】-【零星材料】页签数据，统计该结算单的“本次计划付款金额”
	 *
	 * @param settleIds 结算单ids
	 *
	 * @return 已申请金额(不含本次)
	 */
	@PostMapping(value = "/fetchMaterialValue")
	public CommonResponse<Map<String, Object>> fetchMaterialValue(@RequestBody Collection<Long> settleIds) {
		return CommonResponse.success("查询详情数据成功！", planMaterialService.fetchMaterialValue(settleIds));
	}

	/**
	 * 已申请金额(不含本次)=根据结算单ids查询所有审批通过或已提交的【项目资金计划】-【临时机械租赁】页签数据，统计该结算单的“本次计划付款金额”
	 *
	 * @param settleIds 结算单ids
	 *
	 * @return 已申请金额(不含本次)
	 */
	@PostMapping(value = "/fetchTemporaryValue")
	public CommonResponse<Map<String, Object>> fetchTemporaryValue(@RequestBody Collection<Long> settleIds) {
		return CommonResponse.success("查询详情数据成功！", planTemporaryService.fetchTemporaryValue(settleIds));
	}

	/**
	 * 已申请金额(不含本次)=根据零星费用单ids查询所有审批通过或已提交的【项目资金计划】-【零星费用】页签数据，统计该结算单的“本次计划付款金额”
	 *
	 * @param feeIds 零星费用单ids
	 *
	 * @return 已申请金额(不含本次)
	 */
	@PostMapping(value = "/fetchFeeValue")
	public CommonResponse<Map<String, Object>> fetchFeeValue(@RequestBody Collection<Long> feeIds) {
		return CommonResponse.success("查询详情数据成功！", planFeeService.fetchFeeValue(feeIds));
	}

}
