package com.ejianc.business.promaterial.utils;

import com.ejianc.framework.core.exception.BusinessException;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.Instant;
import java.time.LocalDate;
import java.time.Period;
import java.time.ZoneId;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.regex.Pattern;

/**
 * 日期处理工具类
 */
public class DateUtils {
    private static Logger logger = LoggerFactory.getLogger(DateUtils.class);
    private DateUtils() {
    }

    /** 日期格式化：yyyy-MM-dd */
    public static final String YYYY_MM_DD = "yyyy-MM-dd";

    /**
     * 计算2个日期之间相差的  以年、月、日为单位，各自计算结果是多少
     * 比如：2011-02-02 到  2017-03-03
     * 计算结果为 1,1  一个月零一天
     *
     * @param fromDate 开始日期
     * @param toDate   结束日期
     * @return 计算结果
     */
    public static int[] dayCompare(Date fromDate, Date toDate) {
        Instant instant = fromDate.toInstant();
        Instant instant1 = toDate.toInstant();
        ZoneId zoneId = ZoneId.systemDefault();
        LocalDate one = instant.atZone(zoneId).toLocalDate();
        LocalDate two = instant1.atZone(zoneId).toLocalDate();
        int year = Period.between(one, two).getYears();
        int month = Period.between(one, two).getMonths();
        int day = Period.between(one, two).getDays();
        month = year * 12 + month;
        return new int[]{month, day};
    }

    /**
     * 计算两个日期之间相差的天数
     *
     * @param smdate 较小的时间
     * @param bdate  较大的时间
     * @return 相差天数
     */
    public static Integer daysBetween(Date smdate, Date bdate) {
        try {
            SimpleDateFormat sdf = new SimpleDateFormat(YYYY_MM_DD);
            smdate = sdf.parse(sdf.format(smdate));
            bdate = sdf.parse(sdf.format(bdate));
            Calendar cal = Calendar.getInstance();
            cal.setTime(smdate);
            long time1 = cal.getTimeInMillis();
            cal.setTime(bdate);
            long time2 = cal.getTimeInMillis();
            long between_days = (time2 - time1) / (1000 * 3600 * 24);
            return Integer.parseInt(String.valueOf(between_days));
        }
        catch (Exception e) {
            logger.error("计算日期失败，失败原因：{}", e.getMessage(), e);
            throw new BusinessException("计算日期失败，失败原因：" + e.getMessage());
        }
    }

    /**
     * 计算月份，加几月或者减几月  整数  往后推,负数往前移动
     *
     * @param date  需要计算的日期
     * @param month 计算月数
     * @return 计算结果
     */
    public static Date calculationMonth(Date date, Integer month) {
        Calendar calendar = new GregorianCalendar();
        calendar.setTime(date);
        calendar.add(Calendar.MONTH, month); //把日期往后增加一天,整数  往后推,负数往前移动
        return calendar.getTime(); //这个时间就是日期往后推一天的结果
    }

    /**
     * 计算日期，加几天或者减几天  整数  往后推,负数往前移动
     *
     * @param date 需要计算的日期
     * @param days 计算天数
     * @return 计算结果
     */
    public static Date calculationDay(Date date, Integer days) {
        Calendar calendar = new GregorianCalendar();
        calendar.setTime(date);
        calendar.add(Calendar.DATE, days); //把日期往后增加一天,整数  往后推,负数往前移动
        return calendar.getTime(); //这个时间就是日期往后推一天的结果
    }

    /**
     * 创建格式为YYYY-MM-DD的日期
     *
     * @param date 日期
     * @return 创建结果
     */
    public static Date createDate(String date) {
        return createDate(date, YYYY_MM_DD);
    }

    /**
     * 创建日期
     *
     * @param date    需要创建日期的对象
     * @param pattern 日期格式
     * @return 创建结果
     */
    public static Date createDate(String date, String pattern) {
        try {
            return new SimpleDateFormat(pattern).parse(date);
        }
        catch (Exception e) {
            return null;
        }
    }

    /**
     * 格式化日期
     *
     * @param date    需要格式化的日期
     * @param pattern 格式
     * @return 格式化结果
     */
    public static String transformationDate(Date date, String pattern) {
        if (null == date) return null;
        return new SimpleDateFormat(pattern).format(date);
    }

    /**
     * date2String  默认yyyy-MM-dd
     *
     * @param date 日期
     * @return 转换结果
     */
    public static String transformationDefaultDate(Date date) {
        if (null == date) return null;
        return transformationDate(date, YYYY_MM_DD);
    }

    /**
     * 格式化日期
     *
     * @param date    需要格式化的日期
     * @param pattern 格式
     * @return 格式化结果
     */
    public static Date formatDate(Date date, String pattern) {
        return createDate(transformationDate(date, pattern), pattern);
    }

    /**
     * 将任意日期格式字符串，重新翻译为标准日期格式
     * @param str
     * @return
     * @throws ParseException
     */
    public static String formatExcelDate(String str) throws ParseException {
        if(StringUtils.isEmpty(str)){
            return str;
        }
        Date date = new Date();
        try {
            Double par = Double.parseDouble(str);
            date = DoubleToDate(par);
        } catch (NumberFormatException e) {
            SimpleDateFormat bsdf = new SimpleDateFormat(getDateFormat(str));
            date = bsdf.parse(str);
        }
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");
        return sdf.format(date);
    }

    public static Date DoubleToDate(Double dVal) {
        Date tDate = new Date();
        long localOffset = tDate.getTimezoneOffset() * 60000; //系统时区偏移 1900/1/1 到 1970/1/1 的 25569 天
        tDate.setTime((long) ((dVal - 25569) * 24 * 3600 * 1000 + localOffset));
        return tDate;
    }

    /**
     * 将任意日期格式字符串，重新翻译为标准日期格式
     * @param str
     * @return
     * @throws ParseException
     */
    public static String formatDateString(String str) {
        if(StringUtils.isEmpty(str)){
            return str;
        }
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");
        SimpleDateFormat bsdf = new SimpleDateFormat(getDateFormat(str));
        String dateStr = null;
        try {
            dateStr = sdf.format(bsdf.parse(str));
        } catch (ParseException e) {
            e.printStackTrace();
        }
        return dateStr;
    }

    /**
     * 常规自动日期格式识别
     * @param str 时间字符串
     * @return Date
     * @author dc
     */
    public static String getDateFormat(String str) {
        boolean year = false;
        Pattern pattern = Pattern.compile("^[-\\+]?[\\d]*$");
        if(pattern.matcher(str.substring(0, 4)).matches()) {
            year = true;
        }
        StringBuilder sb = new StringBuilder();
        int index = 0;
        if(!year) {
            if(str.contains("月") || str.contains("-") || str.contains("/")) {
                if(Character.isDigit(str.charAt(0))) {
                    index = 1;
                }
            }else {
                index = 3;
            }
        }
        for (int i = 0; i < str.length(); i++) {
            char chr = str.charAt(i);
            if(Character.isDigit(chr)) {
                if(index==0) {
                    sb.append("y");
                }
                if(index==1) {
                    sb.append("M");
                }
                if(index==2) {
                    sb.append("d");
                }
                if(index==3) {
                    sb.append("H");
                }
                if(index==4) {
                    sb.append("m");
                }
                if(index==5) {
                    sb.append("s");
                }
                if(index==6) {
                    sb.append("S");
                }
            }else {
                if(i>0) {
                    char lastChar = str.charAt(i-1);
                    if(Character.isDigit(lastChar)) {
                        index++;
                    }
                }
                sb.append(chr);
            }
        }
        return sb.toString();
    }
}
