package com.ejianc.business.process.controller;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.process.bean.SalaryEntity;
import com.ejianc.business.process.service.ISalaryService;
import com.ejianc.business.process.vo.SalaryVO;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.foundation.support.vo.BillCodeParam;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.collection.ListUtil;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.ExcelExport;
import com.ejianc.framework.core.util.ImportTemplate;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.math.BigDecimal;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
/**
 * 工人工资单主实体
 *
 * @author generator
 *
 */
@Controller
@RequestMapping("salary")
public class SalaryController {
    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private IBillCodeApi billCodeApi;

    @Autowired
    private IOrgApi iOrgApi;

    private static final String BILL_CODE = "SALARY_CODE";//此处需要根据实际修改
    private static final String regex = "^\\d{16}|\\d{19}$";

    @Autowired
    private ISalaryService service;

    @Autowired
    private SessionManager sessionManager;

    /**
     * @Description saveOrUpdate 新增或者修改
     */
    @RequestMapping(value = "/saveOrUpdate", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<SalaryVO> saveOrUpdate(@RequestBody SalaryVO saveOrUpdateVO) {
    	SalaryEntity entity = BeanMapper.map(saveOrUpdateVO, SalaryEntity.class);
    	if(entity.getId() == null || entity.getId() == 0){
            BillCodeParam billCodeParam = BillCodeParam.build(BILL_CODE,InvocationInfoProxy.getTenantid(),saveOrUpdateVO);
            CommonResponse<String> billCode = billCodeApi.generateBillCode(billCodeParam);
            if(billCode.isSuccess()) {
                entity.setBillCode(billCode.getData());//此处需要根据实际修改 删除本行或者上一行
            }else{
                throw new BusinessException("网络异常， 编码生成失败， 请稍后再试");
            }
        }
    	service.saveOrUpdate(entity, false);
    	SalaryVO vo = BeanMapper.map(entity, SalaryVO.class);
    	return CommonResponse.success("保存或修改单据成功！",vo);
    }

    /**
     * @Description queryDetail 查询详情
     * @param id
     */
    @RequestMapping(value = "/queryDetail", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<SalaryVO> queryDetail(Long id) {
    	SalaryEntity entity = service.selectById(id);
    	SalaryVO vo = BeanMapper.map(entity, SalaryVO.class);
        return CommonResponse.success("查询详情数据成功！",vo);
    }

    /**
     * @Description delete 批量删除单据
     * @Param [ids]
     */
    @RequestMapping(value = "/delete", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<String> delete(@RequestBody List<SalaryVO> vos) {
        if(ListUtil.isNotEmpty(vos)){
            for (SalaryVO vo : vos) {
                // 参数是单据类型编码字符串 根据需求是否打开下面代码
                /* CommonResponse<String> resp = billTypeApi.checkQuote("billTypeCode", vo.getId());
                if(!resp.isSuccess()){
                    return CommonResponse.error("删除失败！"+resp.getMsg());
                }*/
            }
        }
        service.removeByIds(vos.stream().map(SalaryVO::getId).collect(Collectors.toList()),true);
        return CommonResponse.success("删除成功！");
    }

    /**
     * @Description queryList 查询列表
     * @param param
     * @Return com.ejianc.framework.core.response.CommonResponse<java.lang.String>
     */
    @RequestMapping(value = "/queryList", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<IPage<SalaryVO>> queryList(@RequestBody QueryParam param) {


        /** 模糊搜索配置字段示例 */
        List<String> fuzzyFields = param.getFuzzyFields();
        fuzzyFields.add("projectName");
        fuzzyFields.add("contractName");
        /** 租户隔离 */
        param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));


        //查询本下范围内日的合同
        Long orgId = InvocationInfoProxy.getOrgId();
        CommonResponse<List<OrgVO>> orgResp = iOrgApi.findChildrenByParentId(orgId);
        if(!orgResp.isSuccess()) {
            logger.error("分页查询失败，获取当前本下组织信息失败, {}", orgResp.getMsg());
            return CommonResponse.error("查询失败，获取组织信息失败！");
        }
        List<Long> orgIdList = orgResp.getData().stream().map(OrgVO::getId).collect(Collectors.toList());
        param.getParams().put("orgId", new Parameter(QueryParam.IN, orgIdList));

        IPage<SalaryEntity> page = service.queryPage(param,false);
        IPage<SalaryVO> pageData = new Page<>(page.getCurrent(), page.getSize(), page.getTotal());
        pageData.setRecords(BeanMapper.mapList(page.getRecords(), SalaryVO.class));
        return CommonResponse.success("查询列表数据成功！",pageData);
    }

    /**
     * 获取RPC数据
     * resp 返回值
     * isMustSuc 是否必须成功
     * errMsg 失败提示
     */
    private Object getRespData(CommonResponse<?> resp, boolean isMustSuc, String errMsg) {
        if(isMustSuc && !resp.isSuccess()) {
            throw new BusinessException(StringUtils.isNoneBlank(errMsg) ? errMsg : "调用Rpc服务失败");
        }
        return resp.getData();
    }


    /**
     * @Description 导出
     * @param param
     * @Return void
     */
    @RequestMapping(value = "/excelExport", method = RequestMethod.POST)
    @ResponseBody
    public void excelExport(@RequestBody QueryParam param, HttpServletResponse response) {
        /** 模糊搜索配置字段示例 */
        List<String> fuzzyFields = param.getFuzzyFields();
        param.getParams().put("tenant_id",new Parameter(QueryParam.EQ,InvocationInfoProxy.getTenantid()));
        param.setPageIndex(1);
        param.setPageSize(-1);
        /** 数据隔离 本下 没有组织orgId的删除下面代码 */
        param.getParams().put("orgId",new Parameter(QueryParam.IN,iOrgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        List<SalaryEntity> list = service.queryList(param);
        //todo:字段翻译等等
        Map<String, Object> beans = new HashMap<>();
        beans.put("records", list);
        ExcelExport.getInstance().export("Salary-export.xlsx", beans, response);
    }

    /**
     * @Description 参照
     * @param param
     * @Return void
     */
    @RequestMapping(value = "/refSalaryData", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<IPage<SalaryVO>> refSalaryData(@RequestParam Integer pageNumber, @RequestParam Integer pageSize,
                                                                        String condition,
                                                                        String searchObject,
                                                                        String searchText) {
        QueryParam param = new QueryParam();
        param.setPageSize(pageSize);
        param.setPageIndex(pageNumber);
        param.setSearchText(searchText);
        param.setSearchObject(searchObject);
        List<String> fuzzyFields = param.getFuzzyFields();
        fuzzyFields.add("projectName");
        fuzzyFields.add("contractName");
        /** 租户隔离 */
        param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        param.getParams().put("orgId",new Parameter(QueryParam.IN,iOrgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        if(StringUtils.isNotEmpty(condition)){
            /** 处理condition */
            JSONObject _con = JSONObject.parseObject(condition);
        }
        param.getParams().put("billState",new Parameter(QueryParam.IN, Arrays.asList(
                BillStateEnum.COMMITED_STATE.getBillStateCode(),
                BillStateEnum.PASSED_STATE.getBillStateCode()
        )));
        IPage<SalaryEntity> page = service.queryPage(param,false);
        IPage<SalaryVO> pageData = new Page<>(page.getCurrent(), page.getSize(), page.getTotal());
        pageData.setRecords(BeanMapper.mapList(page.getRecords(), SalaryVO.class));

        return CommonResponse.success("查询参照数据成功！",pageData);
     }
    /**
     * @Description 获取工资汇总
     * @param param
     * @Return void
     */
    @RequestMapping(value = "/queryTotalMny", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<SalaryVO> queryTotalMny(@RequestBody QueryParam param) {


        /** 模糊搜索配置字段示例 */
        List<String> fuzzyFields = param.getFuzzyFields();
        fuzzyFields.add("projectName");
        fuzzyFields.add("contractName");
        /** 租户隔离 */
        param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));

        //查询本下范围内日的合同
        Long orgId = InvocationInfoProxy.getOrgId();
        CommonResponse<List<OrgVO>> orgResp = iOrgApi.findChildrenByParentId(orgId);
        if(!orgResp.isSuccess()) {
            logger.error("分页查询失败，获取当前本下组织信息失败, {}", orgResp.getMsg());
            return CommonResponse.error("查询失败，获取组织信息失败！");
        }
        List<Long> orgIdList = orgResp.getData().stream().map(OrgVO::getId).collect(Collectors.toList());
        param.getParams().put("orgId", new Parameter(QueryParam.IN, orgIdList));
        IPage<SalaryEntity> salaryEntityIPage = service.queryPage(param);
        List<SalaryEntity> salaryEntities = salaryEntityIPage.getRecords();
        BigDecimal reduce = salaryEntities.stream().filter(s -> s.getSumSalaryMny() != null).map(SalaryEntity::getSumSalaryMny).reduce(BigDecimal.ZERO, BigDecimal::add);
        SalaryVO salaryVO = new SalaryVO();
        salaryVO.setSumMny(reduce);
        return CommonResponse.success("获取工资汇总成功！",salaryVO);
    }

    /**
     * @Description checkContract 每个合同每个月的自能有一份
     * @param contractId,belongMonth
     */
    @RequestMapping(value = "/checkContract", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<String> checkContract( Long contractId,String belongMonth ) {
        List<SalaryVO> list= service.checkContract(contractId,belongMonth);
        if (list.size()>0){
            return  CommonResponse.error("该月份已有工资单，不可以再新增工资单！");
        }
        return CommonResponse.success("该月份无工资单，再新增工资单！");
    }

    /**
     * 导入模板下载
     *
     * @param request
     * @param response
     */
    @RequestMapping(value = "/download")
    @ResponseBody
    public void download(HttpServletRequest request, HttpServletResponse response) {
        ImportTemplate.initialize(response);
        ImportTemplate.templetdownload(request, "salary-import.xlsx", "工人工资单导入模板");
    }

    /**
     * excel导入
     *
     * @param request
     * @return
     */
    @RequestMapping(value = "/excelImport", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<JSONObject> excelImport(HttpServletRequest request, HttpServletResponse response) {
        CommonResponse<JSONObject> commonResponse=  service.excelImport(request,response);
        return commonResponse;
    }

}
