package com.ejianc.business.settle.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.ejianc.business.process.enums.BillPushStatusEnum;
import com.ejianc.business.process.enums.SupplierSignStatusEnum;
import com.ejianc.business.settle.bean.SettleEntity;
import com.ejianc.business.settle.enums.SettleTypeEnum;
import com.ejianc.business.settle.enums.SignatureStatusEnum;
import com.ejianc.business.settle.service.INodeSettleService;
import com.ejianc.business.settle.service.ISettleService;
import com.ejianc.business.settle.vo.SettleVO;
import com.ejianc.foundation.share.api.IShareCooperateApi;
import com.ejianc.foundation.share.vo.CooperateVO;
import com.ejianc.foundation.support.api.IBillTypeApi;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.auth.session.UserContext;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.skeleton.billState.service.ICommonBusinessService;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

/**
 * @author baipengyan
 */
@Service("nodeSettle")
public class NodeSettleBpmServiceImpl implements ICommonBusinessService {

    private final Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private IBillTypeApi billTypeApi;

    @Autowired
    private INodeSettleService service;

    @Autowired
    private ISettleService settleService;

    @Autowired
    private SessionManager sessionManager;

    @Autowired
    private IShareCooperateApi shareCooperateApi;


    /**
     * 提交前回调
     *
     * @param billId
     * @param state
     * @return
     */
    @Override
    public CommonResponse<String> beforeSubmitProcessor(Long billId, Integer state, String billTypeCode) {
/*         // 结算单日期校验
        SettleEntity settleEntity = service.selectById(billId);
        if (settleEntity == null) {
            return CommonResponse.error("查询不到该单据信息！");
        }
        // 打印结算单日志
        logger.info("结算单提交前回调，结算单信息：{}", JSON.toJSONString(settleEntity));
        QueryWrapper<SettleEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("contract_id", settleEntity.getContractId());
        queryWrapper.select("ifnull(max(settle_date),0) as max_settle_date");
        Map<String, Object> map = service.getMap(queryWrapper);
        // 打印结果日志
        logger.info("查询结算单最大日期：{}", JSON.toJSONString(map));
        // 结算单日期不能小于等于最大结算日期
        if (settleEntity.getSettleDate().compareTo((Date) map.get("max_settle_date")) <= 0) {
            return CommonResponse.error("结算单日期不能小于等于最大结算日期！");
        } */
        return CommonResponse.success("节点结算-提交前回调成功！");
    }

    /**
     * 终审审核完回调
     *
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public CommonResponse<String> afterApprovalProcessor(Long billId, Integer state, String billTypeCode) {
        SettleEntity settleEntity = service.selectById(billId);
        if (settleEntity != null) {
            settleEntity.setEffectiveDate(new Date());
        }
        if (state.equals(BillStateEnum.COMMITED_STATE.getBillStateCode())) {
            //直审时添加单据提交人等信息
            UserContext userContext = sessionManager.getUserContext();
            settleEntity.setCommitDate(new Date());
            settleEntity.setCommitUserCode(userContext.getUserCode());
            settleEntity.setCommitUserName(userContext.getUserName());
        }

        //查询该单据是否支持协同分享，则向供方协同服务推送该单据
        CommonResponse<CooperateVO> cooperateResp = shareCooperateApi.queryCooperateBybillTypeCode(billTypeCode);
        if (!cooperateResp.isSuccess()) {
            logger.error("根据单据类型-{}查询其协同配置信息失败, 不进行单据推送操作，{}", billTypeCode, cooperateResp.getMsg());
            settleEntity.setBillPushFlag(BillPushStatusEnum.未成功推送.getStatus());
        }

        service.saveOrUpdate(settleEntity, false);

        // 协同推送
        if (cooperateResp.isSuccess()) {
            settleService.pushBillToSupCenter(BeanMapper.map(settleEntity, SettleEntity.class), billTypeCode, cooperateResp.getData());
        }

        // 结算池推送
        String msg = settleService.pushSettleToPool(BeanMapper.map(settleEntity, SettleVO.class));
        if(StringUtils.isNotBlank(msg)) {
            logger.error("节点结算单-【{}】推送结算池失败, 原因：{}", JSONObject.toJSONString(settleEntity), msg);
        }

        // 结算审批通过后/撤回后更新合同池累计结算金额（含税、无税、税额），type：是审批通过还是撤回标识，"approve"审批通过后，"back"撤回后
        updateContractPoolTotalSettleMnyAfterSettle(settleEntity, "approve");

        return CommonResponse.success("节点结算终审回调成功！");
    }

    /**
     * 有审批流的撤回前回调
     *
     * @param billId
     * @param state
     * @return
     */
    @Override
    public CommonResponse<String> beforeHasBpmBack(Long billId, Integer state, String billTypeCode) {
        return bpmBackCheck(billId, state, billTypeCode);
    }

    /**
     * 弃审前事件回调
     *
     * @param billId
     * @param state
     * @return
     */
    @Override
    public CommonResponse<String> beforeAbstainingProcessor(Long billId, Integer state, String billTypeCode) {
        return bpmBackCheck(billId, state, billTypeCode);
    }

    @Transactional(rollbackFor = Exception.class)
    public CommonResponse<String> bpmBackCheck(Long billId, Integer state, String billTypeCode) {

        // 签章流程校验
        SettleEntity settleEntity = service.selectById(billId);
        if (settleEntity == null) {
            return CommonResponse.error("查询不到该单据信息！");
        }

        logger.info("节点结算弃审/撤回前回调结算实体信息为：{}", JSONObject.toJSONString(settleEntity, SerializerFeature.PrettyFormat, SerializerFeature.WriteMapNullValue));

        //检测单据是否供应商已签字，
        if (SupplierSignStatusEnum.乙方已签字.getCode().equals(settleEntity.getSupplierSignStatus())) {
            return CommonResponse.error("供应商已签字，无法执行此操作！");
        }

        // 审批可以撤回的签章状态集合
        ArrayList<Integer> statusList = new ArrayList<>();
        statusList.add(SignatureStatusEnum.未签章.getCode());
        statusList.add(SignatureStatusEnum.已退回.getCode());
        statusList.add(SignatureStatusEnum.已撤回.getCode());
        statusList.add(SignatureStatusEnum.已过期.getCode());

        if (!statusList.contains(settleEntity.getSignatureStatus())) {
            return CommonResponse.error("当前单据已走签章流程，不能弃审/撤回！");
        }

        //若当前合同有完工结算单子则不能弃审/撤回！
        LambdaQueryWrapper<SettleEntity> lambda = new LambdaQueryWrapper<>();
        lambda.eq(SettleEntity::getContractId, settleEntity.getContractId());
        lambda.eq(SettleEntity::getSettleType, SettleTypeEnum.完工.getCode());
        int resultCount = settleService.count(lambda);
        logger.info("查询完工结算信息：contractId-" + settleEntity.getContractId());
        logger.info("查询结果：resultCount-" + resultCount);
        if (resultCount > 0) {
            return CommonResponse.error("当前单据所用合同已被完工结算引用，不能弃审/撤回！");
        }

        //若当前合同有其他结算的自由态的结算单，也不能弃审/撤回！  24-06-25 非自由态也不可以撤回
        LambdaQueryWrapper<SettleEntity> settleLambda = new LambdaQueryWrapper<>();
        settleLambda.eq(SettleEntity::getContractId, settleEntity.getContractId());
        settleLambda.ne(SettleEntity::getId, billId);
        //settleLambda.and(c -> c.notIn(SettleEntity::getBillState, BillStateEnum.PASSED_STATE.getBillStateCode(), BillStateEnum.COMMITED_STATE.getBillStateCode()));

        List<SettleEntity> resultList = settleService.list(settleLambda);

        if (resultList.size() > 0 && null != resultList.get(0)){
            String settleTypeName = (null != resultList.get(0).getSettleType()) ? SettleTypeEnum.getDescriptionByCode(resultList.get(0).getSettleType()).getDescription() : "";
            return CommonResponse.error("当前合同有未审批的" + settleTypeName + "结算单，不能弃审/撤回!");
        }

        //若当前过程结算单不是最新的，则不能弃审/撤回
        QueryParam param = new QueryParam();
        param.getParams().put("contract_id", new Parameter(QueryParam.EQ, settleEntity.getContractId()));
        param.getOrderMap().put("create_time", QueryParam.DESC);
        List<SettleEntity> settleList = settleService.queryList(param);
        logger.info("查询节点结算信息：contractId-" + settleEntity.getContractId());
        if (CollectionUtils.isNotEmpty(settleList) && null != settleList.get(0)){
            logger.info("查询最新的节点结算单结果：{}", JSONObject.toJSONString(settleList.get(0), SerializerFeature.PrettyFormat, SerializerFeature.WriteMapNullValue));
            SettleEntity nodeSettle = settleList.get(0);
            logger.info("nodeSettle.id-" + nodeSettle.getId());
            logger.info("billId-" + billId);
            if (!nodeSettle.getId().equals(billId)){
                //为32部关闭
                return CommonResponse.error("当前单据已有新的结算单，不能弃审/撤回！");
            }
        }


        //合同是否被其他单据引用
        CommonResponse<String> res = billTypeApi.checkQuote(billTypeCode, billId);
        logger.info("平台返回查询被引用情况" + res.isSuccess() + "----" + res.getMsg());

        if (!res.isSuccess()) {
            return CommonResponse.error("当前单据已被下游业务引用，不能弃审/撤回！");
        }

        //提交或审批通过的单据进行撤回，需更新结算池、合同池
        if (BillStateEnum.PASSED_STATE.getBillStateCode().equals(settleEntity.getBillState()) || BillStateEnum.COMMITED_STATE.getBillStateCode().equals(settleEntity.getBillState())){
            // 从结算池中删除数据
            String unPushRes = settleService.delSettleFromPool(settleEntity.getId());
            if (StringUtils.isBlank(unPushRes)) {
                settleEntity.setPushPoolFlag(BillPushStatusEnum.未成功推送.getStatus());
            } else {
                logger.error("将节点结算单-{}从结算池删除失败！", settleEntity);
                return CommonResponse.error("操作失败，单据从结算池删除失败，原因：" + unPushRes);
            }

            // 结算审批通过后/撤回后更新合同池累计结算金额（含税、无税、税额），type：是审批通过还是撤回标识，"approve"审批通过后，"back"撤回后
            updateContractPoolTotalSettleMnyAfterSettle(settleEntity, "back");
        }

        //将推送至供方的单据作废
        if (BillPushStatusEnum.推送成功.getStatus().equals(settleEntity.getBillPushFlag())) {

            boolean pushResult = settleService.delPushBill(settleEntity, billTypeCode);
            if (!pushResult) {
                return CommonResponse.error("节点结算审批回调失败，删除协同服务单据失败！");
            } else {
                settleEntity.setBillPushFlag(BillPushStatusEnum.未成功推送.getStatus());
                service.saveOrUpdate(settleEntity, false);
            }
        }

        return CommonResponse.success("弃审/撤回前校验回调成功！");
    }

    private void updateContractPoolTotalSettleMnyAfterSettle(SettleEntity settleEntity, String type) {
        boolean updateRes = settleService.updateContractPoolTotalSettleMnyAfterSettle(settleEntity, type);
        if (updateRes) {
            logger.info("结算审批通过后/撤回后更新合同池累计结算金额（含税、无税、税额）成功");
        } else {
            logger.error("结算审批通过后/撤回后更新合同池累计结算金额（含税、无税、税额）失败");
        }
    }

}
