package com.ejianc.business.quatity.service.impl;

import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.StrUtil;
import com.alibaba.excel.EasyExcel;
import com.alibaba.excel.context.AnalysisContext;
import com.alibaba.excel.event.AnalysisEventListener;
import com.alibaba.excel.exception.ExcelDataConvertException;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.beust.jcommander.internal.Lists;
import com.ejianc.business.quatity.dao.CompanyGoalsDao;
import com.ejianc.business.quatity.entity.CompanyGoalsEntity;
import com.ejianc.business.quatity.model.po.CompanyExcelPo;
import com.ejianc.business.quatity.model.vo.CompanyTargetAddVo;
import com.ejianc.business.quatity.model.vo.CompanyTargetListVo;
import com.ejianc.business.quatity.service.TargetManagementServer;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.util.DateFormater;
import com.ejianc.framework.core.util.ExcelExport;
import com.ejianc.framework.core.util.ImportTemplate;
import com.ejianc.framework.skeleton.template.BaseEntity;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;
import org.springframework.util.ObjectUtils;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.InputStream;
import java.sql.Date;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 * @Author: LCL
 * @Date: 2024/5/13 下午2:08
 * @Description: 目标管理
 */
@Service
@Slf4j
@RequiredArgsConstructor
public class TargetManagementServerImpl implements TargetManagementServer {
    /**
     * 公司目标
     */
    private final CompanyGoalsDao companyGoalsDao;

    /**
     * excel模板名称
     */
    private static final List<String> DOWNLOAD_EXCEL_TEMPLATE_LIST = Lists.newArrayList("company");

    @Override
    public CompanyGoalsEntity companyAdd(CompanyTargetAddVo vo) {
        CompanyGoalsEntity companyGoalsEntity = new CompanyGoalsEntity();
        BeanUtils.copyProperties(vo, companyGoalsEntity);
        try {
            companyGoalsDao.save(companyGoalsEntity);
            return companyGoalsEntity;
        } catch (Exception e) {
            throw new BusinessException("新增失败，原因：" + e.getMessage());
        }
    }

    @Override
    public IPage<CompanyGoalsEntity> companyList(CompanyTargetListVo vo) {
        String targetFileName = vo.getTargetFileName();

        Date startUploadTime = vo.getStartUploadTime();
        Date endUploadTime = vo.getEndUploadTime();

        LambdaQueryWrapper<CompanyGoalsEntity> lambdaQuery = Wrappers.lambdaQuery();
        if (StrUtil.isNotEmpty(targetFileName)) {
            lambdaQuery.like(CompanyGoalsEntity::getTargetFileName, targetFileName);
        }
        if (Objects.nonNull(startUploadTime)) {
            lambdaQuery.between(CompanyGoalsEntity::getUploadTime, startUploadTime, endUploadTime);
        }
        return companyGoalsDao.page(vo.buildPage(), lambdaQuery);

    }

    @Override
    public void companyDel(List<Long> ids) {
        try {
            companyGoalsDao.removeByIds(ids);
        } catch (Exception e) {
            throw new BusinessException("公司目标删除失败：" + e.getMessage());
        }
    }

    @Override
    public void companyExcelExport(List<Long> ids, HttpServletResponse response) {
        List<CompanyGoalsEntity> list = companyGoalsDao.lambdaQuery().in(BaseEntity::getId, ids).list();

        Map<String, Object> beans = new HashMap<>();
        beans.put("records", list);
        try {
            ExcelExport.getInstance().export("companyGoalsExport.xlsx", beans, response);
        } catch (Exception e) {
            throw new BusinessException("公司目标导出失败：" + e.getMessage());
        }
    }

    @Override
    public void importDownTemplateDetail(String excelName, HttpServletRequest request, HttpServletResponse response) {
        try {
            ImportTemplate.initialize(response);
            if (!DOWNLOAD_EXCEL_TEMPLATE_LIST.contains(excelName)) {
                throw new BusinessException("请指定正确的excel模板名称或联系管理人员");
            }
            if ("company".equals(excelName)) {
                ImportTemplate.templetdownload(request, "companyGoalsImport.xlsx", "公司目标导入模板");
            }
        } catch (BusinessException e) {
            throw new BusinessException("公司目标excel模板下载失败：" + e.getMessage());
        }
    }

    @Override
    public void companyExcelImport(MultipartFile file, HttpServletRequest request, HttpServletResponse response) {
        List<CompanyExcelPo> companyExcelPoList = new ArrayList<>();
        if (file == null || file.isEmpty() || ObjectUtils.isEmpty(file.getOriginalFilename())) {
            throw new BusinessException(500, "文件不能为空");
        }
        String fileName = file.getOriginalFilename();
        if (!(fileName.endsWith(".xlsx") || fileName.endsWith(".xls"))) {
            throw new BusinessException(500, "文件类型错误，只支持：xlsx、xls");
        }
        try {
            InputStream inputStream = file.getInputStream();
            EasyExcel.read(inputStream, CompanyExcelPo.class, new AnalysisEventListener<CompanyExcelPo>() {
                        @Override
                        public void invoke(CompanyExcelPo companyExcelPo, AnalysisContext analysisContext) {
                            companyExcelPoList.add(companyExcelPo);
                        }

                        @Override
                        public void doAfterAllAnalysed(AnalysisContext analysisContext) {
                        }
                    })
                    .sheet()
                    .doRead();
        } catch (IOException e) {
            log.info("公司目标导入异常:{}", e.getMessage());
        } catch (ExcelDataConvertException e) {
            throw new BusinessException("公司目标导入异常");
        }
        try {
            List<CompanyGoalsEntity> companyGoalsEntityList = Lists.newArrayList();
            companyExcelPoList.forEach(companyExcelPo -> {
                CompanyGoalsEntity companyGoalsEntity = new CompanyGoalsEntity();
                BeanUtils.copyProperties(companyExcelPo, companyGoalsEntity);
                companyGoalsEntity.setUploadTime(DateUtil.parse(companyExcelPo.getUploadTime(), DateFormater.SHORT_FORMAT).toSqlDate());
                companyGoalsEntityList.add(companyGoalsEntity);
            });
            companyGoalsDao.saveBatch(companyGoalsEntityList);
        } catch (Exception e) {
            throw new BusinessException("公司目标导入保存异常");
        }
    }
}
