package com.ejianc.business.quatity.service.impl;

import cn.hutool.core.util.StrUtil;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.beust.jcommander.internal.Lists;
import com.ejianc.business.quatity.dao.ExperienceFeedbackIssuedDao;
import com.ejianc.business.quatity.dao.ExperienceFeedbackIssuedDetailDao;
import com.ejianc.business.quatity.dao.ReleaseUnitSettingsDao;
import com.ejianc.business.quatity.entity.ExperienceFeedbackIssuedDetailEntity;
import com.ejianc.business.quatity.entity.ExperienceFeedbackIssuedEntity;
import com.ejianc.business.quatity.entity.ReleaseUnitSettingsEntity;
import com.ejianc.business.quatity.model.res.ExperienceFeedbackIssuedRes;
import com.ejianc.business.quatity.model.vo.ExpFeedBackIssuedAddVo;
import com.ejianc.business.quatity.model.vo.ExpFeedBackIssuedEditVo;
import com.ejianc.business.quatity.model.vo.ExpFeedBackReportVo;
import com.ejianc.business.quatity.service.ExperienceFeedbackServer;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.auth.session.UserContext;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.ExcelExport;
import com.ejianc.framework.skeleton.template.BaseEntity;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.servlet.http.HttpServletResponse;
import java.sql.Date;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * @Author: LCL
 * @Date: 2024/5/15 上午10:10
 */
@Service
@Slf4j
@RequiredArgsConstructor
public class ExperienceFeedbackServerImpl implements ExperienceFeedbackServer {
    private final ExperienceFeedbackIssuedDao dao;
    private final ExperienceFeedbackIssuedDetailDao detailDao;

    private final ReleaseUnitSettingsDao releaseUnitSettingsDao;

    private final SessionManager sessionManager;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ExperienceFeedbackIssuedEntity issuedAdd(ExpFeedBackIssuedAddVo vo) {
        UserContext userContext = sessionManager.getUserContext();

        ExperienceFeedbackIssuedEntity mainEntity = new ExperienceFeedbackIssuedEntity();
        BeanUtils.copyProperties(vo, mainEntity);
        if (vo.getIsOneClickRelease()) {
            List<Long> settingIds = vo.getSettingIds();
            if (settingIds.isEmpty()) {
                throw new BusinessException("下达单位配置信息不能为空");
            }
            List<ReleaseUnitSettingsEntity> settingsEntities = releaseUnitSettingsDao.lambdaQuery().in(BaseEntity::getId, settingIds).list();

            List<String> releaseUnitsCodeList = new ArrayList<>(new LinkedHashSet<>(settingsEntities.stream()
                    .map(ReleaseUnitSettingsEntity::getReleaseUnitsId)
                    .collect(Collectors.toList())));

            List<String> releaseUnitsNameList = new ArrayList<>(new LinkedHashSet<>(settingsEntities.stream()
                    .map(ReleaseUnitSettingsEntity::getReleaseUnitsName)
                    .collect(Collectors.toList())));
            //下达单位
            mainEntity.setReleaseUnitsId(JSONObject.toJSONString(releaseUnitsCodeList));
            mainEntity.setReleaseUnitsName(JSONObject.toJSONString(releaseUnitsNameList));
        } else {
            //下达单位
            mainEntity.setReleaseUnitsId(JSONObject.toJSONString(vo.getReleaseUnitsCode()));
            mainEntity.setReleaseUnitsName(JSONObject.toJSONString(vo.getReleaseUnitsName()));
        }
        //完成情况
        mainEntity.setCompletion("未完成");
        //下达时间
        mainEntity.setReleaseTime(new Date(System.currentTimeMillis()));
        //org
        mainEntity.setOrgId(userContext.getOrgId());
        mainEntity.setOrgName(userContext.getOrgName());
        try {
            boolean mainBoolean = dao.save(mainEntity);
            if (mainBoolean) {
                return saveDetail(mainEntity);
            }
        } catch (Exception e) {
            throw new BusinessException("经验反馈下达新增失败，原因：" + e.getMessage());
        }
        return mainEntity;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void issuedDel(List<Long> ids) {
        try {
            dao.removeByIds(ids);
            detailDao.lambdaUpdate().in(ExperienceFeedbackIssuedDetailEntity::getIssuedId, ids).remove();
        } catch (Exception e) {
            throw new BusinessException("经验反馈下达-删除失败：" + e.getMessage());
        }
    }

    @Override
    public ExperienceFeedbackIssuedEntity issuedEdit(ExpFeedBackIssuedEditVo vo) {
        ExperienceFeedbackIssuedEntity updateMainEntity = new ExperienceFeedbackIssuedEntity();
        BeanUtils.copyProperties(vo, updateMainEntity);
        //下达单位
        updateMainEntity.setReleaseUnitsId(JSONObject.toJSONString(vo.getReleaseUnitsCode()));
        updateMainEntity.setReleaseUnitsName(JSONObject.toJSONString(vo.getReleaseUnitsName()));
        try {
            boolean mainBoolean = dao.updateById(updateMainEntity);
            if (mainBoolean) {
                detailDao.lambdaUpdate().eq(ExperienceFeedbackIssuedDetailEntity::getIssuedId, updateMainEntity.getId()).remove();
                return saveDetail(updateMainEntity);
            }
        } catch (Exception e) {
            throw new BusinessException("经验反馈下达修改失败，原因：" + e.getMessage());
        }
        return updateMainEntity;
    }

    @Override
    public ExperienceFeedbackIssuedRes issuedDetail(Long id) {
        ExperienceFeedbackIssuedRes res = new ExperienceFeedbackIssuedRes();
        ExperienceFeedbackIssuedEntity one = dao.lambdaQuery().eq(BaseEntity::getId, id).one();
        if (Objects.isNull(one)) {
            throw new BusinessException("查询失败");
        }
        BeanUtils.copyProperties(one, res);
        List<ExperienceFeedbackIssuedDetailEntity> detailList = detailDao.lambdaQuery().in(ExperienceFeedbackIssuedDetailEntity::getIssuedId, id).list();
        res.setDetailList(detailList);
        return res;
    }

    @Override
    public IPage<ExperienceFeedbackIssuedRes> issuedList(QueryParam param) {
        param.getFuzzyFields().addAll(Arrays.asList("attributionTypeSpecialized", "topic", "substance"
                , "releaseUnitsCode", "releaseUnitsName", "completion", "preparedUserName"));
        param.getOrderMap().put("createTime", QueryParam.DESC);

        IPage<ExperienceFeedbackIssuedEntity> experienceFeedbackIssuedEntityIPage = dao.queryPage(param, false);

        if (Objects.isNull(experienceFeedbackIssuedEntityIPage)) {
            return new Page<>(experienceFeedbackIssuedEntityIPage.getCurrent(),
                    experienceFeedbackIssuedEntityIPage.getSize(), experienceFeedbackIssuedEntityIPage.getTotal());
        }

        List<ExperienceFeedbackIssuedRes> releaseUnitSettingsResList = experienceFeedbackIssuedEntityIPage.getRecords().stream()
                .map(data -> {
                    ExperienceFeedbackIssuedRes res = new ExperienceFeedbackIssuedRes();
                    BeanUtils.copyProperties(data, res);
                    List<ExperienceFeedbackIssuedDetailEntity> detailList = detailDao.lambdaQuery()
                            .in(ExperienceFeedbackIssuedDetailEntity::getIssuedId, data.getId()).list();
                    res.setDetailList(detailList);
                    return res;
                })
                .collect(Collectors.toList());

        return new Page<ExperienceFeedbackIssuedRes>(experienceFeedbackIssuedEntityIPage.getCurrent(),
                experienceFeedbackIssuedEntityIPage.getSize(), experienceFeedbackIssuedEntityIPage.getTotal())
                .setRecords(releaseUnitSettingsResList);
    }

    @Override
    public void issuedExcelExport(List<Long> ids, HttpServletResponse response) {
        List<ExperienceFeedbackIssuedEntity> list = dao.lambdaQuery().in(BaseEntity::getId, ids).list();
        //导出展示类别中文处理
        list.forEach(item -> {
            String releaseUnitsName = item.getReleaseUnitsName();
            String releaseUnitsNames = StrUtil.join(",", JSONObject.parseArray(releaseUnitsName, String.class));
            item.setReleaseUnitsName(releaseUnitsNames);
        });
        Map<String, Object> beans = new HashMap<>();
        beans.put("records", list);
        try {
            ExcelExport.getInstance().export("experienceFeedbackIssuedExport.xlsx", beans, response);
        } catch (Exception e) {
            throw new BusinessException("经验反馈下达导出失败：" + e.getMessage());
        }
    }

    @Override
    public IPage<ExperienceFeedbackIssuedDetailEntity> getTasksList(QueryParam param) {
        param.getFuzzyFields().addAll(Arrays.asList("attributionType", "specialized", "topic", "substance"));

        param.getOrderMap().put("createTime", QueryParam.DESC);
        //获取当前组织树的任务
        param.getParams().put("uploadDepartmentCode", new Parameter(QueryParam.EQ, sessionManager.getUserContext().getOrgId()));

        return detailDao.queryPage(param, false);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void report(ExpFeedBackReportVo vo) {
        Long id = vo.getId();
        ExperienceFeedbackIssuedDetailEntity reportEntity = detailDao.getById(id);
        if (Objects.isNull(reportEntity)) {
            throw new BusinessException("上报数据查询失败");
        }

        reportEntity.setAnnex(vo.getAnnex());
        reportEntity.setPreparedUserCode(vo.getPreparedUserCode());
        reportEntity.setPreparedUserName(vo.getPreparedUserName());
        reportEntity.setUploadTime(new Date(System.currentTimeMillis()));
        reportEntity.setOrgId(sessionManager.getUserContext().getOrgId());
        reportEntity.setOrgName(sessionManager.getUserContext().getOrgName());
        try {
            detailDao.saveOrUpdate(reportEntity);
        } catch (Exception e) {
            throw new BusinessException("上报失败，原因" + e.getMessage());
        }
    }

    private ExperienceFeedbackIssuedEntity saveDetail(ExperienceFeedbackIssuedEntity updateMainEntity) {
        Long issuedId = updateMainEntity.getId();
        String[] attributionTypeSpecialized = updateMainEntity.getAttributionTypeSpecialized().split("-");
        String topic = updateMainEntity.getTopic();
        String substance = updateMainEntity.getSubstance();

        List<String> releaseUnitsCodeList = JSONObject.parseArray(updateMainEntity.getReleaseUnitsId(), String.class);
        List<String> releaseUnitsNameList = JSONObject.parseArray(updateMainEntity.getReleaseUnitsName(), String.class);
        List<ExperienceFeedbackIssuedDetailEntity> detailEntities = Lists.newArrayList();
        for (int i = 0; i < releaseUnitsCodeList.size(); i++) {
            ExperienceFeedbackIssuedDetailEntity detailEntity = new ExperienceFeedbackIssuedDetailEntity();
            detailEntity.setIssuedId(issuedId);
            detailEntity.setReleaseTime(new Date(System.currentTimeMillis()));
            detailEntity.setAttributionType(attributionTypeSpecialized[0]);
            detailEntity.setSpecialized(attributionTypeSpecialized[1]);
            detailEntity.setTopic(topic);
            detailEntity.setSubstance(substance);
            detailEntity.setUploadDepartmentId(releaseUnitsCodeList.get(i));
            detailEntity.setUploadDepartmentName(releaseUnitsNameList.get(i));
            detailEntities.add(detailEntity);
        }
        detailDao.saveBatch(detailEntities);

        return updateMainEntity;
    }
}
