package com.ejianc.business.quatity.util;

import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.util.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.poi.hssf.usermodel.HSSFCell;
import org.apache.poi.hssf.usermodel.HSSFRow;
import org.apache.poi.hssf.usermodel.HSSFSheet;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.xssf.usermodel.XSSFCell;
import org.apache.poi.xssf.usermodel.XSSFRow;
import org.apache.poi.xssf.usermodel.XSSFSheet;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.springframework.web.multipart.MultipartFile;

import java.io.BufferedInputStream;
import java.io.IOException;
import java.math.BigDecimal;
import java.text.DecimalFormat;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;


/**
* @Author: LCL
* @Date: 2024/5/14 下午3:05
* @Description: 平台Excel工具类
*/
public class DetailIndexExcelReader {

    public static List<List<List<String>>> readExcel(MultipartFile file) {
        String originalFileName = file.getOriginalFilename();
        String extName = null;
        originalFileName = originalFileName.replaceAll("\\/|\\/|\\||:|\\?|\\*|\"|<|>|\\p{Cntrl}", "_");
        originalFileName.replaceAll("00.", "");
        extName = FileUtils.getFileExt(originalFileName, false);
        if ("xls".equals(extName)) {
            return read2003Excel(file);
        } else if ("xlsx".equals(extName)) {
            return read2007Excel(file);
        } else {
            throw new BusinessException("不支持的文件类型");
        }
    }

    private static List<List<List<String>>> read2003Excel(MultipartFile file) {
        LinkedList resList = new LinkedList();

        HSSFWorkbook hwb;
        try {
            hwb = new HSSFWorkbook(new BufferedInputStream(file.getInputStream()));
        } catch (IOException var17) {
            throw new BusinessException(var17.getMessage());
        }

        int size = hwb.getNumberOfSheets();

        for(int index = 0; index < size; ++index) {
            LinkedList list = new LinkedList();
            HSSFSheet sheet = hwb.getSheetAt(index);
            String value = null;
            HSSFRow row = null;
            HSSFCell cell = null;
            boolean isEmpty = true;
            int rowFirstCellNum = 0;

            for(int i = sheet.getFirstRowNum() + 1; i < sheet.getPhysicalNumberOfRows(); ++i) {
                isEmpty = true;
                row = sheet.getRow(i);
                if (row != null) {
                    int rowLastCellNum = sheet.getRow(0).getLastCellNum();
                    List<String> linked = new LinkedList();

                    int headLength;
                    for(headLength = rowFirstCellNum; headLength < rowLastCellNum; ++headLength) {
                        cell = row.getCell(headLength);
                        if (cell == null) {
                            linked.add("");
                        } else if (headLength == 0) {
                            // 首列树形编码直接转成文本处理
                            linked.add(get2003DetailIndex(cell));
                        } else {
                            switch(cell.getCellType()) {
                                case 0:
                                    DecimalFormat df = new DecimalFormat("#.########");
                                    value = df.format(cell.getNumericCellValue());
                                    if (null != value && !"".equals(value.trim())) {
                                        String[] item = value.split("[.]");
                                        if (1 < item.length && ("0".equals(item[1]) || "00".equals(item[1]) || "000".equals(item[1]) || "0000".equals(item[1]))) {
                                            value = item[0];
                                        }
                                    }
                                    break;
                                case 1:
                                    value = cell.getStringCellValue().toString();
                                    break;
                                case 2:
                                    value = String.valueOf(cell.getNumericCellValue());
                                    if (value.equals("NaN")) {
                                        value = cell.getStringCellValue().toString();
                                    }
                                    break;
                                case 3:
                                default:
                                    value = cell.getStringCellValue() + "";
                                    break;
                                case 4:
                                    value = " " + cell.getBooleanCellValue();
                            }

                            linked.add(value);
                        }
                    }

                    if (list.size() > 0) {
                        headLength = ((List)list.get(0)).size();
                        int nowLinkLength = linked.size();
                        if (nowLinkLength < headLength) {
                            for(int k = nowLinkLength; k < headLength; ++k) {
                                linked.add(null);
                            }
                        }
                    }

                    if (isEmpty) {
                        Iterator var19 = linked.iterator();

                        while(var19.hasNext()) {
                            Object object = var19.next();
                            if (object != null && StringUtils.isNotBlank(String.valueOf(object))) {
                                isEmpty = false;
                                break;
                            }
                        }
                    }

                    if (isEmpty) {
                        break;
                    }

                    list.add(linked);
                }
            }
            resList.add(list);
        }

        return resList;
    }

    private static List<List<List<String>>> read2007Excel(MultipartFile file) {

        LinkedList resList = new LinkedList();

        XSSFWorkbook xwb;
        try {
            xwb = new XSSFWorkbook(new BufferedInputStream(file.getInputStream()));
        } catch (IOException var17) {
            throw new BusinessException(var17.getMessage());
        }

        int size = xwb.getNumberOfSheets();

        for(int index = 0; index < size; ++index) {
            LinkedList list = new LinkedList();
            XSSFSheet sheet = xwb.getSheetAt(index);
            String value = null;
            XSSFRow row = null;
            XSSFCell cell = null;
            boolean isEmpty = true;
            int rowFirstCellNum = 0;

            for(int i = sheet.getFirstRowNum() + 1; i < sheet.getPhysicalNumberOfRows(); ++i) {
                isEmpty = true;
                row = sheet.getRow(i);
                int rowLastCellNum = sheet.getRow(0).getLastCellNum();
                if (row != null) {
                    List<String> linked = new LinkedList();

                    int headLength;
                    for(headLength = rowFirstCellNum; headLength < rowLastCellNum; ++headLength) {
                        cell = row.getCell(headLength);
                        if (cell == null) {
                            linked.add("");
                        } else if (headLength == 0) {
                            // 首列树形编码直接转成文本处理
                            linked.add(get2007DetailIndex(cell));
                        } else {
                            switch(cell.getCellType()) {
                                case 0:
                                    DecimalFormat df = new DecimalFormat("#.########");
                                    value = df.format(cell.getNumericCellValue());
                                    if (null != value && !"".equals(value.trim())) {
                                        String[] item = value.split("[.]");
                                        if (1 < item.length && ("0".equals(item[1]) || "00".equals(item[1]) || "000".equals(item[1]) || "0000".equals(item[1]))) {
                                            value = item[0];
                                        }
                                    }
                                    break;
                                case 1:
                                    value = cell.getStringCellValue().toString();
                                    break;
                                case 2:
                                    value = String.valueOf(cell.getNumericCellValue());
                                    if (value.equals("NaN")) {
                                        value = cell.getStringCellValue().toString();
                                    }
                                    break;
                                case 3:
                                default:
                                    value = cell.getStringCellValue() + "";
                                    break;
                                case 4:
                                    value = " " + cell.getBooleanCellValue();
                            }

                            linked.add(value);
                        }
                    }

                    if (list.size() > 0) {
                        headLength = ((List)list.get(0)).size();
                        int nowLinkLength = linked.size();
                        if (nowLinkLength < headLength) {
                            for(int k = nowLinkLength; k < headLength; ++k) {
                                linked.add(null);
                            }
                        }
                    }

                    if (isEmpty) {
                        Iterator var19 = linked.iterator();

                        while(var19.hasNext()) {
                            Object object = var19.next();
                            if (object != null && StringUtils.isNotBlank(String.valueOf(object))) {
                                isEmpty = false;
                                break;
                            }
                        }
                    }

                    if (isEmpty) {
                        break;
                    }

                    list.add(linked);
                }
            }
            resList.add(list);
        }

        return resList;
    }

    /**
     * 2003默认转成文本处理
     * @param cell
     * @return
     */
    private static String get2003DetailIndex(HSSFCell cell) {
        String val = "";
        try {
            val = cell.getStringCellValue();
        } catch (Exception e) {
            try {
                val = String.valueOf(new BigDecimal(String.valueOf(cell.getNumericCellValue())).stripTrailingZeros());
            } catch (Exception e1) {
                val = " " + cell.getBooleanCellValue();
            }
        }
        return val;
    }

    /**
     * 2007默认转成文本处理
     * @param cell
     * @return
     */
    private static String get2007DetailIndex(XSSFCell cell) {
        String val = "";
        try {
            val = cell.getStringCellValue();
        } catch (Exception e) {
            try {
                val = String.valueOf(new BigDecimal(String.valueOf(cell.getNumericCellValue())).stripTrailingZeros());
            } catch (Exception e1) {
                val = " " + cell.getBooleanCellValue();
            }
        }
        return val;
    }

    /**
     * 获取excel文件页签数
     * @param file
     * @return
     */
    public static Integer getNumberOfSheets(MultipartFile file){
        String originalFileName = new String(file.getOriginalFilename());
        String extName = null;
        originalFileName = originalFileName.replaceAll("\\/|\\/|\\||:|\\?|\\*|\"|<|>|\\p{Cntrl}", "_");
        originalFileName.replaceAll("00.", "");
        extName = FileUtils.getFileExt(originalFileName, false);
        if ("xls".equals(extName)) {
            HSSFWorkbook hwb;
            try {
                hwb = new HSSFWorkbook(new BufferedInputStream(file.getInputStream()));
            } catch (IOException var17) {
                throw new BusinessException(var17.getMessage());
            }
            return hwb.getNumberOfSheets();
        } else if ("xlsx".equals(extName)) {
            XSSFWorkbook xwb;
            try {
                xwb = new XSSFWorkbook(new BufferedInputStream(file.getInputStream()));
            } catch (IOException var17) {
                throw new BusinessException(var17.getMessage());
            }
            return xwb.getNumberOfSheets();
        } else {
            throw new BusinessException("不支持的文件类型");
        }
    }
}
