package com.ejianc.business.quality.service.impl;

import cn.hutool.core.date.DateUtil;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.beust.jcommander.internal.Lists;
import com.ejianc.business.quality.dao.CompanyGoalsDao;
import com.ejianc.business.quality.entity.CompanyGoalsEntity;
import com.ejianc.business.quality.enums.QualityTypeEnum;
import com.ejianc.business.quality.model.po.CompanyGoalsExcelPo;
import com.ejianc.business.quality.model.vo.CompanyTargetAddVo;
import com.ejianc.business.quality.model.vo.CompanyTargetEditVo;
import com.ejianc.business.quality.service.TargetManagementServer;
import com.ejianc.business.quality.util.DetailIndexExcelReader;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.auth.session.UserContext;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.DateFormater;
import com.ejianc.framework.core.util.ExcelExport;
import com.ejianc.framework.core.util.ExcelReader;
import com.ejianc.framework.core.util.FileUtils;
import com.ejianc.framework.skeleton.template.BaseEntity;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.multipart.MultipartHttpServletRequest;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

/**
 * @Author: LCL
 * @Date: 2024/5/13 下午2:08
 * @Description: 目标管理
 */
@Service
@Slf4j
@RequiredArgsConstructor
public class TargetManagementServerImpl implements TargetManagementServer {
    /**
     * 公司目标
     */
    private final CompanyGoalsDao companyGoalsDao;

    private final SessionManager sessionManager;

    @Override
    public CompanyGoalsEntity companyAdd(CompanyTargetAddVo vo) {
        CompanyGoalsEntity companyGoalsEntity = new CompanyGoalsEntity();
        BeanUtils.copyProperties(vo, companyGoalsEntity);
        try {
            UserContext userContext = sessionManager.getUserContext();
            companyGoalsEntity.setOrgId(userContext.getOrgId());
            companyGoalsEntity.setOrgName(userContext.getOrgName());
            companyGoalsEntity.setCreateUserName(userContext.getUserName());

            companyGoalsDao.save(companyGoalsEntity);
            return companyGoalsEntity;
        } catch (Exception e) {
            throw new BusinessException("新增失败，原因：" + e.getMessage());
        }
    }

    @Override
    public CompanyGoalsEntity companyEdit(CompanyTargetEditVo vo) {
        CompanyGoalsEntity companyGoalsEntity = new CompanyGoalsEntity();
        BeanUtils.copyProperties(vo, companyGoalsEntity);
        try {
            UserContext userContext = sessionManager.getUserContext();
            companyGoalsEntity.setUpdateUserName(userContext.getUserName());
            companyGoalsDao.updateById(companyGoalsEntity);
            return companyGoalsEntity;
        } catch (Exception e) {
            throw new BusinessException("编辑失败，原因：" + e.getMessage());
        }
    }

    @Override
    public IPage<CompanyGoalsEntity> companyList(QueryParam param) {
        List<String> fuzzyFields = param.getFuzzyFields();
        fuzzyFields.add("targetFileName");
        fuzzyFields.add("orgName");
        fuzzyFields.add("parentOrgName");

        //排序
        LinkedHashMap<String, String> orderMap = new LinkedHashMap<>(param.getOrderMap());
        orderMap.put("createTime", QueryParam.DESC);
        param.setOrderMap(orderMap);

        return companyGoalsDao.queryPage(param, false);
    }

    @Override
    public CompanyGoalsEntity companyDetail(Long id) {
        return companyGoalsDao.getById(id);
    }

    @Override
    public void companyDel(List<Long> ids) {
        try {
            companyGoalsDao.removeByIds(ids);
        } catch (Exception e) {
            throw new BusinessException("公司目标删除失败：" + e.getMessage());
        }
    }

    @Override
    public void companyExcelExport(List<Long> ids, HttpServletResponse response) {
        List<CompanyGoalsEntity> list = companyGoalsDao.lambdaQuery().in(BaseEntity::getId, ids).list();
        //导出展示类别中文处理
        list.forEach(item -> {
            String type = QualityTypeEnum.getNameByCode(item.getType());

            item.setType(type);
        });
        Map<String, Object> beans = new HashMap<>();
        beans.put("records", list);
        try {
            ExcelExport.getInstance().export("companyGoalsExport.xlsx", beans, response);
        } catch (Exception e) {
            throw new BusinessException("公司目标导出失败：" + e.getMessage());
        }
    }

    @Override
    public CommonResponse<JSONObject> companyExcelImport(HttpServletRequest request) {
        MultipartHttpServletRequest multipartRequest = (MultipartHttpServletRequest) request;
        Map<String, MultipartFile> fileMap = multipartRequest.getFileMap();
        boolean isFailed = false;
        MultipartFile mf = null;
        List<CompanyGoalsExcelPo> companyGoalsExcelPoList = new ArrayList<>();
        List<CompanyGoalsExcelPo> errorList = new ArrayList<>();

        for (Map.Entry<String, MultipartFile> entity : fileMap.entrySet()) {
            mf = entity.getValue();
            String originalFileName = mf.getOriginalFilename();
            String extName;
            originalFileName = originalFileName.replaceAll("\\/|\\/|\\||:|\\?|\\*|\"|<|>|\\p{Cntrl}", "_");
            originalFileName.replaceAll("00.", "");
            extName = FileUtils.getFileExt(originalFileName, false);
            if (!"xls".equals(extName) && !"xlsx".equals(extName)) {
                isFailed = true;
                break;
            }
        }
        if (isFailed) {
            throw new BusinessException("文件格式不合法");
        } else {
            Integer size = DetailIndexExcelReader.getNumberOfSheets(mf);
            if (size != 1) {
                throw new BusinessException("文件页签不完整，请下载最新模板！");
            }
            List<List<String>> result = ExcelReader.readExcel(mf);
            for (int i = 0; i < result.size(); i++) {
                CompanyGoalsExcelPo companyGoalsExcelPo = getCompanyGoalsExcelPo(result, i);
                companyGoalsExcelPoList.add(companyGoalsExcelPo);
            }
        }


        try {
            List<CompanyGoalsEntity> successList = Lists.newArrayList();
            UserContext userContext = sessionManager.getUserContext();
            companyGoalsExcelPoList.forEach(companyGoalsExcelPo -> {
                CompanyGoalsEntity companyGoalsEntity = new CompanyGoalsEntity();
                BeanUtils.copyProperties(companyGoalsExcelPo, companyGoalsEntity);
                companyGoalsEntity.setUploadTime(DateUtil.parse(companyGoalsExcelPo.getUploadTime(), DateFormater.SHORT_FORMAT).toSqlDate());
                companyGoalsEntity.setCreateUserName(userContext.getUserName());
                successList.add(companyGoalsEntity);
            });

            JSONObject json = new JSONObject();
            json.put("successList", successList);
            json.put("errorList", errorList);
            return CommonResponse.success(json);
        } catch (Exception e) {
            throw new BusinessException("公司目标导入异常");
        }
    }

    @Override
    public void companyExcelImportSave(List<CompanyGoalsEntity> importVos) {
        try {
            companyGoalsDao.saveBatch(importVos);
        } catch (Exception e) {
            throw new BusinessException("公司目标导入保存异常");
        }
    }

    private static CompanyGoalsExcelPo getCompanyGoalsExcelPo(List<List<String>> result, int i) {
        List<String> data = result.get(i);
        CompanyGoalsExcelPo companyGoalsExcelPo = new CompanyGoalsExcelPo();
        //类别
        String type = data.get(0);
        //目标文件名称
        String targetFileName = data.get(1);
        //目标文件
        String targetFile = data.get(2);
        //目标上传日期
        String uploadTime = data.get(3);
        //所属项目部名称
        String orgName = data.get(4);
        //项目部上级组织名称
        String parentOrgName = data.get(5);
        companyGoalsExcelPo.setType(type);
        companyGoalsExcelPo.setTargetFileName(targetFileName);
        companyGoalsExcelPo.setTargetFile(targetFile);
        companyGoalsExcelPo.setTargetFile(targetFile);
        companyGoalsExcelPo.setUploadTime(uploadTime);
        companyGoalsExcelPo.setOrgName(orgName);
        companyGoalsExcelPo.setParentOrgName(parentOrgName);
        return companyGoalsExcelPo;
    }
}
