package com.ejianc.business.quality.service.impl;

import cn.hutool.core.date.DateTime;
import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.RandomUtil;
import cn.hutool.core.util.StrUtil;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.beust.jcommander.internal.Lists;
import com.ejianc.business.quality.dao.InternalAuditsDao;
import com.ejianc.business.quality.dao.NonConformityDao;
import com.ejianc.business.quality.entity.InternalAuditsEntity;
import com.ejianc.business.quality.entity.NonConformityEntity;
import com.ejianc.business.quality.enums.ManagementSystemEnum;
import com.ejianc.business.quality.enums.NonConformityTypeEnum;
import com.ejianc.business.quality.enums.QualityTypeEnum;
import com.ejianc.business.quality.model.po.CompanyGoalsExcelPo;
import com.ejianc.business.quality.model.po.InternalAuditsExcelPo;
import com.ejianc.business.quality.model.po.NonConformityNcrExcelPo;
import com.ejianc.business.quality.model.po.NonConformityObserveExcelPo;
import com.ejianc.business.quality.model.vo.InternalAuditsAddVo;
import com.ejianc.business.quality.model.vo.InternalAuditsEditVo;
import com.ejianc.business.quality.model.vo.NonConformityAddVo;
import com.ejianc.business.quality.model.vo.NonConformityEditVo;
import com.ejianc.business.quality.service.InternalAuditsServer;
import com.ejianc.business.quality.util.DetailIndexExcelReader;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.auth.session.UserContext;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.DateFormater;
import com.ejianc.framework.core.util.ExcelExport;
import com.ejianc.framework.core.util.ExcelReader;
import com.ejianc.framework.core.util.FileUtils;
import com.ejianc.framework.skeleton.template.BaseEntity;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.tuple.Triple;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.multipart.MultipartHttpServletRequest;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @Author: LCL
 * @Date: 2024/5/14 上午9:43
 */
@Service
@Slf4j
@RequiredArgsConstructor
public class InternalAuditsServerImpl implements InternalAuditsServer {
    private final InternalAuditsDao internalAuditsDao;
    private final NonConformityDao nonConformityDao;
    private final SessionManager sessionManager;

    @Override
    public InternalAuditsEntity add(InternalAuditsAddVo vo) {
        InternalAuditsEntity internalAuditsEntity = new InternalAuditsEntity();
        BeanUtils.copyProperties(vo, internalAuditsEntity);
        try {
            UserContext userContext = sessionManager.getUserContext();
            internalAuditsEntity.setOrgId(userContext.getOrgId());
            internalAuditsEntity.setOrgName(userContext.getOrgName());
            internalAuditsEntity.setCreateUserName(userContext.getUserName());
            internalAuditsDao.save(internalAuditsEntity);
            return internalAuditsEntity;
        } catch (Exception e) {
            throw new BusinessException("新增失败，原因：" + e.getMessage());
        }
    }

    @Override
    public InternalAuditsEntity edit(InternalAuditsEditVo vo) {
        InternalAuditsEntity internalAuditsEntity = new InternalAuditsEntity();
        BeanUtils.copyProperties(vo, internalAuditsEntity);
        try {
            UserContext userContext = sessionManager.getUserContext();
            internalAuditsEntity.setUpdateUserName(userContext.getUserName());
            internalAuditsEntity.setCreateUserName(internalAuditsDao.getById(vo.getId()).getCreateUserName());
            internalAuditsDao.updateById(internalAuditsEntity);
            return internalAuditsEntity;
        } catch (Exception e) {
            throw new BusinessException("编辑失败，原因：" + e.getMessage());
        }
    }

    @Override
    public IPage<InternalAuditsEntity> list(QueryParam param) {
        List<String> fuzzyFields = param.getFuzzyFields();
        fuzzyFields.add("auditReportsName");
        fuzzyFields.add("auditPlanName");
        fuzzyFields.add("orgName");


        //排序
        LinkedHashMap<String, String> orderMap = new LinkedHashMap<>(param.getOrderMap());
        orderMap.put("createTime", QueryParam.DESC);
        param.setOrderMap(orderMap);

        return internalAuditsDao.queryPage(param, false);
    }

    @Override
    public InternalAuditsEntity detail(Long id) {
        return internalAuditsDao.getById(id);
    }

    @Override
    public void del(List<Long> ids) {
        try {
            internalAuditsDao.removeByIds(ids);
        } catch (Exception e) {
            throw new BusinessException("内部审核删除失败：" + e.getMessage());
        }
    }

    @Override
    public void internalAuditsExcelExport(List<Long> ids, HttpServletResponse response) {
        List<InternalAuditsEntity> list = internalAuditsDao.lambdaQuery().in(BaseEntity::getId, ids).list();
        list.forEach(item -> {
            String type = QualityTypeEnum.getNameByCode(item.getType());
            item.setType(type);
        });

        Map<String, Object> beans = new HashMap<>();
        beans.put("records", list);
        try {
            ExcelExport.getInstance().export("internalAuditsExport.xlsx", beans, response);
        } catch (Exception e) {
            throw new BusinessException("内部审核导出失败：" + e.getMessage());
        }

    }

    @Override
    public CommonResponse<JSONObject> internalAuditsExcelImport(HttpServletRequest request) {
        MultipartHttpServletRequest multipartRequest = (MultipartHttpServletRequest) request;
        Map<String, MultipartFile> fileMap = multipartRequest.getFileMap();
        boolean isFailed = false;
        MultipartFile mf = null;
        List<InternalAuditsExcelPo> internalAuditsExcelPoList = new ArrayList<>();
        List<CompanyGoalsExcelPo> errorList = new ArrayList<>();

        for (Map.Entry<String, MultipartFile> entity : fileMap.entrySet()) {
            mf = entity.getValue();
            String originalFileName = mf.getOriginalFilename();
            String extName;
            originalFileName = originalFileName.replaceAll("\\/|\\/|\\||:|\\?|\\*|\"|<|>|\\p{Cntrl}", "_");
            originalFileName.replaceAll("00.", "");
            extName = FileUtils.getFileExt(originalFileName, false);
            if (!"xls".equals(extName) && !"xlsx".equals(extName)) {
                isFailed = true;
                break;
            }
        }
        if (isFailed) {
            throw new BusinessException("文件格式不合法");
        }
        Integer size = DetailIndexExcelReader.getNumberOfSheets(mf);
        if (size != 1) {
            throw new BusinessException("文件页签不完整，请下载最新模板！");
        }
        List<List<String>> result = ExcelReader.readExcel(mf);
        for (int i = 0; i < result.size(); i++) {
            InternalAuditsExcelPo internalAuditsExcelPo = getInternalAuditsExcelPo(result, i);
            internalAuditsExcelPoList.add(internalAuditsExcelPo);
        }

        try {
            List<InternalAuditsEntity> successList = Lists.newArrayList();
            UserContext userContext = sessionManager.getUserContext();
            internalAuditsExcelPoList.forEach(companyGoalsExcelPo -> {
                InternalAuditsEntity companyGoalsEntity = new InternalAuditsEntity();
                BeanUtils.copyProperties(companyGoalsExcelPo, companyGoalsEntity);
                companyGoalsEntity.setUploadTime(DateUtil.parse(companyGoalsExcelPo.getUploadTime(), DateFormater.SHORT_FORMAT).toSqlDate());
                companyGoalsEntity.setCreateUserName(userContext.getUserName());
                successList.add(companyGoalsEntity);
            });

            JSONObject json = new JSONObject();
            json.put("successList", successList);
            json.put("errorList", errorList);
            return CommonResponse.success(json);
        } catch (Exception e) {
            throw new BusinessException("内部审核导入异常");
        }
    }

    @Override
    public void excelImportSave(List<InternalAuditsEntity> importVos) {
        try {
            internalAuditsDao.saveBatch(importVos);
        } catch (Exception e) {
            throw new BusinessException("公司目标导入保存异常");
        }
    }

    @Override
    public NonConformityEntity nonConformityAdd(NonConformityAddVo vo) {
        NonConformityEntity nonConformityEntity = new NonConformityEntity();
        BeanUtils.copyProperties(vo, nonConformityEntity);
        try {
            UserContext userContext = sessionManager.getUserContext();
            nonConformityEntity.setOrgId(userContext.getOrgId());
            nonConformityEntity.setOrgName(userContext.getOrgName());
            nonConformityEntity.setCreateUserName(userContext.getUserName());
            nonConformityEntity.setYear(String.valueOf(DateUtil.thisYear()));
            nonConformityDao.save(nonConformityEntity);
            return nonConformityEntity;
        } catch (Exception e) {
            throw new BusinessException("新增失败，原因：" + e.getMessage());
        }
    }

    @Override
    public NonConformityEntity nonConformityEdit(NonConformityEditVo vo) {
        NonConformityEntity nonConformityEntity = new NonConformityEntity();
        BeanUtils.copyProperties(vo, nonConformityEntity);
        try {
            UserContext userContext = sessionManager.getUserContext();
            nonConformityEntity.setUpdateUserName(userContext.getUserName());
            nonConformityEntity.setCreateUserName(nonConformityDao.getById(vo.getId()).getCreateUserName());
            nonConformityDao.updateById(nonConformityEntity);
            return nonConformityEntity;
        } catch (Exception e) {
            throw new BusinessException("编辑失败，原因：" + e.getMessage());
        }
    }

    @Override
    public IPage<NonConformityEntity> nonConformityList(QueryParam param) {
        List<String> fuzzyFields = param.getFuzzyFields();
        fuzzyFields.add("factsName");
        fuzzyFields.add("gslx");
        fuzzyFields.add("orgName");

        //排序
        LinkedHashMap<String, String> orderMap = new LinkedHashMap<>(param.getOrderMap());
        orderMap.put("createTime", QueryParam.DESC);
        param.setOrderMap(orderMap);

        return nonConformityDao.queryPage(param, false);
    }

    @Override
    public NonConformityEntity nonConformityDetail(Long id) {
        return nonConformityDao.getById(id);
    }

    @Override
    public void nonConformityDel(List<Long> ids) {
        try {
            nonConformityDao.removeByIds(ids);
        } catch (Exception e) {
            throw new BusinessException("内部审核-不符合项删除失败：" + e.getMessage());
        }
    }

    @Override
    public CommonResponse<JSONObject> nonConformityExcelImport(String excelType, HttpServletRequest request) {
        if (!"observe".equals(excelType) && !"ncr".equals(excelType)) {
            throw new BusinessException("内部审核-不符合项导入失败，excelType错误");
        }
        MultipartHttpServletRequest multipartRequest = (MultipartHttpServletRequest) request;
        Map<String, MultipartFile> fileMap = multipartRequest.getFileMap();
        boolean isFailed = false;
        MultipartFile mf = null;
        List<NonConformityNcrExcelPo> ncrExcelPoList = new ArrayList<>();
        List<NonConformityObserveExcelPo> observeExcelPoList = new ArrayList<>();

        JSONObject json = new JSONObject();

        for (Map.Entry<String, MultipartFile> entity : fileMap.entrySet()) {
            mf = entity.getValue();
            String originalFileName = mf.getOriginalFilename();
            String extName;
            originalFileName = originalFileName.replaceAll("\\/|\\/|\\||:|\\?|\\*|\"|<|>|\\p{Cntrl}", "_");
            originalFileName.replaceAll("00.", "");
            extName = FileUtils.getFileExt(originalFileName, false);
            if (!"xls".equals(extName) && !"xlsx".equals(extName)) {
                isFailed = true;
                break;
            }
        }
        if (isFailed) {
            throw new BusinessException("文件格式不合法");
        }
        Integer size = DetailIndexExcelReader.getNumberOfSheets(mf);
        if (size != 1) {
            throw new BusinessException("文件页签不完整，请下载最新模板！");
        }
        List<List<String>> result = ExcelReader.readExcel(mf);

        List<NonConformityEntity> successList = Lists.newArrayList();
        //当前登录人信息
        UserContext userContext = sessionManager.getUserContext();
        String userCode = userContext.getUserCode();
        Long orgId = userContext.getOrgId();
        String orgName = userContext.getOrgName();

        List<NonConformityEntity> errorList = new ArrayList<>();


        //观察项
        if ("observe".equals(excelType)) {
            for (int i = 0; i < result.size(); i++) {
                NonConformityObserveExcelPo nonConformityObserveExcelPo = getNonConformityObserveExcelPo(result, i);
                observeExcelPoList.add(nonConformityObserveExcelPo);
            }
            try {
                observeExcelPoList.forEach(companyGoalsExcelPo -> {
                    NonConformityEntity companyGoalsEntity = new NonConformityEntity();
                    BeanUtils.copyProperties(companyGoalsExcelPo, companyGoalsEntity);
                    companyGoalsEntity.setConformityType("观察项");
                    //编制人
                    companyGoalsEntity.setPreparedUserCode(userCode);
                    companyGoalsEntity.setCreateUserName(userContext.getUserName());
                    //上传部门
                    companyGoalsEntity.setOrgId(orgId);
                    companyGoalsEntity.setOrgName(orgName);
                    if (NonConformityTypeEnum.CODELIST.contains(companyGoalsEntity.getGllx()) &&
                            ManagementSystemEnum.CODELIST.contains(companyGoalsEntity.getGstx()) &&
                            QualityTypeEnum.CODELIST.contains(companyGoalsEntity.getType())) {
                        successList.add(companyGoalsEntity);
                    } else {
                        errorList.add(companyGoalsEntity);
                    }
                });
                json.put("successList", successList);
                errorList.forEach(e -> {
                    e.setId(RandomUtil.randomLong());
                    e.setType(QualityTypeEnum.getNameByCode(e.getType()));
                    e.setGllx(NonConformityTypeEnum.getNameByCode(e.getGllx()));
                    e.setGstx(ManagementSystemEnum.getNameByCode(e.getGstx()));
                });
                json.put("errorList", errorList);
                return CommonResponse.success(json);
            } catch (Exception e) {
                throw new BusinessException("不符合观察项异常");
            }

        }
        //NCR
        for (int i = 0; i < result.size(); i++) {
            NonConformityNcrExcelPo conformityNcrExcelPo = getNonConformityNcrExcelPo(result, i);
            ncrExcelPoList.add(conformityNcrExcelPo);
        }

        try {
            ncrExcelPoList.forEach(companyGoalsExcelPo -> {
                NonConformityEntity companyGoalsEntity = new NonConformityEntity();
                BeanUtils.copyProperties(companyGoalsExcelPo, companyGoalsEntity);
                companyGoalsEntity.setConformityType("NCR");
                //编制人
                companyGoalsEntity.setPreparedUserCode(userCode);
                companyGoalsEntity.setCreateUserName(userContext.getUserName());
                //上传部门
                companyGoalsEntity.setOrgId(orgId);
                companyGoalsEntity.setOrgName(orgName);
                if (NonConformityTypeEnum.CODELIST.contains(companyGoalsEntity.getGllx()) &&
                        ManagementSystemEnum.CODELIST.contains(companyGoalsEntity.getGstx()) &&
                        QualityTypeEnum.CODELIST.contains(companyGoalsEntity.getType())) {
                    successList.add(companyGoalsEntity);
                } else {
                    errorList.add(companyGoalsEntity);
                }
            });
            json.put("successList", successList);
            errorList.forEach(e -> {
                e.setId(RandomUtil.randomLong());
                e.setType(QualityTypeEnum.getNameByCode(e.getType()));
                e.setGllx(NonConformityTypeEnum.getNameByCode(e.getGllx()));
                e.setGstx(ManagementSystemEnum.getNameByCode(e.getGstx()));
            });
            json.put("errorList", errorList);
            return CommonResponse.success(json);
        } catch (Exception e) {
            throw new BusinessException("不符合NCR导入异常");
        }
    }

    @Override
    public void nonConformityExcelImportSave(List<NonConformityEntity> importVos) {
        try {
            nonConformityDao.saveBatch(importVos);
        } catch (Exception e) {
            throw new BusinessException("不符合导入保存异常");
        }
    }

    @Override
    public void nonConformityExcelExport(List<Long> ids, HttpServletResponse response) {
        List<NonConformityEntity> list = nonConformityDao.lambdaQuery().in(BaseEntity::getId, ids).list();
        list.forEach(item -> {
            if (StrUtil.isNotEmpty(item.getGllx())) {
                String gllx = NonConformityTypeEnum.getNameByCode(item.getGllx());
                item.setGllx(gllx);
            }

            String type = QualityTypeEnum.getNameByCode(item.getType());
            String gstx = ManagementSystemEnum.getNameByCode(item.getGstx());
            item.setType(type);
            item.setGstx(gstx);
        });

        Map<String, Object> beans = new HashMap<>();
        beans.put("records", list);
        try {
            ExcelExport.getInstance().export("internalAuditsExport.xlsx", beans, response);
        } catch (Exception e) {
            throw new BusinessException("内部审核导出失败：" + e.getMessage());
        }
    }

    @Override
    public CommonResponse<List<Object>> nonConformityGetYear() {
        List<NonConformityEntity> list = nonConformityDao.lambdaQuery().list();
        List<NonConformityEntity> observationsList = list.stream()
                .filter(e -> "观察项".equals(e.getConformityType()))
                .sorted(Comparator.comparing(BaseEntity::getCreateTime))
                .collect(Collectors.toList());

        String oldObservations = DateUtil.format(observationsList.get(0).getCreateTime(), "yyyy");
        String newObservations = DateUtil.format(observationsList.get(observationsList.size() - 1).getCreateTime(), "yyyy");


        List<NonConformityEntity> ncrList = list.stream()
                .filter(e -> "NCR".equals(e.getConformityType()))
                .sorted(Comparator.comparing(BaseEntity::getCreateTime))
                .collect(Collectors.toList());

        String oldNcr = DateUtil.format(ncrList.get(0).getCreateTime(), "yyyy");
        String newNcr = DateUtil.format(ncrList.get(ncrList.size() - 1).getCreateTime(), "yyyy");

        List<Object> objects = Lists.newArrayList();
        Triple<String, String, String> ncr = Triple.of("NCR", oldNcr, newNcr);
        Triple<String, String, String> observations = Triple.of("观察项", oldObservations, newObservations);
        objects.add(ncr);
        objects.add(observations);

        return CommonResponse.success(objects);
    }

    @Override
    public CommonResponse<JSONObject> nonConformityStatistics() {
        List<NonConformityEntity> list = nonConformityDao.list();
        Map<String, Long> gllx = list.stream()
                .filter(e -> e.getGllx() != null)
                .collect(Collectors.groupingBy(NonConformityEntity::getGllx, Collectors.counting()));

        Map<String, Long> gstx = list.stream()
                .filter(e -> e.getGstx() != null)
                .collect(Collectors.groupingBy(NonConformityEntity::getGstx, Collectors.counting()));
        JSONObject jsonObject = new JSONObject();
        jsonObject.put("gllx", gllx);
        jsonObject.put("gstx", gstx);
        return CommonResponse.success(jsonObject);
    }

    private static NonConformityNcrExcelPo getNonConformityNcrExcelPo(List<List<String>> result, int i) {
        List<String> data = result.get(i);
        NonConformityNcrExcelPo conformityNcrExcelPo = new NonConformityNcrExcelPo();
        //类别
        String type = data.get(0);
        //不符合事实名称
        String factsName = data.get(1);
        //不符合事实url
        String facts = data.get(2);
        //不符合标准号(NCR)
        String bfhbzh = data.get(3);
        //归属体系
        String gstx = data.get(4);

        conformityNcrExcelPo.setType(QualityTypeEnum.getNameByCode(type));
        conformityNcrExcelPo.setFactsName(factsName);
        conformityNcrExcelPo.setFacts(facts);
        conformityNcrExcelPo.setBfhbzh(bfhbzh);
        conformityNcrExcelPo.setGstx(ManagementSystemEnum.getCodeByName(gstx));
        return conformityNcrExcelPo;
    }

    private static NonConformityObserveExcelPo getNonConformityObserveExcelPo(List<List<String>> result, int i) {
        List<String> data = result.get(i);
        NonConformityObserveExcelPo nonConformityObserveExcelPo = new NonConformityObserveExcelPo();
        //类别
        String type = data.get(0);
        //不符合事实名称
        String factsName = data.get(1);
        //不符合事实url
        String facts = data.get(2);
        //管理类型(观察项)
        String gllx = data.get(3);
        //归属体系
        String gstx = data.get(4);

        nonConformityObserveExcelPo.setType(QualityTypeEnum.getCodeByName(type));
        nonConformityObserveExcelPo.setFactsName(factsName);
        nonConformityObserveExcelPo.setFacts(facts);
        nonConformityObserveExcelPo.setGllx(NonConformityTypeEnum.getCodeByName(gllx));
        nonConformityObserveExcelPo.setGstx(ManagementSystemEnum.getCodeByName(gstx));
        return nonConformityObserveExcelPo;
    }

    private static InternalAuditsExcelPo getInternalAuditsExcelPo(List<List<String>> result, int i) {
        List<String> data = result.get(i);
        InternalAuditsExcelPo internalAuditsExcelPo = new InternalAuditsExcelPo();
        //类别
        String type = data.get(0);
        //审核报告名称
        String auditReportsName = data.get(1);
        //审核报告url
        String auditReports = data.get(2);
        //审核计划名称
        String auditPlanName = data.get(3);
        //审核计划名称
        String auditPlan = data.get(4);
        //目标上传日期
        String uploadTime = data.get(5);
        //所属项目部名称
        String orgName = data.get(6);
        //项目部上级组织名称
        String parentOrgName = data.get(7);

        internalAuditsExcelPo.setType(type);
        internalAuditsExcelPo.setAuditReportsName(auditReportsName);
        internalAuditsExcelPo.setAuditReports(auditReports);
        internalAuditsExcelPo.setAuditPlanName(auditPlanName);
        internalAuditsExcelPo.setAuditPlan(auditPlan);
        internalAuditsExcelPo.setUploadTime(uploadTime);
        internalAuditsExcelPo.setOrgName(orgName);
        internalAuditsExcelPo.setParentOrgName(parentOrgName);
        return internalAuditsExcelPo;
    }
}
