package com.ejianc.business.quality.service.impl;

import cn.hutool.core.util.RandomUtil;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.beust.jcommander.internal.Lists;
import com.ejianc.business.quality.dao.ReleaseUnitSettingsDao;
import com.ejianc.business.quality.dao.ReleaseUnitSettingsDetailDao;
import com.ejianc.business.quality.entity.ReleaseUnitSettingsDetailEntity;
import com.ejianc.business.quality.entity.ReleaseUnitSettingsEntity;
import com.ejianc.business.quality.model.vo.ReleaseUnitSettingsDetailVo;
import com.ejianc.business.quality.model.vo.ReleaseUnitSettingsVo;
import com.ejianc.business.quality.service.ReleaseUnitSettingServer;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.auth.session.UserContext;
import com.ejianc.framework.cache.utils.RedisTool;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.ExcelExport;
import com.ejianc.framework.skeleton.template.BaseEntity;
import com.ejianc.framework.skeleton.template.BaseVO;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springside.modules.nosql.redis.pool.JedisPool;
import redis.clients.jedis.Jedis;

import javax.servlet.http.HttpServletResponse;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * @Author: LCL
 * @Date: 2024/5/17 下午3:34
 * @Description: 下达单位设置
 */
@Service
@Slf4j
@RequiredArgsConstructor
public class ReleaseUnitSettingServerImpl implements ReleaseUnitSettingServer {
    private final ReleaseUnitSettingsDao dao;
    private final ReleaseUnitSettingsDetailDao detailDao;
    private final SessionManager sessionManager;
    private final JedisPool jedisPool;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public CommonResponse<Object> saveOrUpdate(ReleaseUnitSettingsVo saveOrUpdateVO) {
        ReleaseUnitSettingsEntity entity = BeanMapper.map(saveOrUpdateVO, ReleaseUnitSettingsEntity.class);

        UserContext userContext = sessionManager.getUserContext();
        Long id = entity.getId();
        String type = entity.getType();
        List<ReleaseUnitSettingsDetailEntity> subList = entity.getSubList();
        ReleaseUnitSettingsEntity one = dao.lambdaQuery().eq(ReleaseUnitSettingsEntity::getType, type).one();

        if (id == null) {
            boolean locked = false;
            Jedis jedis = jedisPool.getResource();
            String key = saveOrUpdateVO.getCreateUserCode() + "ReleaseUnitSettingAdd";
            if (Objects.nonNull(one)) {
                throw new BusinessException("已存在相同类型的配置");
            }
            try {
                locked = RedisTool.tryLock(jedis, key,
                        RandomUtil.randomNumbers(10), 1000);
                if (!locked) {
                    log.error("下达单位设置新增失败，重复添加，锁获取失败！");
                    releaseLock(jedis, false, key, "下达单位设置新增");
                    return CommonResponse.error("下达单位设置新增失败，重复添加!");
                }
                entity.setOrgId(userContext.getOrgId());
                entity.setOrgName(userContext.getOrgName());
                entity.setCreateUserName(userContext.getUserName());
                entity.setPreparedUserCode(userContext.getUserCode());
                subList.forEach(e -> e.setCharge(e.getCharge() + ","));
                subList.forEach(e -> e.setLeader(e.getLeader() + ","));
                dao.saveOrUpdate(entity, false);
                detailDao.lambdaUpdate().set(ReleaseUnitSettingsDetailEntity::getType, entity.getType())
                        .set(BaseEntity::getCreateUserCode, entity.getCreateUserCode())
                        .eq(ReleaseUnitSettingsDetailEntity::getSettingId, entity.getId()).update();
            } catch (Exception e) {
                throw new BusinessException("新增失败，原因：" + e.getMessage());
            } finally {
                releaseLock(jedis, locked, key, "下达单位设置新增");
            }

        } else {
            if (Objects.nonNull(one) && !one.getId().equals(id)) {
                throw new BusinessException("已存在相同类型的配置");
            }
            try {
                entity.setUpdateUserName(sessionManager.getUserContext().getUserName());
                subList.forEach(e -> {
                    e.setCharge(e.getCharge() + ",");
                    e.setLeader(e.getLeader() + ",");
                    e.setType(type);
                });
                dao.saveOrUpdate(entity, false);
            } catch (Exception e) {
                throw new BusinessException("编辑失败，原因：" + e.getMessage());
            }
        }
        return CommonResponse.success("下达单位设置-新增/编辑成功", BeanMapper.map(entity, ReleaseUnitSettingsVo.class));
    }

    @Override
    public IPage<ReleaseUnitSettingsVo> list(QueryParam param) {
        param.getFuzzyFields().addAll(Collections.emptyList());
        param.getOrderMap().put("createTime", QueryParam.DESC);

        IPage<ReleaseUnitSettingsEntity> settingsEntityPage = dao.queryPage(param, false);

        List<ReleaseUnitSettingsEntity> records = settingsEntityPage.getRecords();
        if (records.isEmpty()) {
            return new Page<>(settingsEntityPage.getCurrent(), settingsEntityPage.getSize(), settingsEntityPage.getTotal());
        }
        List<Long> ids = records.stream().map(BaseEntity::getId).collect(Collectors.toList());
        List<ReleaseUnitSettingsDetailEntity> detalList = detailDao.lambdaQuery().in(ReleaseUnitSettingsDetailEntity::getSettingId, ids)
                .list();
        if (detalList.isEmpty()) {
            return new Page<>(settingsEntityPage.getCurrent(), settingsEntityPage.getSize(), settingsEntityPage.getTotal());
        }
        Map<Long, List<ReleaseUnitSettingsDetailEntity>> idAndSubListMap = detalList
                .stream()
                .collect(Collectors.groupingBy(ReleaseUnitSettingsDetailEntity::getSettingId));
        List<ReleaseUnitSettingsVo> res = new java.util.ArrayList<>(Collections.emptyList());
        for (ReleaseUnitSettingsEntity record : records) {
            List<ReleaseUnitSettingsDetailVo> childList = Lists.newArrayList();
            ReleaseUnitSettingsVo releaseUnitSettingsVo = new ReleaseUnitSettingsVo();
            BeanUtils.copyProperties(record, releaseUnitSettingsVo);
            if (!idAndSubListMap.containsKey(record.getId())) {
                continue;
            }
            idAndSubListMap.get(releaseUnitSettingsVo.getId()).forEach(e -> childList.add(BeanMapper.map(e, ReleaseUnitSettingsDetailVo.class)));
            releaseUnitSettingsVo.setSubList(childList);
            res.add(releaseUnitSettingsVo);
        }

        return new Page<ReleaseUnitSettingsVo>(settingsEntityPage.getCurrent(), settingsEntityPage.getSize(), settingsEntityPage.getTotal())
                .setRecords(res);
    }

    @Override
    public ReleaseUnitSettingsVo detail(Long id) {
        ReleaseUnitSettingsEntity byId = dao.getById(id);
        ReleaseUnitSettingsVo releaseUnitSettingsVo = new ReleaseUnitSettingsVo();
        if (Objects.isNull(byId)) {
            return releaseUnitSettingsVo;
        }
        BeanUtils.copyProperties(byId, releaseUnitSettingsVo);
        List<ReleaseUnitSettingsDetailEntity> child = detailDao.lambdaQuery().eq(ReleaseUnitSettingsDetailEntity::getSettingId, id).list();
        List<ReleaseUnitSettingsDetailVo> childList = Lists.newArrayList();
        child.forEach(e -> childList.add(BeanMapper.map(e, ReleaseUnitSettingsDetailVo.class)));
        releaseUnitSettingsVo.setSubList(childList);

        return releaseUnitSettingsVo;
    }

    @Override
    public void del(List<ReleaseUnitSettingsVo> vos) {
        try {
            List<Long> ids = vos.stream().map(BaseVO::getId).collect(Collectors.toList());
            dao.lambdaUpdate().set(BaseEntity::getDr, 1).in(BaseEntity::getId, ids).update();
            detailDao.lambdaUpdate().set(BaseEntity::getDr, 1).eq(ReleaseUnitSettingsDetailEntity::getSettingId, ids).update();
        } catch (Exception e) {
            throw new BusinessException("下达单位设置删除失败：" + e.getMessage());
        }
    }

    @Override
    public void excelExport(List<Long> ids, HttpServletResponse response) {
        List<ReleaseUnitSettingsDetailEntity> list = detailDao.lambdaQuery().in(ReleaseUnitSettingsDetailEntity::getSettingId, ids).list();
        Map<String, Object> beans = new HashMap<>();
        beans.put("records", list);
        try {
            ExcelExport.getInstance().export("releaseUnitSettingExport.xlsx", beans, response);
        } catch (Exception e) {
            throw new BusinessException("下达单位设置导出失败：" + e.getMessage());
        }
    }

    public void releaseLock(Jedis jedis, boolean locked, String key, String operate) {
        try {
            if (locked) {
                RedisTool.releaseLock(jedis, key, operate);
            }
        } finally {
            if (null != jedis) {
                jedis.close();
            }
        }
    }
}
