package com.ejianc.business.pay.controller;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.pay.bean.PayEntity;
import com.ejianc.business.pay.service.IPayService;
import com.ejianc.business.pay.vo.PayVO;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.foundation.support.api.IBillTypeApi;
import com.ejianc.foundation.support.vo.BillCodeParam;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.auth.session.UserContext;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.ExcelExport;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.io.Serializable;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 风险金缴纳表
 *
 * @author baipengyan
 */
@RestController
@RequestMapping("pay")
public class PayController implements Serializable {
	private static final long serialVersionUID = 1L;

	private Logger logger = LoggerFactory.getLogger(this.getClass());
	private static final String BILL_CODE = "EJCBT202206000055";

	private final IBillTypeApi billTypeApi;
	private final IBillCodeApi billCodeApi;
	private final IOrgApi iOrgApi;
	private final IPayService service;
	private final SessionManager sessionManager;

	public PayController(IBillTypeApi billTypeApi, IBillCodeApi billCodeApi, IOrgApi iOrgApi, IPayService service, SessionManager sessionManager) {
		this.billTypeApi = billTypeApi;
		this.billCodeApi = billCodeApi;
		this.iOrgApi = iOrgApi;
		this.service = service;
		this.sessionManager = sessionManager;
	}


	/**
	 * 保存或修改
	 *
	 * @param saveOrUpdateVO 保存或修改数据
	 *
	 * @return 保存或修改结果
	 */
	@PostMapping(value = "/saveOrUpdate")
	public CommonResponse<PayVO> saveOrUpdate(@RequestBody PayVO saveOrUpdateVO) {
		PayEntity entity = BeanMapper.map(saveOrUpdateVO, PayEntity.class);
		if (entity.getId() == null || entity.getId() == 0) {
			BillCodeParam billCodeParam = BillCodeParam.build(BILL_CODE, InvocationInfoProxy.getTenantid(), saveOrUpdateVO);
			CommonResponse<String> billCode = billCodeApi.generateBillCode(billCodeParam);
			if (billCode.isSuccess()) {
				entity.setBillCode(billCode.getData());//此处需要根据实际修改 删除本行或者上一行
			} else {
				throw new BusinessException("网络异常， 编码生成失败， 请稍后再试");
			}
		}
		service.saveOrUpdate(entity, false);
		PayVO vo = BeanMapper.map(entity, PayVO.class);
		return CommonResponse.success("保存或修改单据成功！", vo);
	}


	/**
	 * 查询详情
	 *
	 * @param id 主键id
	 *
	 * @return 查询结果
	 */
	@GetMapping(value = "/queryDetail")
	public CommonResponse<PayVO> queryDetail(Long id) {
		PayEntity entity = service.selectById(id);
		PayVO vo = BeanMapper.map(entity, PayVO.class);
		return CommonResponse.success("查询详情数据成功！", vo);
	}


	/**
	 * 批量删除
	 *
	 * @param vos 删除数据
	 *
	 * @return 删除结果
	 */
	@PostMapping(value = "/delete")
	public CommonResponse<String> delete(@RequestBody List<PayVO> vos) {
		if (CollectionUtils.isNotEmpty(vos)) {
			for (PayVO vo : vos) {
				CommonResponse<String> resp = billTypeApi.checkQuote(BILL_CODE, vo.getId());
				if (!resp.isSuccess()) {
					return CommonResponse.error("删除失败！" + resp.getMsg());
				}
			}
		}
		service.removeByIds(vos.stream().map(PayVO::getId).collect(Collectors.toList()), true);
		return CommonResponse.success("删除成功！");
	}


	/**
	 * 分页查询
	 *
	 * @param param 请求参数
	 *
	 * @return 分页数据
	 */
	@PostMapping(value = "/queryList")
	public CommonResponse<IPage<PayVO>> queryList(@RequestBody QueryParam param) {

		List<String> fuzzyFields = param.getFuzzyFields();

		param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));

		// 数据隔离 本下 没有组织orgId的删除下面代码-------------开始
		UserContext userContextCache = sessionManager.getUserContext();
		// 当前应用有权限的根orgId，以逗号分割，可据此查询其本下数据，需判空
		String authOrgIds = userContextCache.getAuthOrgIds();
		List<OrgVO> orgVOList = null;
		if (StringUtils.isNotBlank(authOrgIds)) {
			//移动端查询
			orgVOList = (List<OrgVO>) getRespData(iOrgApi.findChildrenByParentIds(Arrays.stream(authOrgIds.split(",")).map(Long::parseLong).collect(Collectors.toList())), true, "查询失败，获取当前本下组织信息失败。");
		} else {
			//pc端查询
			orgVOList = (List<OrgVO>) getRespData(iOrgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()), true, "查询失败，获取当前本下组织信息失败。");
		}
		//普通组织 id
		List<Long> commonOrgIds = new ArrayList<>();
		//项目部 id
		List<Long> departmentIds = new ArrayList<>();
		orgVOList.stream().forEach(org -> {
			if (5 == org.getOrgType()) {
				//项目部
				departmentIds.add(org.getId());
			} else {
				//普通组织
				commonOrgIds.add(org.getId());
			}
		});
		if (CollectionUtils.isNotEmpty(commonOrgIds)) {
			// 要求主表有orgId字段，保存单据所属组织
			param.getParams().put("orgId", new Parameter(QueryParam.IN, commonOrgIds));
		} else if (CollectionUtils.isNotEmpty(departmentIds)) {
			// 要求主表有projectDepartmentId字段，保存单据所属项目部
			param.getParams().put("projectDepartmentId", new Parameter(QueryParam.IN, departmentIds));
		}
		// 数据隔离 本下 没有组织orgId的删除上面代码-------------结束！！！

		IPage<PayEntity> page = service.queryPage(param, false);
		IPage<PayVO> pageData = new Page<>(page.getCurrent(), page.getSize(), page.getTotal());
		pageData.setRecords(BeanMapper.mapList(page.getRecords(), PayVO.class));

		return CommonResponse.success("查询列表数据成功！", pageData);
	}


	/**
	 * 获取RPC数据
	 * resp 返回值
	 * isMustSuc 是否必须成功
	 * errMsg 失败提示
	 */
	private Object getRespData(CommonResponse<?> resp, boolean isMustSuc, String errMsg) {
		if (isMustSuc && !resp.isSuccess()) {
			throw new BusinessException(StringUtils.isNoneBlank(errMsg) ? errMsg : "调用Rpc服务失败");
		}
		return resp.getData();
	}


	/**
	 * 导出excel
	 *
	 * @param param    请求参数
	 * @param response 响应
	 */
	@PostMapping(value = "/excelExport")
	public void excelExport(@RequestBody QueryParam param, HttpServletResponse response) {
		List<String> fuzzyFields = param.getFuzzyFields();

		param.getParams().put("tenant_id", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
		param.setPageIndex(1);
		param.setPageSize(-1);

		// 数据隔离 本下 没有组织orgId的删除下面代码
		param.getParams().put("orgId", new Parameter(QueryParam.IN, iOrgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
		List<PayEntity> list = service.queryList(param);
		// todo:字段翻译等等
		Map<String, Object> beans = new HashMap<>();
		beans.put("records", list);
		ExcelExport.getInstance().export("Pay-export.xlsx", beans, response);
	}
}
