package com.ejianc.business.repay.service.impl;

import com.ejianc.business.pay.enums.ConfirmStatusEnum;
import com.ejianc.business.repay.bean.RepayEntity;
import com.ejianc.business.repay.service.IRepayService;
import com.ejianc.foundation.support.api.IBillTypeApi;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.auth.session.UserContext;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.skeleton.billState.service.ICommonBusinessService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import java.util.Date;

@Service("repay")
public class RepayBpmServiceImpl implements ICommonBusinessService {

	private final IRepayService service;
	private final SessionManager sessionManager;
	private final IBillTypeApi billTypeApi;
	private final Logger logger = LoggerFactory.getLogger(this.getClass());

	public RepayBpmServiceImpl(IRepayService service, SessionManager sessionManager, IBillTypeApi billTypeApi) {
		this.service = service;
		this.sessionManager = sessionManager;
		this.billTypeApi = billTypeApi;
	}

	/**
	 * 终审审核完回调
	 *
	 * @param
	 *
	 * @return
	 */
	@Override
	public CommonResponse<String> afterApprovalProcessor(Long billId, Integer state, String billTypeCode) {
		logger.info("终审审核完回调--start，billId={},state={},billTypeCode={}", billId, state, billTypeCode);

		// 审批通过/已提交---财务确认前，确认中
		RepayEntity entity = service.selectById(billId);
		if (entity == null) {
			throw new BusinessException("查询不到单据信息");
		}

		UserContext userContext = sessionManager.getUserContext();

		// 区分提交和审批
		if (state.equals(BillStateEnum.COMMITED_STATE.getBillStateCode())) {
			// 直审更新提交相关字段
			entity.setCommitDate(new Date());
			entity.setCommitUserCode(userContext.getUserCode());
			entity.setCommitUserName(userContext.getUserName());
		}
		entity.setBillStateName(BillStateEnum.getEnumByStateCode(state).getDescription());

		// 更新风险金退还确认状态--确认中
		entity.setConfirmStatus(ConfirmStatusEnum.CONFIRMING.getCode());
		entity.setConfirmStatusName(ConfirmStatusEnum.CONFIRMING.getName());
		entity.setConfirmDate(new Date());
		entity.setConfirmUserId(userContext.getUserId());
		entity.setConfirmUserName(userContext.getUserName());
		entity.setConfirmUserCode(userContext.getUserCode());

		// 生效时间
		entity.setEffectiveDate(new Date());

		// 执行更新
		service.saveOrUpdate(entity, false);

		logger.info("终审审核完回调--end");
		return CommonResponse.success("终审审核完回调成功");
	}

	/**
	 * 有审批流的撤回前回调
	 *
	 * @param billId
	 * @param state
	 *
	 * @return
	 */
	@Override
	public CommonResponse<String> beforeHasBpmBack(Long billId, Integer state, String billTypeCode) {
		return bpmBackCheck(billId, state, billTypeCode);
	}

	/**
	 * 弃审前事件回调
	 *
	 * @param billId
	 * @param state
	 *
	 * @return
	 */
	@Override
	public CommonResponse<String> beforeAbstainingProcessor(Long billId, Integer state, String billTypeCode) {
		return bpmBackCheck(billId, state, billTypeCode);
	}

	private CommonResponse<String> bpmBackCheck(Long billId, Integer state, String billTypeCode) {
		/*
		单据撤回逻辑：
        撤回逻辑：1、当存在审批流程时且第一个人没有审批时，支持撤回；
                2、当不存在审批流程、没有下游业务，且为最新一期得缴纳/退还单据时支持撤回；
        弃审逻辑：1、审批过程中，支持后台弃审；
                2、当审批通过，没有下游业务，且为最新一期得缴纳/退还单据时支持后台弃审；
        下游业务：已确认
		*/

		RepayEntity entity = service.selectById(billId);
		if (entity == null) {
			throw new BusinessException("查询不到单据信息");
		}

		// 是否被其他单据引用
		CommonResponse<String> res = billTypeApi.checkQuote(billTypeCode, billId);
		logger.info("平台返回查询被引用情况：{}，----{}", res.isSuccess(), res.getMsg());
		if (!res.isSuccess()) {
			return CommonResponse.error("当前单据已被下游业务引用，不能弃审！");
		}

		//  是否是最新单据
		if (!service.isLatestBill(billId, entity.getUserId(), entity.getOrgId())) {
			return CommonResponse.error("当前单据不是最新单据，不能弃审！");
		}

		// 下游业务：风险金缴纳确认，已经确认的不能撤回弃审
		if (entity.getConfirmStatus().equals(ConfirmStatusEnum.CONFIRM.getCode())) {
			return CommonResponse.error("当前单据已经确认，不能弃审！");
		}

		return CommonResponse.success("单据撤回成功");
	}
}
