package com.ejianc.business.repay.controller;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.repay.bean.RepayEntity;
import com.ejianc.business.repay.enums.ConfirmStatusEnum;
import com.ejianc.business.repay.enums.RepayTypeEnum;
import com.ejianc.business.repay.service.IRepayService;
import com.ejianc.business.repay.vo.RepayVO;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.foundation.support.api.IBillTypeApi;
import com.ejianc.foundation.support.vo.BillCodeParam;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.auth.session.UserContext;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.ExcelExport;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.io.Serializable;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 风险金退还表
 *
 * @author baipengyan
 */
@RestController
@RequestMapping("repay")
public class RepayController implements Serializable {
	private static final long serialVersionUID = 1L;

	private Logger logger = LoggerFactory.getLogger(this.getClass());
	private static final String RULE_CODE = "RISK_REPAY";
	private static final String BILL_CODE = "EJCBT202206000056";

	private final IBillTypeApi billTypeApi;
	private final IBillCodeApi billCodeApi;
	private final IOrgApi iOrgApi;
	private final IRepayService service;
	private final SessionManager sessionManager;

	public RepayController(IBillTypeApi billTypeApi, IBillCodeApi billCodeApi, IOrgApi iOrgApi, IRepayService service, SessionManager sessionManager) {
		this.billTypeApi = billTypeApi;
		this.billCodeApi = billCodeApi;
		this.iOrgApi = iOrgApi;
		this.service = service;
		this.sessionManager = sessionManager;
	}


	/**
	 * 保存或修改
	 *
	 * @param saveOrUpdateVO 保存或修改数据
	 *
	 * @return 保存或修改结果
	 */
	@PostMapping(value = "/saveOrUpdate")
	public CommonResponse<RepayVO> saveOrUpdate(@RequestBody RepayVO saveOrUpdateVO) {
		RepayEntity entity = BeanMapper.map(saveOrUpdateVO, RepayEntity.class);
		if (entity.getId() == null || entity.getId() == 0) {
			BillCodeParam billCodeParam = BillCodeParam.build(RULE_CODE, InvocationInfoProxy.getTenantid(), saveOrUpdateVO);
			CommonResponse<String> billCode = billCodeApi.generateBillCode(billCodeParam);
			if (billCode.isSuccess()) {
				entity.setBillCode(billCode.getData());//此处需要根据实际修改 删除本行或者上一行
			} else {
				throw new BusinessException("网络异常， 编码生成失败， 请稍后再试");
			}
		}

		// 单据名称、单据类型
		entity.setBillName("风险金退还");
		entity.setBillType(BILL_CODE);
		entity.setBillStateName(BillStateEnum.UNCOMMITED_STATE.getDescription());

		// 默认值: 类型：缴纳，确认状态：未确认
		entity.setTypeName(RepayTypeEnum.getEnumByCode(entity.getType()).getName());
		entity.setConfirmStatus(ConfirmStatusEnum.UNCONFIRM.getCode());
		entity.setConfirmStatusName(ConfirmStatusEnum.UNCONFIRM.getName());

		service.saveOrUpdate(entity, false);
		RepayVO vo = BeanMapper.map(entity, RepayVO.class);
		return CommonResponse.success("保存或修改单据成功！", vo);
	}


	/**
	 * 查询详情
	 *
	 * @param id 主键id
	 *
	 * @return 查询结果
	 */
	@GetMapping(value = "/queryDetail")
	public CommonResponse<RepayVO> queryDetail(Long id) {
		RepayEntity entity = service.selectById(id);
		RepayVO vo = BeanMapper.map(entity, RepayVO.class);
		return CommonResponse.success("查询详情数据成功！", vo);
	}


	/**
	 * 批量删除
	 *
	 * @param vos 删除数据
	 *
	 * @return 删除结果
	 */
	@PostMapping(value = "/delete")
	public CommonResponse<String> delete(@RequestBody List<RepayVO> vos) {
		if (CollectionUtils.isNotEmpty(vos)) {
			for (RepayVO vo : vos) {
				CommonResponse<String> resp = billTypeApi.checkQuote(BILL_CODE, vo.getId());
				if (!resp.isSuccess()) {
					return CommonResponse.error("删除失败！" + resp.getMsg());
				}
			}
		}
		service.removeByIds(vos.stream().map(RepayVO::getId).collect(Collectors.toList()), true);
		return CommonResponse.success("删除成功！");
	}


	/**
	 * 分页查询
	 *
	 * @param param 请求参数
	 *
	 * @return 分页数据
	 */
	@PostMapping(value = "/queryList")
	public CommonResponse<IPage<RepayVO>> queryList(@RequestBody QueryParam param) {

		// 模糊查询
		List<String> fuzzyFields = param.getFuzzyFields();
		fuzzyFields.add("orgName");
		fuzzyFields.add("employeeName");
		fuzzyFields.add("billCode");

		if (param.getParams().containsKey("confirmStatus")) {
			// 查询本下组织缴纳申请列表
			UserContext userContextCache = sessionManager.getUserContext();
			String authOrgIds = userContextCache.getAuthOrgIds();
			List<OrgVO> orgVOList = null;
			if (StringUtils.isNotBlank(authOrgIds)) {
				//移动端查询
				orgVOList = (List<OrgVO>) getRespData(iOrgApi.findChildrenByParentIds(Arrays.stream(authOrgIds.split(",")).map(Long::parseLong).collect(Collectors.toList())), true, "查询失败，获取当前本下组织信息失败。");
			} else {
				//pc端查询
				orgVOList = (List<OrgVO>) getRespData(iOrgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()), true, "查询失败，获取当前本下组织信息失败。");
			}
			//普通组织 id
			List<Long> commonOrgIds = new ArrayList<>();
			//项目部 id
			List<Long> departmentIds = new ArrayList<>();
			orgVOList.forEach(org -> {
				if (5 == org.getOrgType()) {
					//项目部
					departmentIds.add(org.getId());
				} else {
					//普通组织
					commonOrgIds.add(org.getId());
				}
			});
			if (CollectionUtils.isNotEmpty(commonOrgIds)) {
				param.getParams().put("orgId", new Parameter(QueryParam.IN, commonOrgIds));
			} else if (CollectionUtils.isNotEmpty(departmentIds)) {
				param.getParams().put("projectDepartmentId", new Parameter(QueryParam.IN, departmentIds));
			}

			// 生效的风险金缴纳单据
			param.getParams().put("bill_state", new Parameter(QueryParam.IN, Arrays.asList(BillStateEnum.COMMITED_STATE.getBillStateCode(), BillStateEnum.PASSED_STATE.getBillStateCode())));
		} else {
			// 仅可以看到登陆人本人的退还信息
			UserContext userContextCache = sessionManager.getUserContext();
			param.getParams().put("userId", new Parameter(QueryParam.EQ, userContextCache.getUserId()));
		}

		IPage<RepayEntity> page = service.queryPage(param, false);
		IPage<RepayVO> pageData = new Page<>(page.getCurrent(), page.getSize(), page.getTotal());
		pageData.setRecords(BeanMapper.mapList(page.getRecords(), RepayVO.class));

		return CommonResponse.success("查询列表数据成功！", pageData);
	}


	/**
	 * 获取RPC数据
	 * resp 返回值
	 * isMustSuc 是否必须成功
	 * errMsg 失败提示
	 */
	private Object getRespData(CommonResponse<?> resp, boolean isMustSuc, String errMsg) {
		if (isMustSuc && !resp.isSuccess()) {
			throw new BusinessException(StringUtils.isNoneBlank(errMsg) ? errMsg : "调用Rpc服务失败");
		}
		return resp.getData();
	}


	/**
	 * 导出excel
	 *
	 * @param param    请求参数
	 * @param response 响应
	 */
	@PostMapping(value = "/excelExport")
	public void excelExport(@RequestBody QueryParam param, HttpServletResponse response) {
		// 模糊查询
		List<String> fuzzyFields = param.getFuzzyFields();
		fuzzyFields.add("orgName");
		fuzzyFields.add("employeeName");
		fuzzyFields.add("billCode");

		if (param.getParams().containsKey("confirmStatus")) {
			// 查询本下组织缴纳申请列表
			UserContext userContextCache = sessionManager.getUserContext();
			String authOrgIds = userContextCache.getAuthOrgIds();
			List<OrgVO> orgVOList = null;
			if (StringUtils.isNotBlank(authOrgIds)) {
				//移动端查询
				orgVOList = (List<OrgVO>) getRespData(iOrgApi.findChildrenByParentIds(Arrays.stream(authOrgIds.split(",")).map(Long::parseLong).collect(Collectors.toList())), true, "查询失败，获取当前本下组织信息失败。");
			} else {
				//pc端查询
				orgVOList = (List<OrgVO>) getRespData(iOrgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()), true, "查询失败，获取当前本下组织信息失败。");
			}
			//普通组织 id
			List<Long> commonOrgIds = new ArrayList<>();
			//项目部 id
			List<Long> departmentIds = new ArrayList<>();
			orgVOList.forEach(org -> {
				if (5 == org.getOrgType()) {
					//项目部
					departmentIds.add(org.getId());
				} else {
					//普通组织
					commonOrgIds.add(org.getId());
				}
			});
			if (CollectionUtils.isNotEmpty(commonOrgIds)) {
				param.getParams().put("orgId", new Parameter(QueryParam.IN, commonOrgIds));
			} else if (CollectionUtils.isNotEmpty(departmentIds)) {
				param.getParams().put("projectDepartmentId", new Parameter(QueryParam.IN, departmentIds));
			}

			// 生效的风险金缴纳单据
			param.getParams().put("bill_state", new Parameter(QueryParam.IN, Arrays.asList(BillStateEnum.COMMITED_STATE.getBillStateCode(), BillStateEnum.PASSED_STATE.getBillStateCode())));
		} else {
			// 仅可以看到登陆人本人的退还信息
			UserContext userContextCache = sessionManager.getUserContext();
			param.getParams().put("userId", new Parameter(QueryParam.EQ, userContextCache.getUserId()));
		}

		param.setPageIndex(1);
		param.setPageSize(-1);

		IPage<RepayEntity> page = service.queryPage(param, false);
		ArrayList<RepayVO> repays = new ArrayList<>();
		// 字段翻译
		page.getRecords().forEach(e -> {
			RepayVO repayVO = BeanMapper.map(e, RepayVO.class);
			repayVO.setConfirmStatusName(ConfirmStatusEnum.getEnumByCode(repayVO.getConfirmStatus()).getName());
			repayVO.setTypeName(RepayTypeEnum.getEnumByCode(repayVO.getType()).getName());
			repayVO.setBillStateName(BillStateEnum.getEnumByStateCode(repayVO.getBillState()).getDescription());
			repays.add(repayVO);
		});

		Map<String, Object> beans = new HashMap<>();
		beans.put("records", repays);

		// 区分缴纳、缴纳确认导出模版
		if (param.getParams().containsKey("confirmStatus")) {
			// 缴纳确认区分待确认(确认中)、已确认导出模版
			Integer confirmStatus = (Integer) param.getParams().get("confirmStatus").getValue();
			if (Objects.equals(confirmStatus, ConfirmStatusEnum.CONFIRMING.getCode())) {
				// 待确认(确认中)
				ExcelExport.getInstance().export("repay-unconfirm-export.xlsx", beans, response);
			}
			if (Objects.equals(confirmStatus, ConfirmStatusEnum.CONFIRM.getCode())) {
				// 已确认
				ExcelExport.getInstance().export("repay-confirm-export.xlsx", beans, response);
			}
		} else {
			ExcelExport.getInstance().export("repay-export.xlsx", beans, response);
		}
	}

	/**
	 * 风险金退还：批量确认
	 *
	 * @param repays 要确认的单据信息
	 *
	 * @return 确认结果
	 */
	@PostMapping(value = "/batchConfirm")
	public CommonResponse<String> batchConfirm(@RequestBody List<RepayVO> repays) {
		service.batchConfirm(repays);
		return CommonResponse.success("确认成功！");
	}


	/**
	 * 校验：一个员工+员工明细只可以有一个未生效的单据（退还+调动+缴纳）
	 *
	 * @param userId 员工id
	 * @param orgId  组织id
	 * @param info   缴纳/退还
	 */
	@GetMapping(value = "/checkBeforeSelectRefer")
	public CommonResponse<String> checkBeforeSelectRefer(@RequestParam("sourceBillId") Long sourceBillId,
	                                                     @RequestParam("info") String info) {
		return service.checkBeforeSave(sourceBillId, info);
	}
}
