package com.ejianc.business.rmat.api;


import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.ejianc.business.finance.pub.vo.PubContractSettleVO;
import com.ejianc.business.finance.util.MathUtil;
import com.ejianc.business.rmat.bean.RentContractEntity;
import com.ejianc.business.rmat.bean.RentSettlementEntity;
import com.ejianc.business.rmat.service.IRentContractService;
import com.ejianc.business.rmat.service.IRentSettlementService;
import com.ejianc.business.rmat.vo.RmatSettlementVO;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.response.*;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;

@RestController
@RequestMapping("/api/rmat/")
public class RmatContractApi {

    private Logger logger = LoggerFactory.getLogger(this.getClass());


    @Autowired
    private IRentSettlementService rentSettleService;

    @Autowired
    private IRentContractService rentContractService;


    @PostMapping(value = "getRmatSettle")
    public CommonResponse<JSONObject> getRmatSettle(@RequestBody QueryParam param) {
        try {
            /** 模糊搜索配置字段示例 */
            List<String> fuzzyFields = param.getFuzzyFields();
            fuzzyFields.add("billCode");
            fuzzyFields.add("orgName");
            fuzzyFields.add("employeeName");
            fuzzyFields.add("customerName");
            fuzzyFields.add("supplierName");
            param.getParams().put("tenantId", new Parameter("eq", InvocationInfoProxy.getTenantid()));
            param.getOrderMap().put("createTime", QueryParam.DESC);
            //已生效状态的单据
            param.getComplexParams().add(ComplexParam.getApprovedComplexParam(ComplexParam.AND));

            IPage<RentSettlementEntity> pageData = rentSettleService.queryPage(param, false);
            List<RentSettlementEntity> entityList = pageData.getRecords();
            List<PubContractSettleVO> voList = new ArrayList<>();
            for (RentSettlementEntity entity : entityList) {
                PubContractSettleVO vo = new PubContractSettleVO();
                vo.setId(entity.getId());
                vo.setBillCode(entity.getBillCode());
                vo.setSettleMny(MathUtil.safeSub(entity.getSettlementTaxMny(), entity.getOffsetMny()));
                vo.setCreateUserCode(entity.getCreateUserCode());
                vo.setCreateTime(entity.getCreateTime());
                voList.add(vo);
            }
            JSONObject page = new JSONObject();
            page.put("records", voList);
            page.put("total", pageData.getTotal());
            page.put("current", pageData.getCurrent());
            page.put("size", pageData.getSize());
            page.put("pages", pageData.getPages());
            return CommonResponse.success("查询列表数据成功！", page);
        } catch (Exception e) {
            logger.error("系统异常：" + e.getMessage());
            e.printStackTrace();
        }
        return CommonResponse.error("查询失败");
    }

    /**
     * @Author yqls
     * @Date 2020/7/4
     * @Description 回写周转材租赁结算单累计付款金额
     * @Param settleId  结算单id
     * @Param payMny  本期已付款金额，正为回写，负为逆回写
     * @Return com.ejianc.framework.core.response.CommonResponse<java.lang.String>
     */
    @GetMapping(value = "updateRentSettleSumPayMny")
    public CommonResponse<String> updateRentSettleSumPayMny(@RequestParam(value = "settleId", required = true) Long settleId,
                                                            @RequestParam("payMny") BigDecimal payMny) {
        RentSettlementEntity entity = rentSettleService.selectById(settleId);
        if (entity == null) {
            return CommonResponse.error("没有找到相应周转材租赁结算单！");
        }
        //累计付款金额
        BigDecimal sumPayMny = entity.getSumPayMny();
        sumPayMny = MathUtil.safeAdd(sumPayMny, payMny);
        entity.setSumPayMny(sumPayMny);
        rentSettleService.saveOrUpdate(entity, false);
        return CommonResponse.success("回写周转材租赁结算单累计付款金额成功！");
    }


    /**
     * @Author sunyj
     * @Date 2020/7/4
     * @Description 根据条件查询结算单累计冲抵金额
     * @Param contractId  合同id
     */
    @GetMapping(value = "getRentSettlementById")
    public CommonResponse<RmatSettlementVO> getRentSettlementById(@RequestParam(value = "contractId", required = true) Long contractId, @RequestParam(value = "orgId", required = true) Long orgId) {
        RmatSettlementVO vo = new RmatSettlementVO();
        vo.setSumOffsetMny(new BigDecimal("0.00").setScale(2, BigDecimal.ROUND_HALF_UP));
        LambdaQueryWrapper<RentSettlementEntity> lambdachange = Wrappers.<RentSettlementEntity>lambdaQuery();
        lambdachange.eq(RentSettlementEntity::getTenantId, InvocationInfoProxy.getTenantid());
        lambdachange.eq(RentSettlementEntity::getContractId, contractId);
        lambdachange.eq(RentSettlementEntity::getOrgId, orgId);
        lambdachange.in(RentSettlementEntity::getBillState, BillStateEnum.PASSED_STATE.getBillStateCode(), BillStateEnum.COMMITED_STATE.getBillStateCode());
        List<RentSettlementEntity> list = rentSettleService.list(lambdachange);
        if (CollectionUtils.isNotEmpty(list)) {
            final BigDecimal[] sumOffsetMny = {new BigDecimal("0.00").setScale(2, BigDecimal.ROUND_HALF_UP)};
            list.forEach(e -> {
                sumOffsetMny[0] = sumOffsetMny[0].add(e.getOffsetMny() == null ? BigDecimal.ZERO : e.getOffsetMny());
            });
            //累计冲抵金额
            vo.setSumOffsetMny(sumOffsetMny[0]);
        }
        return CommonResponse.success("获取信息成功", vo);
    }


    /**
     * @Author yqls
     * @Date 2020/7/4
     * @Description 回写设备租赁结算单累计申请金额、剩余可申请金额
     * @Param settleId  结算单id
     * @Param payMny  本期已付款金额，正为回写，负为逆回写
     * @Return com.ejianc.framework.core.response.CommonResponse<java.lang.String>
     */
    @GetMapping(value = "updateRentSettleSumApplyMny")
    public CommonResponse<String> updateRentSettleSumApplyMny(@RequestParam(value = "settleId", required = true) Long settleId,
                                                              @RequestParam("applyMny") BigDecimal applyMny) {
        RentSettlementEntity entity = rentSettleService.selectById(settleId);
        if (entity == null) {
            return CommonResponse.error("没有找到相应周转材租赁结算单！");
        }
        //累计申请金额
        BigDecimal sumApplyMny = entity.getSumApplyMny();
        sumApplyMny = MathUtil.safeAdd(sumApplyMny, applyMny);
        entity.setSumApplyMny(sumApplyMny);
        //剩余可申请金额
        BigDecimal surplusApplyMny = MathUtil.safeSub(MathUtil.safeSub(entity.getSettlementTaxMny(), entity.getOffsetMny()), applyMny);
        entity.setSurplusApplyMny(surplusApplyMny);
        rentSettleService.saveOrUpdate(entity, false);
        return CommonResponse.success("回写周转材租赁结算单累计申请金额、剩余可申请金额成功！");
    }


    /**
     * @Author yqls
     * @Date 2020/7/4
     * @Description 回写周转材租赁合同累计付款金额、累计预付款金额
     * @Param contractId  合同id
     * @Param payMny  本期已付款金额，正为回写，负为逆回写
     * @param prepayMny 本期预付款金额，正为回写，负为逆回写
     * @Return com.ejianc.framework.core.response.CommonResponse<java.lang.String>
     */
    @GetMapping(value = "updateRentContractSumPayMny")
    public CommonResponse<String> updateRentContractSumPayMny(@RequestParam(value = "contractId", required = true) Long contractId,
                                                              @RequestParam("payMny") BigDecimal payMny, @RequestParam("prepayMny") BigDecimal prepayMny) {
        RentContractEntity entity = rentContractService.selectById(contractId);
        if (entity == null) {
            return CommonResponse.error("没有找到相应周转材租赁合同！");
        }
        //累计付款金额
        BigDecimal sumPayMny = entity.getSumPayMny();
        sumPayMny = MathUtil.safeAdd(sumPayMny, payMny);
        entity.setSumPayMny(sumPayMny);
        //累计预付款金额
        BigDecimal sumPrepayMny = entity.getSumPrepayMny();
        sumPrepayMny = MathUtil.safeAdd(sumPrepayMny, prepayMny);
        entity.setSumPrepayMny(sumPrepayMny);
        rentContractService.saveOrUpdate(entity, false);
        return CommonResponse.success("回写周转材租赁合同累计付款金额、累计预付款金额成功！");
    }

}


