package com.ejianc.integration.sdbjmaterial.controller;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.framework.cache.redis.CacheManager;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.HttpTookit;
import com.ejianc.integration.sdbjmaterial.bean.OutstoreEntity;
import com.ejianc.integration.sdbjmaterial.service.IOutstoreService;
import com.ejianc.integration.sdbjmaterial.util.XmlUtils;
import com.ejianc.integration.sdbjmaterial.vo.OutstoreVO;
import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.*;

import java.util.*;
import java.util.stream.Collectors;

/**
　* @Description: 物资出库
　* @author CJ
　* @date 2020/10/10 9:25
　*/
@RestController
@RequestMapping(value = "/outStore/")
public class OutstoreController {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    private final String SDBJ_CACHE_KEY_PREFIX = "SDBJ::";

    @Autowired
    private CacheManager cacheManager;

    @Value("${sdbj.serverUrl}")
    private String SDBJ_SERVER_URL;

    @Value("${sdbj.appid}")
    private String APPID;

    @Value("${sdbj.accessTokenUrl}")
    private String ACCESS_TOKEN_URL;

    @Value("${sdbj.materialWareHouseListUrl}")
    private String MATERIAL_WARE_HOUSE_LIST_URL;

    @Value("${sdbj.contactUnitListUrl}")
    private String CONTACT_UNIT_LIST_URL;

    @Value("${sdbj.materialContactListUrl}")
    private String MATERIAL_CONTACT_LIST_URL;

    @Value("${sdbj.contractListUrl}")
    private String CONTRACT_LIST_URL;

    @Value("${sdbj.employeeListUrl}")
    private String EMPLOYEE_LIST_URL;

    @Value("${sdbj.subContractListUrl}")
    private String SUB_CONTRACT_LIST_URL;

    @Value("${sdbj.securityCostsList}")
    private String SECURITY_COSTS_LIST_URL;

    @Value("${sdbj.greenDreamProjectList}")
    private String GREEN_DREAM_PROJECT_LIST_URL;

    @Value("${sdbj.inventoryList}")
    private String INVENTORY_LIST_URL;

    @Value("${sdbj.processTemplateList}")
    private String PROCESS_TEMPLATE_LIST_URL;

    @Value("${sdbj.flowEmployeeList}")
    private String FLOW_EMPLOYEE_LIST_URL;

    @Autowired
    private IOutstoreService service;

    /**
     * @Description saveOrUpdate 新增或者修改
     */
    @RequestMapping(value = "/saveOrUpdate", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<OutstoreVO> saveOrUpdate(@RequestBody OutstoreVO saveorUpdateVO) {
        OutstoreEntity entity = BeanMapper.map(saveorUpdateVO, OutstoreEntity.class);
        service.saveOrUpdate(entity, false);
        OutstoreVO vo = BeanMapper.map(entity, OutstoreVO.class);
        return CommonResponse.success("保存或修改单据成功！",vo);
    }

    /**
     * @Description queryDetail 查询详情
     * @param id
     */
    @RequestMapping(value = "/queryDetail", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<OutstoreVO> queryDetail(Long id) {
        OutstoreEntity entity = service.selectById(id);
        OutstoreVO vo = BeanMapper.map(entity, OutstoreVO.class);
        return CommonResponse.success("查询详情数据成功！",vo);
    }

    /**
     * @Description delete 批量删除单据
     * @Param [ids]
     */
    @RequestMapping(value = "/delete", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<String> delete(@RequestBody List<OutstoreVO> vos) {
        service.removeByIds(vos.stream().map(OutstoreVO::getId).collect(Collectors.toList()),true);
        return CommonResponse.success("删除成功！");
    }

    /**
     * @Description queryList 查询列表
     * @param param
     * @Return com.ejianc.framework.core.response.CommonResponse<java.lang.String>
     */
    @RequestMapping(value = "/queryList", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<IPage<OutstoreVO>> queryList(@RequestBody QueryParam param) {
        /** 模糊搜索配置字段示例 */
        List<String> fuzzyFields = param.getFuzzyFields();
        /** 租户隔离 */
        param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));

        IPage<OutstoreEntity> page = service.queryPage(param,false);
        IPage<OutstoreVO> pageData = new Page<>(page.getCurrent(), page.getSize(), page.getTotal());
        pageData.setRecords(BeanMapper.mapList(page.getRecords(), OutstoreVO.class));

        return CommonResponse.success("查询列表数据成功！",pageData);
    }

    /**
     * 获取水电八局服务访问Token
     *
     * @return success: Boolean,服务响应结果编码；
     *         message: String, 服务响应结果信息；
     *         data: {access_token：String，平台服务访问密钥；invalidate: DateTime，过期时间}
     */
    public JSONObject getSDBJAccessToken() {
        JSONObject resp = new JSONObject();
        String key = SDBJ_CACHE_KEY_PREFIX + "ACCESS_TOKEN";
        String reqUrl = SDBJ_SERVER_URL + ACCESS_TOKEN_URL;

        String tokenInfo = cacheManager.get(key);
        if(StringUtils.isNotBlank(tokenInfo)) {
            JSONObject tokenData = JSONObject.parseObject(tokenInfo);
            Long invalidateTime = Long.valueOf(tokenData.get("invalidateTime").toString());
            Long curTime = System.currentTimeMillis();
            if(invalidateTime > curTime) {
                return tokenData;
            }
        }

        Map<String, Object> params = new HashMap<>(2);
        params.put("appId", APPID);
        try {
            String reqRespStr = HttpTookit.getAndHeader(reqUrl, params);
            logger.info("获取水电八局服务访问Token服务-[地址：{}, 参数：{}]，返回结果：{}", reqUrl, JSONObject.toJSONString(params), reqRespStr);

            JSONObject reqResp = JSONObject.parseObject(reqRespStr);
            if(BooleanUtils.isNotTrue((Boolean) reqResp.get("success"))) {
                logger.error(null != reqResp.get("message") ? reqResp.get("message").toString() : "获取水电八局服务请求Token失败！");
                return null;
            }

            Map<String, Object> tokenData = (Map<String, Object>) reqResp.get("data");
            JSONObject invalidateInfo = (JSONObject) tokenData.get("invalidate");
            Long invalidateTime = Long.valueOf(invalidateInfo.get("time").toString());

            resp.put("token", tokenData.get("access_token"));
            resp.put("invalidateTime", invalidateTime);

            //放入缓存，有效时间1小时
            cacheManager.setex(key, JSONObject.toJSONString(resp), 60 * 50);

            return resp;
        } catch (Exception e) {
            logger.error("请求水电八局服务访问Token异常, ", e);
            return null;
        }
    }

    /**
     * @Author mrsir_wxp
     * @Date 2020/10/12 根据项目查询仓库列表参照
     * @Description refPurchaseData
     * @Param [pageNumber, pageSize, condition, searchText]
     * @Return com.ejianc.framework.core.response.CommonResponse<com.baomidou.mybatisplus.core.metadata.IPage<java.util.Map>>
     */
    @RequestMapping(value = "/refWarehouseListData", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<IPage<JSONObject>> refWarehouseListData(@RequestParam Integer pageNumber,
                                                                          @RequestParam Integer pageSize,
                                                                          String condition,
                                                                          String searchText) {
        String projectId = null;
        if(StringUtils.isNotEmpty(condition)){
            JSONObject con = JSON.parseObject(condition);
            if(StringUtils.isNotEmpty(con.getString("projectId"))){
                projectId = con.getString("projectId");
            }
        }
        if(StringUtils.isEmpty(projectId)){
            return CommonResponse.error("查询水电八局仓库参照失败，项目id不能为空！");
        }
        CommonResponse<JSONObject> response = getMaterialWarehouseList(projectId);
        if(response.isSuccess()){
            JSONObject object = response.getData();
            JSONArray page = object.getJSONArray("list");
            IPage<JSONObject> pageData = new Page<>(1, page.size()==0?10:page.size(), page.size());
            List<JSONObject> records = new ArrayList<>();
            for (int i = 0; i < page.size(); i++) {
                records.add(page.getJSONObject(i));
            }
            pageData.setRecords(records);
            return CommonResponse.success(pageData);
        }else {
            return CommonResponse.error(response.getMsg());
        }
    }
    /**
     * 根据项目Id查询对应的物资仓库列表
     *
     * @param projectId
     * @return success: Boolean,服务响应结果编码；
     *         message: String, 服务响应结果信息；
     *         data: {list [{id: String,仓库Id；projectid: String,项目Id；parentid：String,父Id；code：仓库编号；name：仓库名称；type：仓库类别}, ...]}
     */
    @GetMapping(value = "materialWarehouseList")
    public CommonResponse<JSONObject> getMaterialWarehouseList(@RequestParam(value = "projectId") String projectId) {
        try {
            Map<String, Object> params = new HashMap<>(2);
            params.put("prjid", projectId);
            CommonResponse<JSONObject> response = sendGetReq(MATERIAL_WARE_HOUSE_LIST_URL, params);
            if(response.isSuccess()){
                JSONObject back = new JSONObject();
                back.put("list",response.getData().getJSONObject("data").get("list"));
                return CommonResponse.success(back);
            }
            return response;
        } catch (Exception e) {
            logger.error("查询物资仓库列表异常, ", e);

            return CommonResponse.error("查询物资仓库列表失败！");
        }
    }

    /**
     * 分页查询往来单位信息列表
     *
     * @param projectId 项目主键Id
     * @param queryParam 查询参数，可根据单位编号和名称关键字查询
     * @param type 类别（1-5） 1：分包商，2：服务外部商，3：项目组织，4：材料供应商，5：设备供应商
     * @param pageIndex 页码
     * @param pageSize 每页记录数
     * @return success: Boolean,服务响应结果编码；
     *         message: String, 服务响应结果信息；
     *         参数type为1、2时，data: {list [{id: String,单位Id；name: String,单位名称；code：String,单位编号；legalperson：法人代表；aptitudegrade：企业资质}, ...]}
     *         参数type为3时，data: {list [{id: String,组织Id；parentid：String,父Id；code：String,组织编号；name：String,组织名称}, ...]}
     *         参数type为4时，data: {list [{id: String,单位Id；code：String,供应商编号；name：String,材料供应商名称}, ...]}
     *         参数type为5时，data: {list [{id: String,id；code：String,供应商编号；name：String,供应商名称}, ...]}
     */
    @GetMapping(value = "contactUnitList")
    public CommonResponse<JSONObject> getContactUnitList(@RequestParam(value = "projectId") String projectId,
                                                          @RequestParam(value = "searchText", required = false) String searchText,
                                                          @RequestParam(value = "type") String type,
                                                          @RequestParam(value = "pageIndex") int pageIndex,
                                                          @RequestParam(value = "pageSize") int pageSize) {
        try {
            Map<String, Object> params = new HashMap<>(10);
            params.put("prjid", projectId);
            params.put("type", type);
            params.put("querypram", searchText);
            params.put("pageindex", pageIndex);
            params.put("pagesize", pageSize);
            CommonResponse<JSONObject> response = sendGetReq(CONTACT_UNIT_LIST_URL, params);
            if(response.isSuccess()){
                JSONObject page = new JSONObject();
                page.put("current",pageIndex);
                if(Objects.equals("3",type)){
                    page.put("pages",1);
                }else {
                    page.put("pages",((response.getData().getJSONObject("data").getInteger("count")/pageSize)+1));
                }
                page.put("total",response.getData().getJSONObject("data").getInteger("count"));
                page.put("records",response.getData().getJSONObject("data").get("list"));
                return CommonResponse.success(page);
            }
            return response;
        } catch (Exception e) {
            logger.error("获取往来单位信息列表异常，", e);
            return CommonResponse.error("获取往来单位信息列表失败!");
        }
    }

    /**
     * 根据项目Id查询项目主合同信息列表
     *
     * @param projectId 项目主键Id
     * @return success: Boolean,服务响应结果编码；
     *         message: String, 服务响应结果信息；
     *         data: {list [{id: String,合同Id；name: String,合同名称；code：String,合同编号；tyname：String,合同类别；contractsignamount：Number,合同金额；signdate：Date,签约日期}, ...]}
     */
    @GetMapping(value = "contractList")
    public CommonResponse<JSONObject> getContractList(@RequestParam(value = "projectId") String projectId) {
        try {
            Map<String, Object> params = new HashMap<>(2);
            params.put("prjid", projectId);

            CommonResponse<JSONObject> response = sendGetReq(CONTRACT_LIST_URL, params);
            if(response.isSuccess()){
                JSONObject back = new JSONObject();
                back.put("list",response.getData().getJSONObject("data").get("list"));
                return CommonResponse.success(back);
            }
            return response;

        } catch (Exception e) {
            logger.error("获取目主合同信息列表异常，", e);
            return CommonResponse.error("获取目主合同信息列表失败!");
        }
    }

    /**
     * @Author mrsir_wxp
     * @Date 2020/10/12 根据项目查询 物资合同 参照
     * @Description refContractListData
     * @Param [pageNumber, pageSize, condition, searchText]
     * @Return com.ejianc.framework.core.response.CommonResponse<com.baomidou.mybatisplus.core.metadata.IPage<java.util.Map>>
     */
    @RequestMapping(value = "/refContractListData", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<JSONObject> refContractListData(@RequestParam Integer pageNumber,
                                                                  @RequestParam Integer pageSize,
                                                                  String condition,
                                                                  String searchText) {
        String projectId = null;
        if(StringUtils.isNotEmpty(condition)){
            JSONObject con = JSON.parseObject(condition);
            if(StringUtils.isNotEmpty(con.getString("projectId"))){
                projectId = con.getString("projectId");
            }
        }
        if(StringUtils.isEmpty(projectId)){
            return CommonResponse.error("查询水电八局物资合同参照失败，项目id不能为空！");
        }
        Map<String, Object> params = new HashMap<>(10);
        params.put("prjid", projectId);
        params.put("querypram", searchText);
        params.put("pageindex", pageNumber);
        params.put("pagesize", pageSize);
        CommonResponse<JSONObject> response = null;
        try {
            response = sendGetReq(MATERIAL_CONTACT_LIST_URL, params);
        } catch (Exception e) {
            e.printStackTrace();
            return CommonResponse.error("查询水电八局物资合同参照失败");
        }
        if(response.isSuccess()){
            JSONObject page = new JSONObject();
            page.put("current",pageNumber);
            page.put("size",pageSize);
            page.put("total",response.getData().getJSONObject("data").getInteger("count"));
            page.put("pages",((response.getData().getJSONObject("data").getInteger("count")/pageSize)+1));
            page.put("records",response.getData().getJSONObject("data").get("list"));
            return CommonResponse.success(page);
        }else {
            return CommonResponse.error(response.getMsg());
        }
    }

    /**
     * 分页查询验收人员列表
     *
     * @param projectId 项目主键Id
     * @param queryParam 查询参数，可根据合同名称关键字查询
     * @param pageIndex 页码
     * @param pageSize 每页记录数
     * @return success: Boolean,服务响应结果编码；
     *         message: String, 服务响应结果信息；
     *         data: {list [{id: String,员工id；name: String,人员名称；code：String,人员编码；orgname：String,所在组织；orgid：String,组织id}, ...]}
     */
    @GetMapping(value = "employeeList")
    public CommonResponse<JSONObject> getEmployeeList(@RequestParam(value = "projectId") String projectId,
                                                      @RequestParam(value = "searchText", required = false) String searchText,
                                                      @RequestParam(value = "pageIndex") int pageIndex,
                                                      @RequestParam(value = "pageSize") int pageSize) {
        try {
            Map<String, Object> params = new HashMap<>(10);
            params.put("prjid", projectId);
            params.put("querypram", searchText);
            params.put("pageindex", pageIndex);
            params.put("pagesize", pageSize);

            CommonResponse<JSONObject> response = sendGetReq(EMPLOYEE_LIST_URL, params);
            if(response.isSuccess()){
                JSONObject page = new JSONObject();
                page.put("current",pageIndex);
                page.put("pages",((response.getData().getJSONObject("data").getInteger("count")/pageSize)+1));
                page.put("total",response.getData().getJSONObject("data").getInteger("count"));
                page.put("records",response.getData().getJSONObject("data").get("list"));
                return CommonResponse.success(page);
            }
            return response;
        } catch (Exception e) {
            logger.error("查询验收人员列表异常，", e);
            return CommonResponse.error("获取验收人员列表失败!");
        }
    }

    /**
     * 分页查询分包合同列表
     *
     * @param projectId 项目主键Id
     * @param subContractorId 分包商Id
     * @param queryParam 查询参数，可根据人员名称关键字查询
     * @param pageIndex 页码
     * @param pageSize 每页记录数
     * @return success: Boolean,服务响应结果编码；
     *         message: String, 服务响应结果信息；
     *         data: {list [{id: String,合同id，name: String,合同名称，concode：String,合同编码，fbsname：String,分包商名称，total：合同金额}, ...]}
     */
    @GetMapping(value = "subContractList")
    public CommonResponse<JSONObject> getSubContractList(@RequestParam(value = "projectId") String projectId,
                                                      @RequestParam(value = "subContractorId") String subContractorId,
                                                      @RequestParam(value = "searchText", required = false) String searchText,
                                                      @RequestParam(value = "pageIndex") int pageIndex,
                                                      @RequestParam(value = "pageSize") int pageSize) {
        try {
            Map<String, Object> params = new HashMap<>(10);
            params.put("prjid", projectId);
            params.put("fbsid", subContractorId);
            params.put("querypram", searchText);
            params.put("pageindex", pageIndex);
            params.put("pagesize", pageSize);

            CommonResponse<JSONObject> response = sendGetReq(SUB_CONTRACT_LIST_URL, params);
            if(response.isSuccess()){
                JSONObject page = new JSONObject();
                page.put("current",pageIndex);
                page.put("pages",((response.getData().getJSONObject("data").getInteger("count")/pageSize)+1));
                page.put("total",response.getData().getJSONObject("data").getInteger("count"));
                page.put("records",response.getData().getJSONObject("data").get("list"));
                return CommonResponse.success(page);
            }
            return response;

        } catch (Exception e) {
            logger.error("查询分包合同列表异常，", e);
            return CommonResponse.error("获取分包合同列表失败!");
        }
    }

    /**
     * 分页查询安全费用列表
     *
     * @param queryParam 查询参数，可根据编码、名称关键字查询
     * @param pageIndex 页码
     * @param pageSize 每页记录数
     * @return success: Boolean,服务响应结果编码；
     *         message: String, 服务响应结果信息；
     *         data: {list [{id: String,id；eprjname: String,名称；eprjcode：String,编码}, ...]}
     *
     */
    @GetMapping(value = "securityCostsList")
    public CommonResponse<JSONObject> getSecurityCostsList(@RequestParam(value = "queryParam", required = false) String queryParam,
                                                           @RequestParam(value = "pageIndex") int pageIndex,
                                                           @RequestParam(value = "pageSize") int pageSize) {
        try {
            Map<String, Object> params = new HashMap<>(6);
            params.put("querypram", queryParam);
            params.put("pageindex", pageIndex);
            params.put("pagesize", pageSize);

            return sendGetReq(SECURITY_COSTS_LIST_URL, params);
        } catch (Exception e) {
            logger.error("查询安全费用列表异常，", e);
            return CommonResponse.error("获取安全费用列表失败!");
        }
    }

    /**
     * 分页查询研发项目列表
     *
     * @param contractId 合同Id
     * @param pageIndex 页码
     * @param pageSize 每页记录数
     * @return success: Boolean,服务响应结果编码；
     *         message: String, 服务响应结果信息；
     *         data: {list [{id: String,id；name: String,名称；code：String,编码}, ...]}
     */
    @GetMapping(value = "greenDreamProjectList")
    public CommonResponse<JSONObject> getGreenDreamProjectList(@RequestParam(value = "contractId") String contractId,
                                                               @RequestParam(value = "pageIndex") int pageIndex,
                                                               @RequestParam(value = "pageSize") int pageSize) {
        try {
            Map<String, Object> params = new HashMap<>(6);
            params.put("zhtid", contractId);
            params.put("pagesize", pageSize);
            params.put("pageindex", pageIndex);

            CommonResponse<JSONObject> response = sendGetReq(GREEN_DREAM_PROJECT_LIST_URL, params);
            if(response.isSuccess()){
                JSONObject page = new JSONObject();
                page.put("pages",((response.getData().getJSONObject("data").getInteger("count")/pageSize)+1));
                page.put("current",pageIndex);
                page.put("total",response.getData().getJSONObject("data").getInteger("count"));
                page.put("records",response.getData().getJSONObject("data").get("list"));
                return CommonResponse.success(page);
            }
            return response;
        } catch (Exception e) {
            logger.error("查询研发项目列表异常，", e);
            return CommonResponse.error("获取研发项目列表失败!");
        }
    }

    /**
     * 分页查询物资库存列表
     *
     * @param type 出库类别：1:甲供材,6:自购消耗材,2:自购周转材,4:租赁周转材,8:调拨,9:零星采购,10:代保管产品,5:退货出库,11:委托加工原材料出库,12:自有资产改扩建
     * @param warehouseId 仓库Id
     * @param searchText 查询条件，根据材料编码、材料名称、规格型号、类别筛选
     * @param reportDate 发料日期 接受日期格式需为yyyy-mm-dd
     * @param contractId 合同Id
     * @param pageIndex 页码
     * @param pageSize 每页记录数
     * @return success: Boolean,服务响应结果编码；
     *         message: String, 服务响应结果信息；
     *         data: {list [{id: String,id；name: String,材料名称；code：String,材料编码；spec：String,规格/型号；
     *                  sortname：String,类别，unitname：String,单位；price：Number，单价；remainqty：Number,可用库存}, ...]}
     */
    @GetMapping(value = "inventoryList")
    public CommonResponse<JSONObject> getInventoryList(@RequestParam(value = "type") String type,
                                                               @RequestParam(value = "warehouseId") String warehouseId,
                                                               @RequestParam(value = "searchText", required = false) String searchText,
                                                               @RequestParam(value = "reportDate") String reportDate,
                                                               @RequestParam(value = "contractId", required = false) String contractId,
                                                               @RequestParam(value = "pageIndex") int pageIndex,
                                                               @RequestParam(value = "pageSize") int pageSize) {
        try {
            Map<String, Object> params = new HashMap<>(10);
            params.put("intype", type);
            params.put("storehouseid", warehouseId);
            params.put("querypram", searchText);
            params.put("reportdate", reportDate);
            params.put("conid", contractId);
            params.put("pageindex", pageIndex);
            params.put("pagesize", pageSize);

            CommonResponse<JSONObject> response = sendGetReq(INVENTORY_LIST_URL, params);
            if(response.isSuccess()){
                JSONObject page = new JSONObject();
                page.put("current",pageIndex);
                page.put("pages",((response.getData().getJSONObject("data").getInteger("count")/pageSize)+1));
                page.put("total",response.getData().getJSONObject("data").getInteger("count"));
                page.put("records",response.getData().getJSONObject("data").get("list"));
                return CommonResponse.success(page);
            }
            return response;
        } catch (Exception e) {
            logger.error("查询物资库存列表异常，", e);
            return CommonResponse.error("获取物资库存列表失败!");
        }
    }

    /**
     * 根据项目Id查询流程模板列表
     *
     * @param projectId 项目主键Id
     * @return success: Boolean,服务响应结果编码；
     *         message: String, 服务响应结果信息；
     *         data: {list [{id: String,id；name: String,流程模板名称；orgid：String,组织id；flowxl：接口未说明}, ...]}
     */
    @GetMapping(value = "processTemplateList")
    public CommonResponse<JSONObject> getProcessTemplateList(@RequestParam(value = "projectId") String projectId) {
        try {
            Map<String, Object> params = new HashMap<>(10);
            params.put("prjid", projectId);

            CommonResponse<JSONObject> resp = sendGetReq(PROCESS_TEMPLATE_LIST_URL, params);

            JSONArray list = (JSONArray) resp.getData().get("list");
            if(!list.isEmpty()) {
                for(Object obj : list) {
                    JSONObject processTemplate = (JSONObject) obj;
                    processTemplate.put("flow", XmlUtils.parse(processTemplate.get("flowxl").toString()));
                }
            }

            return resp;
        } catch (Exception e) {
            logger.error("查询流程模板列表异常，", e);
            return CommonResponse.error("获取流程模板列表失败!");
        }
    }

    /**
     * 分页查询流程节点选择人员列表
     *
     * @param projectId 项目Id
     * @param roleId 角色Id
     * @param queryParam 查询条件，根据人员编号和名称筛选
     * @param pageIndex 页码
     * @param pageSize 每页记录数
     * @return success: Boolean,服务响应结果编码；
     *         message: String, 服务响应结果信息；
     *         data: {list [{id: String,id；name: String,人员名称；code：String,人员编码；postnames：String,岗位名称}, ...]}
     */
    @GetMapping(value = "flowEmployeeList")
    public CommonResponse<JSONObject> getFlowEmployeeList(@RequestParam(value = "projectId") String projectId,
                                                       @RequestParam(value = "roleId") String roleId,
                                                       @RequestParam(value = "queryParam", required = false) String queryParam,
                                                       @RequestParam(value = "pageIndex") int pageIndex,
                                                       @RequestParam(value = "pageSize") int pageSize) {
        try {
            Map<String, Object> params = new HashMap<>(10);
            params.put("prjid", projectId);
            params.put("roleid", roleId);
            params.put("querypram", queryParam);
            params.put("pageindex", pageIndex);
            params.put("pagesize", pageSize);

            return sendGetReq(FLOW_EMPLOYEE_LIST_URL, params);
        } catch (Exception e) {
            logger.error("查询流程节点选择人员列表异常，", e);
            return CommonResponse.error("获取流程节点选择人员列表失败!");
        }
    }

    private CommonResponse<JSONObject> sendGetReq(String url, Map<String, Object> params) throws Exception {
        Map<String, String> headers = new HashMap<>(5);
        JSONObject resp = new JSONObject();

        JSONObject accessToken = getSDBJAccessToken();
        if(null == accessToken) {
            throw new BusinessException("获取服务accessToken失败！");
        }
        headers.put("access_token", accessToken.get("token").toString());
        String newUrl = SDBJ_SERVER_URL+url;
        String reqResp =  HttpTookit.get(newUrl, params, headers, 10000, 20000);
        logger.info("发送get请求【地址： {}, 参数：{}, header: {}】, 响应结果：{}", newUrl, params, headers, reqResp);

        JSONObject jsonData = JSONObject.parseObject(reqResp);
        if(BooleanUtils.isNotTrue(jsonData.getBoolean("success"))) {
            return CommonResponse.error(null != jsonData.get("message") ? jsonData.get("message").toString() : "查询失败！");
        }
        return CommonResponse.success(jsonData);
    }
}
