package com.ejianc.business.projectapply.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.ejianc.business.projectapply.bean.ExecutiveAgentEntity;
import com.ejianc.business.projectapply.bean.ProjectApplyEntity;
import com.ejianc.business.projectapply.bean.ProjectChangeEntity;
import com.ejianc.business.projectapply.service.IProjectApplyService;
import com.ejianc.business.projectapply.service.IProjectChangeService;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.support.api.IBillTypeApi;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.auth.session.UserContext;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.skeleton.billState.service.ICommonBusinessService;
import com.ejianc.support.idworker.util.IdWorker;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import java.util.Date;
import java.util.List;

/**
 * 项目申请审批处理服务
 *
 * @author CJ
 * @Description:
 * @date 2021/9/8 19:25
 */
@Service(value = "projectApply")
public class ProjectApplyBpmServiceImpl implements ICommonBusinessService {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private IProjectApplyService projectApplyService;

    @Autowired
    private SessionManager sessionManager;

    @Autowired
    private IOrgApi orgApi;

    @Autowired
    private IBillTypeApi billTypeApi;

    @Value("${oms.splitPrjBelongOrgName}")
    private String splitPrjBelongOrgName;

    @Value("${oms.splitPrjBelongOrgType}")
    private Integer splitPrjBelongOrgType;

    @Autowired
    private IProjectChangeService projectChangeService;

    /**
     * 提交审批
     *
     * @param billId
     * @param state
     * @param billTypeCode
     * @return
     */
    @Override
    public CommonResponse<String> afterSubmitProcessor(Long billId, Integer state, String billTypeCode) {
        ProjectApplyEntity e = projectApplyService.selectById(billId);
        UserContext userContext = sessionManager.getUserContext();

        e.setCommitDate(new Date());
        e.setCommitUserCode(userContext.getUserCode());
        e.setCommitUserName(userContext.getUserName());
        projectApplyService.saveOrUpdate(e, false);

        return CommonResponse.success("项目申请审批回调处理成功！");
    }

    /**
     * 直审/审批通过
     *
     * @param billId
     * @param state
     * @param billTypeCode
     * @return
     */
    @Override
    public CommonResponse<String> afterApprovalProcessor(Long billId, Integer state, String billTypeCode) {
        ProjectApplyEntity e = projectApplyService.selectById(billId);
        e.setEffectiveDate(new Date());
        if(state == 1) {
            //直审时记录
            UserContext userContext = sessionManager.getUserContext();
            e.setCommitDate(new Date());
            e.setCommitUserCode(userContext.getUserCode());
            e.setCommitUserName(userContext.getUserName());
        }

        //生成目部
        OrgVO org = new OrgVO();
        String msg = generateOrgInfo(e, org);
        if(StringUtils.isNotBlank(msg)) {
            return CommonResponse.error(msg);
        }
        CommonResponse<OrgVO> orgSaveResp = orgApi.saveOrgInfo(org);
        if(!orgSaveResp.isSuccess()) {
            logger.error("生成项目申请对应项目部信息失败，{}", orgSaveResp.getMsg());
            return CommonResponse.error("生成项目申请对应项目部信息失败! ");
        }

        projectApplyService.saveOrUpdate(e, false);
        return CommonResponse.success("项目申请审批回调处理成功！");
    }

    /**
     * 根据项目申请信息生成对应项目部信息
     *
     * @param e
     * @return
     */
    private String generateOrgInfo(ProjectApplyEntity e, OrgVO org) {
        //设置项目部Id
        org.setId(IdWorker.getId());
        //设置项目部名称
        org.setShortName(e.getCnName());
        org.setName(e.getCnName());
        //组织类型设置为项目部
        org.setOrgType(OrgVO.ORG_TYPE_DEPARTMENT);

        OrgVO parent = null;
        //设置项目部所属组织Id：拆分项目，上级组织固定为组织类型为"集团"的山东电力建设第三工程有限公司
        //非拆分项目，生成的项目部，上级组织为"执行主体"
        if(e.getProjectSplit()) {
            //设置上级组织
            OrgVO query = new OrgVO();
            query.setName(splitPrjBelongOrgName);
            query.setTenantId(InvocationInfoProxy.getTenantid());
            query.setOrgType(splitPrjBelongOrgType);
            CommonResponse<OrgVO> parentOrgInfoResp = orgApi.findOneByOrgVO(query);

            if(!parentOrgInfoResp.isSuccess()) {
                logger.error("拆分项目查询指定上级: 名称-{}，类型-{}的组织失败！", splitPrjBelongOrgName, splitPrjBelongOrgType);
                return "拆分项目查询指定上级: 名称-" + splitPrjBelongOrgName + "，类型-" +splitPrjBelongOrgType + "的组织失败！";
            }
            parent = parentOrgInfoResp.getData();
        } else {
            //上级组织为执行主体
            ExecutiveAgentEntity sub = e.getExecutiveAgentList().get(0);
            CommonResponse<OrgVO> parentOrgInfoResp = orgApi.detailById(sub.getExecutiveAgentId());
            if(!parentOrgInfoResp.isSuccess()) {
                logger.error("非拆分项目查询指定上级: 名称-{}，id-{}的组织失败！", sub.getExecutiveAgentName(), sub.getExecutiveAgentId());
                return "非拆分项目查询指定上级: 名称-" + sub.getExecutiveAgentName() + "的组织失败！";
            }

            parent = parentOrgInfoResp.getData();
        }
        if(null == parent) {
            logger.error("项目申请单据[id-{}]对应上级组织在系统中不存在，审批回调失败！");
            return "项目上级组织在系统中不存在！";
        }

        org.setParentId(parent.getId());
        org.setEnterpriseId(parent.getEnterpriseId());
        org.setIsParent(false);
        org.setTenantId(parent.getTenantId());
        //设置启用状态
        org.setState(1);

        //保存项目部信息以及所属组织信息
        e.setOrgName(parent.getName());
        e.setProjectDepartmentId(org.getId());
        e.setOrgId(org.getParentId());

        return null;
    }


    /**
     * 撤回
     *
     * @param billId
     * @param state
     * @param billTypeCode
     * @return
     */
    @Override
    public CommonResponse<String> beforeHasBpmBack(Long billId, Integer state, String billTypeCode) {
        return bpmBackCheck(billId, state, billTypeCode);
    }

    /**
     * 弃审
     *
     * @param billId
     * @param state
     * @param billTypeCode
     * @return
     */
    @Override
    public CommonResponse<String> beforeAbstainingProcessor(Long billId, Integer state, String billTypeCode) {
        return bpmBackCheck(billId, state, billTypeCode);
    }

    /**
     * 弃审/撤回前回调校验
     *  查看该项目是否已经发生业务单据（合同会签、无合同付款申请单、无合同收款登记、无合同开票申请、无合同收款登记），单据范围为：任何审批状态的、有效的单据
     *  单据变更校验，已变更的单据不可弃审或者撤回
     *
     * @param billId
     * @param state
     * @param billTypeCode
     * @return
     */
    private CommonResponse<String> bpmBackCheck(Long billId, Integer state, String billTypeCode) {
        ProjectApplyEntity e = projectApplyService.selectById(billId);

        logger.info("项目申请单据[billId-{}],[billTypeCode-{}]执行弃审前校验！", billId, billTypeCode);

        //单据变更校验，已变更的单据不可弃审或者撤回
        List<ProjectChangeEntity> changeRecordsList = projectChangeService.getAllByProjectId(e.getId());
        if(CollectionUtils.isNotEmpty(changeRecordsList)) {
            logger.error("项目申请单据【billId-{}】,【billTypeCode-{}】执行撤回操作校验结束，该单据已做变更，不可进行撤回、弃审操作！");
            return CommonResponse.error("单据存在变更记录，不可执行该操作！");
        }

        //下有引用校验
        CommonResponse<String> resp = billTypeApi.checkQuote(billTypeCode, billId);
        logger.info("项目申请单据[billId-{}],[billTypeCode-{}]校验被下游单据引用结果: ", billId, billTypeCode, JSONObject.toJSONString(resp));

        if(!resp.isSuccess()) {
            logger.info("项目申请单据【billId-{}】,【billTypeCode-{}】执行撤回操作校验结束，未通过校验，原因：{}", resp.getMsg());
            return CommonResponse.error(resp.getMsg());
        }

        return CommonResponse.success("项目申请审批回调处理成功!");
    }

    /**
     * 撤回回调处理
     *
     * @param billId
     * @param state
     * @param billTypeCode
     * @return
     */
    @Override
    public CommonResponse<String> afterHasBpmBack(Long billId, Integer state, String billTypeCode) {
        return doBpmBack(billId, state, billTypeCode);
    }

    /**
     * 弃审回调处理
     *
     * @param billId
     * @param state
     * @param billTypeCode
     * @return
     */
    @Override
    public CommonResponse<String> afterAbstainingProcessor(Long billId, Integer state, String billTypeCode) {
        return doBpmBack(billId, state, billTypeCode);
    }

    /**
     * 删除项目对应项目部以及项目部下相关信息
     *
     * @param billId
     * @param state
     * @param billTypeCode
     * @return
     */
    private CommonResponse<String> doBpmBack(Long billId, Integer state, String billTypeCode) {
        ProjectApplyEntity e = projectApplyService.selectById(billId);
        logger.info("项目申请单据-【{}】执行撤回逻辑", JSONObject.toJSONString(e));

        //删除对应项目部信息
        CommonResponse<String> delResp = orgApi.delSDSJPrjDepartmentByOrgId(e.getProjectDepartmentId());
        logger.error("项目申请单据[billId-{}],[billTypeCode-{}]删除对应项目部结果: ", billId, billTypeCode, JSONObject.toJSONString(delResp));
        if(!delResp.isSuccess()) {
            logger.error("项目申请单据[billId-{}],[billTypeCode-{}]删除对应项目部失败: ", billId, billTypeCode, delResp.getMsg());
            return CommonResponse.error("项目申请单据对应项目部删除失败！");
        }

        //清空生效时间/提交人相关信息
        e.setEffectiveDate(null);
        e.setCommitDate(null);
        e.setCommitUserCode(null);
        e.setCommitUserName(null);
        projectApplyService.saveOrUpdate(e, false);

        return CommonResponse.success("执行项目申请单据撤回逻辑成功! ");
    }
}
