package com.ejianc.business.projectapply.controller;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.projectapply.bean.ProjectApplyEntity;
import com.ejianc.business.projectapply.service.IProjectApplyService;
import com.ejianc.business.projectapply.vo.ExecutiveAgentVO;
import com.ejianc.business.projectapply.vo.ProjectApplyVO;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.auth.session.UserContext;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.kit.time.DateFormatUtil;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.ExcelExport;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 项目申请控制器
 *
 * @author CJ
 * @Description: 项目申请控制器
 * @date 2021/9/6 10:10
 */
@RestController
@RequestMapping("/projectApply/")
public class ProjectApplyController {

    private final String BILL_CODE_RULE = "PRJ-APPLY-01";

    @Autowired
    private IBillCodeApi billCodeApi;

    @Autowired
    private IProjectApplyService projectApplyService;

    @Autowired
    private IOrgApi orgApi;

    @Autowired
    private SessionManager sessionManager;

    /**
     * 分页查询项目申请列表
     *
     * @param queryParam
     * @return
     */
    @PostMapping(value = "pageList")
    public CommonResponse<IPage<ProjectApplyVO>> pageList(@RequestBody QueryParam queryParam) {
        return CommonResponse.success("分页查询成功！", queryPage(queryParam));
    }

    private IPage<ProjectApplyVO> queryPage(QueryParam queryParam) {
        Parameter range = queryParam.getParams().get("range");
        queryParam.getParams().remove("range");
        List<Long> orgIds =new ArrayList<>();

        if("selfAndBelow".equals(range.getValue())) {
            //查询本下
            CommonResponse<List<OrgVO>> childOrgResp = orgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId());
            if(!childOrgResp.isSuccess()) {
                throw new BusinessException("分页列表查询失败, 查询组织信息失败！");
            }
            orgIds = childOrgResp.getData().stream().map(OrgVO::getId).collect(Collectors.toList());
        } else {
            //查询本级
            orgIds.add(InvocationInfoProxy.getOrgId());
        }

        Map<String, Parameter> params = queryParam.getParams();
        String billState = null != params.get("billState") ? params.get("billState").getValue().toString() : null;
        String createTime = null != params.get("createTime") ? params.get("createTime").getValue().toString() : null;
        String projectSplit = null != params.get("projectSplit") ? params.get("projectSplit").getValue().toString() : null;
        if(StringUtils.isNotBlank(createTime)) {
            String[] rangeArr = createTime.split(",");
            rangeArr[0] += " 00:00:00";
            rangeArr[1] += " 23:59:59";
            createTime = StringUtils.join(rangeArr, ",");
        }
        long count = projectApplyService.countPage(queryParam.getSearchText(), billState, createTime, projectSplit, orgIds, null);
        IPage<ProjectApplyVO> page = new Page<>();
        if(count > 0) {
            long pageSize = queryParam.getPageSize();
            pageSize = pageSize <= 0 ? 10 : pageSize;
            long totalPage = count / pageSize + (count % pageSize == 0 ? 0 : 1);
            long pageNum = queryParam.getPageIndex();
            if(totalPage < pageNum) {
                pageNum = totalPage;
            }

            long startLine = (pageNum - 1 < 0 ? 0 : pageNum - 1) * pageSize;
            List<ProjectApplyVO> records = projectApplyService.pageList(queryParam.getSearchText(), billState,
                    createTime, projectSplit, orgIds, startLine, pageSize, queryParam.getOrderMap().get("createTime"), null);

            page.setCurrent(pageNum);
            page.setSize(pageSize);
            page.setTotal(count);
            page.setRecords(records);
        }

        return page;
    }

    /**
     * 项目申请保存/更新
     *
     * @param vo
     * @return
     */
    @PostMapping(value = "saveOrUpdate")
    public CommonResponse<ProjectApplyVO> saveOrUpdate(@RequestBody ProjectApplyVO vo) {
        ProjectApplyVO resp = null;
        ProjectApplyEntity saveEntity = null;

        Set<String> repeatCheck = new HashSet<>();
        if(CollectionUtils.isNotEmpty(vo.getExecutiveAgentList())) {
            //重新将所有执行主体名称汇总
            StringBuilder executiveAgentNames = new StringBuilder();
            String repeatCheckStr = null;
            for(ExecutiveAgentVO e : vo.getExecutiveAgentList()) {
                if(!"del".equals(e.getRowState())) {
                    executiveAgentNames.append(e.getExecutiveAgentName()).append(",");
                    e.setProjectSplit(vo.getProjectSplit());
                    repeatCheckStr = getRepeatCheckStr(e.getSplitType(), e.getExecutiveAgentId());
                    if(e.getProjectSplit() && repeatCheck.contains(repeatCheckStr)) {
                        return CommonResponse.error("保存失败，执行主体："+e.getExecutiveAgentName()+"，拆分类型： "+e.getSplitType()+""+" 重复！");
                    } else {
                        repeatCheck.add(repeatCheckStr);
                    }
                }
            }

            vo.setExecutiveAgentNames(executiveAgentNames.substring(0, executiveAgentNames.length()-1));
        } else {
            return CommonResponse.error("保存失败，执行主体不能为空！");
        }
        if(StringUtils.isBlank(vo.getExecutiveAgentNames())) {
            return CommonResponse.error("保存失败，执行主体不能为空！");
        }

        ProjectApplyVO checkVo = new ProjectApplyVO();
        checkVo.setId(vo.getId());
        checkVo.setCnName(vo.getCnName());
        List<ProjectApplyEntity> list = projectApplyService.getDuplicateList(checkVo);
        if(CollectionUtils.isNotEmpty(list)) {
            return CommonResponse.error("保存失败，项目中文名称已被使用.");
        }
        if(StringUtils.isNotBlank(vo.getEnName())) {
            checkVo.setCnName(null);
            checkVo.setEnName(vo.getEnName());
            list = projectApplyService.getDuplicateList(checkVo);
            if(CollectionUtils.isNotEmpty(list)) {
                return CommonResponse.error("保存失败, 项目英文名称已被使用.");
            }
        }

        saveEntity = BeanMapper.map(vo, ProjectApplyEntity.class);
        UserContext userContext = sessionManager.getUserContext();
        if(null == vo.getId()) {
            //生成编码
            CommonResponse<String> billCodeResp = billCodeApi.getCodeBatchByRuleCode(BILL_CODE_RULE, InvocationInfoProxy.getTenantid());
            if(!billCodeResp.isSuccess()) {
                return CommonResponse.error("保存失败，生成单据编码失败！");
            }
            saveEntity.setBillCode(billCodeResp.getData());
            //单据设置为自由态
            saveEntity.setBillState(BillStateEnum.UNCOMMITED_STATE.getBillStateCode());
            //设置为启动状态
            saveEntity.setStatus(ProjectApplyVO.PROJECT_STATUS_OPEN);
            //设置创建人名称
            saveEntity.setCreateUserName(userContext.getUserName());
        } else {
            ProjectApplyEntity dbEntity = projectApplyService.selectById(vo.getId());
            saveEntity.setCreateUserName(dbEntity.getCreateUserName());
            saveEntity.setCreateTime(dbEntity.getCreateTime());
            saveEntity.setCreateUserCode(dbEntity.getCreateUserCode());
            saveEntity.setModifyUserName(userContext.getUserName());
            saveEntity.setBillCode(dbEntity.getBillCode());
            saveEntity.setBillState(dbEntity.getBillState());
        }

        projectApplyService.saveOrUpdate(saveEntity, false);
        resp = BeanMapper.map(saveEntity, ProjectApplyVO.class);

        return CommonResponse.success("保存成功！", resp);
    }

    private String getRepeatCheckStr(String splitType, Long executiveAgentId) {
        if(StringUtils.isBlank(splitType)) {
            return executiveAgentId.toString();
        } else if(splitType.indexOf(",") < 0) {
            return splitType + executiveAgentId;
        } else {
            String[] typeArr = splitType.split(",");
            Arrays.sort(typeArr, String.CASE_INSENSITIVE_ORDER);
            return StringUtils.join(typeArr, ",") + executiveAgentId;
        }
    }

    /**
     * 查询指定的项目申请详情
     *
     * @param id
     * @return
     */
    @GetMapping(value = "queryDetail")
    public CommonResponse<ProjectApplyVO> queryDetail(@RequestParam(value = "id") Long id) {
        ProjectApplyVO resp = null;

        ProjectApplyEntity dbEntity = projectApplyService.selectById(id);
        if(null != dbEntity) {
            resp = BeanMapper.map(dbEntity, ProjectApplyVO.class);
        }

        return CommonResponse.success("查询详情成功！", resp);
    }

    /**
     * 项目申请删除
     *
     * @param vos
     * @return
     */
    @PostMapping(value = "delete")
    public CommonResponse<String> delete(@RequestBody List<ProjectApplyVO> vos) {
        for(ProjectApplyVO vo : vos) {
            if(BillStateEnum.APPROVING_HAS_STATE.getBillStateCode().equals(vo.getBillState()) || BillStateEnum.APPROVING_UNEXAM_STATE.getBillStateCode().equals(vo.getBillState())) {
                return CommonResponse.error("删除失败，不能删除审批中的单据！");
            }
        }
        vos.forEach(vo -> {
            projectApplyService.removeById(vo.getId(), false);
        });

        return CommonResponse.success("删除成功！");
    }

    /**
     * 项目申请列表导出
     *
     * @param queryParam
     * @param response
     */
    @PostMapping(value = "excelExport")
    public void excelExport(@RequestBody QueryParam queryParam, HttpServletResponse response) {
        List<ProjectApplyVO> voList = null;

        queryParam.setPageIndex(1);
        queryParam.setPageSize(10000);
        IPage<ProjectApplyVO> page = queryPage(queryParam);
        voList = page.getRecords();
        voList.forEach(vo -> {
            vo.setCreateTimeStr(DateFormatUtil.formatDate("yyyy-MM-dd HH:mm:ss", vo.getCreateTime()));
            vo.setSplitTypeDisplay(vo.getProjectSplit() ? "拆分" : "非拆分");
            vo.setBillStateName(BillStateEnum.getEnumByStateCode(vo.getBillState()).getDescription());
        });

        Map<String, Object> beans = new HashMap<String, Object>();
        beans.put("records", voList);
        ExcelExport.getInstance().export("sdsjProject-export.xlsx", beans, response);
    }
}
