package com.ejianc.framework.skeleton.template.event;

import java.util.LinkedHashSet;
import java.util.Set;
import java.util.concurrent.Executor;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.context.ApplicationListener;
import org.springframework.context.event.GenericApplicationListener;
import org.springframework.context.event.GenericApplicationListenerAdapter;
import org.springframework.core.ResolvableType;
import org.springframework.lang.Nullable;
import org.springframework.util.ErrorHandler;

/**
 * 
 * 支持读写锁和异步执行的观察者模式实现
 * @author: tongming.wei
 */

public class DefaultPublisher implements YYJZEventPublisher {
	// 保证调用顺序和注册的顺序一致，不使用synchronized 是为了效率
	private final ReadWriteLock readWriteLock = new ReentrantReadWriteLock(true);
	protected final Lock readLock = readWriteLock.readLock();
	protected final Lock writeLock = readWriteLock.writeLock();
//	private final List<YYJZListener> listeners = new ArrayList<>();
	private final Set<YYJZListener<?>> listeners =  new LinkedHashSet<>();
	
	// 实现listener异步执行
	private Executor taskExecutor;
	
	// 异步调用时结果反馈给调用者
	private ErrorHandler errorHandler;
	
	public Executor getTaskExecutor() {
		return taskExecutor;
	}

	public void setTaskExecutor(Executor taskExecutor) {
		this.taskExecutor = taskExecutor;
	}

	public ErrorHandler getErrorHandler() {
		return errorHandler;
	}

	public void setErrorHandler(ErrorHandler errorHandler) {
		this.errorHandler = errorHandler;
	}

	public YYJZListener<?> addListener(YYJZListener<?> listener) {
		this.writeLock.lock();
		try {
			this.listeners.add(listener);
		} finally {
			// Unlock the writer lock
			this.writeLock.unlock();
		}
		return listener;
	}

	public void removeListener(YYJZListener<?> listener) {
		this.writeLock.lock();
		try {
			// Remove the listener from the list of the registered listeners
			this.listeners.remove(listener);
		} finally {
			// Unlock the writer lock
			this.writeLock.unlock();
		}
	}

	public void removeAllListeners() {
		this.writeLock.lock();
		try {
			// Remove the listener from the list of the registered listeners
			this.listeners.clear();
		} finally {
			// Unlock the writer lock
			this.writeLock.unlock();
		}
	}

	public void fireEvent(YYJZEvent event) {
		this.fireEvent(event, ResolvableType.forInstance(event));
	}
	
	protected boolean supportsEvent(
			ApplicationListener<?> listener, ResolvableType eventType, @Nullable Class<?> sourceType) {

		GenericApplicationListener smartListener = (listener instanceof GenericApplicationListener ?
				(GenericApplicationListener) listener : new GenericApplicationListenerAdapter(listener));
		return (smartListener.supportsEventType(eventType) && smartListener.supportsSourceType(sourceType));
	}
	
	public void fireEvent(YYJZEvent event, @Nullable ResolvableType eventType) {
		this.readLock.lock();
		try {
			@SuppressWarnings("unused")
			ResolvableType type = (eventType != null ? eventType : ResolvableType.forInstance(event));
			for (final YYJZListener<?> listener : listeners) {
				Executor executor = getTaskExecutor();
				if (executor != null) {
					executor.execute(() -> invokeListener(listener, event));
				}
				else {
					invokeListener(listener, event);
				}
			}
		} finally {
			// Unlock the reader lock
			this.readLock.unlock();
		}
	}
	
	protected void invokeListener(YYJZListener<?> listener, YYJZEvent event) {
		ErrorHandler errorHandler = getErrorHandler();
		if (errorHandler != null) {
			try {
				doInvokeListener(listener, event);
			}
			catch (Throwable err) {
				errorHandler.handleError(err);
			}
		}
		else {
			doInvokeListener(listener, event);
		}
	}

	@SuppressWarnings({"unchecked", "rawtypes"})
	private void doInvokeListener(YYJZListener listener, YYJZEvent event) {
		try {
			listener.onEvent(event);
		}
		catch (ClassCastException ex) {
			String msg = ex.getMessage();
			if (msg == null || matchesClassCastMessage(msg, event.getClass().getName())) {
				// Possibly a lambda-defined listener which we could not resolve the generic event type for
				// -> let's suppress the exception and just log a debug message.
				Log logger = LogFactory.getLog(getClass());
				if (logger.isDebugEnabled()) {
					logger.debug("Non-matching event type for listener: " + listener, ex);
				}
			}
			else {
				throw ex;
			}
		}
	}
	
	private boolean matchesClassCastMessage(String classCastMessage, String eventClassName) {
		// On Java 8, the message simply starts with the class name: "java.lang.String cannot be cast..."
		if (classCastMessage.startsWith(eventClassName)) {
			return true;
		}
		// On Java 9, the message contains the module name: "java.base/java.lang.String cannot be cast..."
		int moduleSeparatorIndex = classCastMessage.indexOf('/');
		if (moduleSeparatorIndex != -1 && classCastMessage.startsWith(eventClassName, moduleSeparatorIndex + 1)) {
			return true;
		}
		// Assuming an unrelated class cast failure...
		return false;
	}
}
