package com.ejianc.framework.skeleton.billState.controller;

import cn.hutool.json.JSONArray;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.ResponseBody;

import com.ejianc.foundation.metadata.api.IMdApi;
import com.ejianc.foundation.metadata.vo.MdReferVO;
import com.ejianc.foundation.support.api.IBillTypeApi;
import com.ejianc.foundation.support.vo.BillTypeVO;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.skeleton.billState.param.CommonBusinessParam;
import com.ejianc.framework.skeleton.billState.service.ICommonBusinessService;
import com.ejianc.framework.skeleton.refer.util.ContextUtil;

/**
 * 流程中心事件回调调试方法
 * 
 * @author guoming
 */
@Controller
@RequestMapping(value = "common/testBpm")
public class CommonTestBpmController {
	// 日志
	private final Logger logger = LoggerFactory.getLogger(getClass());

	@Autowired
	private IMdApi mdApi;
	
	@Autowired
	private IBillTypeApi billTypeApi;

	public static String toLowerCaseFirstOne(String param) {
		if (Character.isLowerCase(param.charAt(0))) {
			return param;
		} else {
			return (new StringBuilder()).append(Character.toLowerCase(param.charAt(0))).append(param.substring(1)).toString();
		}
	}
	
	private ICommonBusinessService obtainBusinessService(CommonBusinessParam businessParam) {
		if(businessParam.getBillTypeCode()==null){
			throw new BusinessException("缺少单据类型编码");
		}
		if(businessParam.getBillId()==null){
			throw new BusinessException("缺少单据类型编码");
		}
		Long metadataId = businessParam.getMetadataId();
		if(businessParam.getMetadataId()==null){
			CommonResponse<BillTypeVO> billTypeResponse = billTypeApi.getByCode(businessParam.getBillTypeCode());
			if(billTypeResponse.isSuccess()) {
				BillTypeVO billTypeVo = billTypeResponse.getData();
				metadataId = billTypeVo.getMetadataId();
			}else{
				throw new BusinessException("未根据单据类型查询到对应的元数据");
			}
		}
		logger.info("调用元数据api参数：---------------"+metadataId);
		CommonResponse<MdReferVO> mdClassResponse = mdApi.queryMetadataById(metadataId);
		if (mdClassResponse.isSuccess()) {
			MdReferVO mdClassVo = mdClassResponse.getData();
			String serviceName = toLowerCaseFirstOne(mdClassVo.getEntityName().replace("Entity", ""));
			ICommonBusinessService businessService = null;
			try{
				businessService = ContextUtil.getBean(serviceName, ICommonBusinessService.class);
			}catch (Exception e){
				businessService = ContextUtil.getBean("defaultBusinessService", ICommonBusinessService.class);
			}
			return businessService;
		}
		return null;
	}

	/**
	 * 提交前回调
	 * 
	 * @param billId
	 * @param state
	 * @return
	 */
	@RequestMapping(value = "beforeSubmit", method = RequestMethod.POST)
	@ResponseBody
	public CommonResponse<String> beforeSubmitProcessor(@RequestBody CommonBusinessParam businessParam) {
		ICommonBusinessService businessService = obtainBusinessService(businessParam);
		if(businessService != null) {
			return businessService.beforeSubmitProcessor(businessParam.getBillId(), businessParam.getBillState(), businessParam.getBillTypeCode());
		}
		return CommonResponse.error("网络异常，请稍后重试！");
	}

	/**
	 * 提交完回调
	 * 
	 * @param
	 * @return
	 */
	@RequestMapping(value = "afterSubmit", method = RequestMethod.POST)
	@ResponseBody
	public CommonResponse<String> afterSubmitProcessor(@RequestBody CommonBusinessParam businessParam) {
		ICommonBusinessService businessService = obtainBusinessService(businessParam);
		if(businessService != null) {
			return businessService.afterSubmitProcessor(businessParam.getBillId(), businessParam.getBillState(), businessParam.getBillTypeCode());
		}
		return CommonResponse.error("网络异常，请稍后重试！");
	}

	/**
	 * 有审批流的撤回前回调
	 * 
	 * @param billId
	 * @param state
	 * @return
	 */
	@RequestMapping(value = "beforeHasBpmBack", method = RequestMethod.POST)
	@ResponseBody
	public CommonResponse<String> beforeHasBpmBack(@RequestBody CommonBusinessParam businessParam) {
		ICommonBusinessService businessService = obtainBusinessService(businessParam);
		if(businessService != null) {
			return businessService.beforeHasBpmBack(businessParam.getBillId(), businessParam.getBillState(), businessParam.getBillTypeCode());
		}
		return CommonResponse.error("网络异常，请稍后重试！");
	}

	/**
	 * 有审批流的撤回后回调
	 * 
	 * @param
	 * @return
	 */
	@RequestMapping(value = "afterHasBpmBack", method = RequestMethod.POST)
	@ResponseBody
	public CommonResponse<String> afterHasBpmBack(@RequestBody CommonBusinessParam businessParam) {
		ICommonBusinessService businessService = obtainBusinessService(businessParam);
		if(businessService != null) {
			return businessService.afterHasBpmBack(businessParam.getBillId(), businessParam.getBillState(), businessParam.getBillTypeCode());
		}
		return CommonResponse.error("网络异常，请稍后重试！");
	}

	/**
	 * 审批节点审批中时节点审批前回调
	 * 
	 * @param billId
	 * @param state
	 * @return
	 */
	@RequestMapping(value = "beforeInApprovalBack", method = RequestMethod.POST)
	@ResponseBody
	public CommonResponse<String> beforeInApprovalBack(@RequestBody CommonBusinessParam businessParam) {
		ICommonBusinessService businessService = obtainBusinessService(businessParam);
		if(businessService != null) {
			return businessService.beforeInApprovalBack(businessParam.getBillId(), businessParam.getBillState(), businessParam.getBillTypeCode(), businessParam.getSign());
		}
		return CommonResponse.error("网络异常，请稍后重试！");
	}

	/**
	 * 审批节点审批中时节点审批后回调
	 * 
	 * @param billId	//单据id
	 * @param state		//单据状态
	 * @param billTypeCode		//单据类型code
	 * @param taskId		//该流程id
	 * @param isEnd		//该节点是否结束
	 * @return
	 */
	@RequestMapping(value = "afterInApprovalBack", method = RequestMethod.POST)
	@ResponseBody
	public CommonResponse<String> afterInApprovalBack(@RequestBody CommonBusinessParam businessParam) {
		ICommonBusinessService businessService = obtainBusinessService(businessParam);
		if(businessService != null) {
			return businessService.afterInApprovalBack(businessParam.getBillId(), businessParam.getBillState(), businessParam.getBillTypeCode(), businessParam.getTaskId(), businessParam.getIsEnd(), businessParam.getSign(), businessParam.getOther());
		}
		return CommonResponse.error("网络异常，请稍后重试！");
	}

	/**
	 * 终审审核前回调
	 * 
	 * @param billId
	 * @param state
	 * @return
	 */
	@RequestMapping(value = "beforeApproval", method = RequestMethod.POST)
	@ResponseBody
	public CommonResponse<String> beforeApprovalProcessor(@RequestBody CommonBusinessParam businessParam) {
		ICommonBusinessService businessService = obtainBusinessService(businessParam);
		if(businessService != null) {
			return businessService.beforeApprovalProcessor(businessParam.getBillId(), businessParam.getBillState(), businessParam.getBillTypeCode());
		}
		return CommonResponse.error("网络异常，请稍后重试！");
	}

	/**
	 * 终审审核完回调
	 * 
	 * @param
	 * @return
	 */
	@RequestMapping(value = "afterApproval", method = RequestMethod.POST)
	@ResponseBody
	public CommonResponse<String> afterApprovalProcessor(@RequestBody CommonBusinessParam businessParam) {
		ICommonBusinessService businessService = obtainBusinessService(businessParam);
		if(businessService != null) {
			return businessService.afterApprovalProcessor(businessParam.getBillId(), businessParam.getBillState(), businessParam.getBillTypeCode());
		}
		return CommonResponse.error("网络异常，请稍后重试！");
	}

	/**
	 * 弃审前事件回调
	 * 
	 * @param billId
	 * @param state
	 * @return
	 */
	@RequestMapping(value = "beforeAbstaining", method = RequestMethod.POST)
	@ResponseBody
	public CommonResponse<String> beforeAbstainingProcessor(@RequestBody CommonBusinessParam businessParam) {
		ICommonBusinessService businessService = obtainBusinessService(businessParam);
		if(businessService != null) {
			return businessService.beforeAbstainingProcessor(businessParam.getBillId(), businessParam.getBillState(), businessParam.getBillTypeCode());
		}
		return CommonResponse.error("网络异常，请稍后重试！");
	}

	/**
	 * 弃审后事件回调
	 * 
	 * @param billId
	 * @param state
	 * @return
	 */
	@RequestMapping(value = "afterAbstaining", method = RequestMethod.POST)
	@ResponseBody
	public CommonResponse<String> afterAbstainingProcessor(@RequestBody CommonBusinessParam businessParam) {
		ICommonBusinessService businessService = obtainBusinessService(businessParam);
		if(businessService != null) {
			return businessService.afterAbstainingProcessor(businessParam.getBillId(), businessParam.getBillState(), businessParam.getBillTypeCode());
		}
		return CommonResponse.error("网络异常，请稍后重试！");
	}

	/**
	 * 审批--获取当前节点的审批人---从服务中选择
	 *
	 * @param billId 单据id
	 * @param billTypeCode 单据类型编码
	 * @param sign 节点标志
	 * @param other	其他参数（为了预防后面添加参数）
	 * @return
	 */
	@RequestMapping(value = "queryApproveUser", method = RequestMethod.POST)
	@ResponseBody
	public CommonResponse<JSONArray> queryApproveUser(@RequestBody CommonBusinessParam businessParam) {
		ICommonBusinessService businessService = obtainBusinessService(businessParam);
		if(businessService != null) {
			return businessService.queryApproveUser(businessParam.getBillId(), businessParam.getBillTypeCode(), businessParam.getSign(), businessParam.getOther());
		}
		return CommonResponse.error("网络异常，请稍后重试！");
	}

}
