package com.ejianc.framework.skeleton.template.es;

import java.io.IOException;
import java.lang.reflect.ParameterizedType;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import org.apache.commons.lang3.StringUtils;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.client.RequestOptions;
import org.elasticsearch.client.RestHighLevelClient;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.search.SearchHit;
import org.elasticsearch.search.SearchHits;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.elasticsearch.search.sort.FieldSortBuilder;
import org.elasticsearch.search.sort.SortOrder;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.metadata.TableInfo;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;

/**
 * 
 * ES的dao接口偷懒，就直接复用了苞米豆的了, 只查询封装
 * 
 * @author guominga 2020-02-15
 */
@SuppressWarnings("unchecked")
@Repository
public class BaseEsQueryDao<T> implements IBaseEsQueryDao<T> {
	
	private final static Integer QUERY_TIMEOUT = 60;
	private final static String ASC = "asc";

	@Autowired(required=false)
	private RestHighLevelClient client;
	
	
	private Class<T> getTClass() {
		return (Class<T>) ((ParameterizedType) getClass().getGenericSuperclass()).getActualTypeArguments()[0];
	}

	/**
     * 根据 参数map 条件，查询总记录数
     *
     * @param queryParam 查询参数
     */
	@Override
	public Long selectCount(QueryParam queryParam) {
		Long total = 0l;
		//根据class对象获取该实体对应的表名
		TableInfo tableInfo = TableInfoTool.initTableInfo(getTClass());
		String index = tableInfo.getTableName();
		SearchRequest searchRequest = new SearchRequest(index);
		SearchSourceBuilder sourceBuilder = new SearchSourceBuilder();

		//查询参数
		Map<String, Parameter> paramMap = queryParam.getParams();
		BoolQueryBuilder mainBool = QueryBuilders.boolQuery();
		BoolQueryBuilder equalBool = QueryBuilders.boolQuery();
		BoolQueryBuilder likeBool = null;
		for(Map.Entry<String, Parameter> entry: paramMap.entrySet()){
			Parameter param = entry.getValue();
			QueryBuilder queryBuilder = null;
			if(QueryParam.EQ.equals(param.getType())) {
				queryBuilder = QueryBuilders.termQuery(entry.getKey(), param.getValue());
			} else if(QueryParam.LIKE.equals(param.getType())) {
				queryBuilder = QueryBuilders.matchQuery(entry.getKey(), param.getValue());
			} else if(QueryParam.IN.equals(param.getType())) {
				queryBuilder = QueryBuilders.termQuery(entry.getKey(), param.getValue().toString().split(","));
			} else if(QueryParam.BETWEEN.equals(param.getType())) {
				String[] dataArr = param.getValue().toString().split(",");
				queryBuilder = QueryBuilders.rangeQuery(entry.getKey()).from(dataArr[0]).to(dataArr[1]).includeLower(true).includeUpper(true);
			} else if(QueryParam.GT.equals(param.getType())) {
				queryBuilder = QueryBuilders.rangeQuery(entry.getKey()).gt(param.getValue());
			} else if(QueryParam.GE.equals(param.getType())){
				queryBuilder = QueryBuilders.rangeQuery(entry.getKey()).gte(param.getValue());
			} else if(QueryParam.LT.equals(param.getType())) {
				queryBuilder = QueryBuilders.rangeQuery(entry.getKey()).lt(param.getValue());
			} else if(QueryParam.LE.equals(param.getType())) {
				queryBuilder = QueryBuilders.rangeQuery(entry.getKey()).lte(param.getValue());
			}
			if(queryBuilder!=null){
				equalBool.must(queryBuilder);
			}
		}

		//模糊搜索
		if(StringUtils.isNotBlank(queryParam.getSearchText())) {
			likeBool = new BoolQueryBuilder();
			List<String> fields = queryParam.getFuzzyFields();
			for(String field:fields) {
				likeBool.should(QueryBuilders.matchQuery(field, queryParam.getSearchText()));
			}
		}
		if(likeBool!=null){
			mainBool.must(equalBool).must(likeBool);
		}else {
			mainBool.must(equalBool);
		}

		//排序
		Map<String, String> orderMap = queryParam.getOrderMap();
		if(orderMap != null && !orderMap.isEmpty()) {
			for(Map.Entry<String, String> entry : orderMap.entrySet()) {
				sourceBuilder.sort(new FieldSortBuilder(entry.getKey()).order(ASC.equals(entry.getValue().toLowerCase())?SortOrder.ASC:SortOrder.DESC));
			}
		}
		sourceBuilder.query(mainBool);
		sourceBuilder.timeout(new TimeValue(QUERY_TIMEOUT, TimeUnit.SECONDS)); //设置超时时间
		searchRequest.source(sourceBuilder);
        try {
        	total = client.search(searchRequest, RequestOptions.DEFAULT).getHits().getTotalHits().value;
        } catch (IOException e) {
        	try { //报IOException时， 重试一次
				total = client.search(searchRequest, RequestOptions.DEFAULT).getHits().getTotalHits().value;
			} catch (IOException e1) {
				e1.printStackTrace();
				throw new BusinessException("根据 参数map 条件，查询总记录数索引失败，MSG："+e1.getMessage());
			}
        }
        return total;
	}

	/**
     * 根据 parammap 条件，查询全部记录
     *
     */
	@Override
	public List<T> selectList(QueryParam queryParam) {
		List<T> list = new ArrayList<>();
		//根据class对象获取该实体对应的表名
		TableInfo tableInfo = TableInfoTool.initTableInfo(getTClass());
		String index = tableInfo.getTableName();
		SearchRequest searchRequest = new SearchRequest(index);
		
		SearchSourceBuilder sourceBuilder = new SearchSourceBuilder();

		//查询参数
		Map<String, Parameter> paramMap = queryParam.getParams();
		BoolQueryBuilder mainBool = QueryBuilders.boolQuery();
		BoolQueryBuilder equalBool = QueryBuilders.boolQuery();
		BoolQueryBuilder likeBool = null;
		for(Map.Entry<String, Parameter> entry: paramMap.entrySet()){
			Parameter param = entry.getValue();
			QueryBuilder queryBuilder = null;
			if(QueryParam.EQ.equals(param.getType())) {
				queryBuilder = QueryBuilders.termQuery(entry.getKey(), param.getValue());
			} else if(QueryParam.LIKE.equals(param.getType())) {
				queryBuilder = QueryBuilders.matchQuery(entry.getKey(), param.getValue());
			} else if(QueryParam.IN.equals(param.getType())) {
				queryBuilder = QueryBuilders.termQuery(entry.getKey(), param.getValue().toString().split(","));
			} else if(QueryParam.BETWEEN.equals(param.getType())) {
				String[] dataArr = param.getValue().toString().split(",");
				queryBuilder = QueryBuilders.rangeQuery(entry.getKey()).from(dataArr[0]).to(dataArr[1]).includeLower(true).includeUpper(true);
			} else if(QueryParam.GT.equals(param.getType())) {
				queryBuilder = QueryBuilders.rangeQuery(entry.getKey()).gt(param.getValue());
			} else if(QueryParam.GE.equals(param.getType())){
				queryBuilder = QueryBuilders.rangeQuery(entry.getKey()).gte(param.getValue());
			} else if(QueryParam.LT.equals(param.getType())) {
				queryBuilder = QueryBuilders.rangeQuery(entry.getKey()).lt(param.getValue());
			} else if(QueryParam.LE.equals(param.getType())) {
				queryBuilder = QueryBuilders.rangeQuery(entry.getKey()).lte(param.getValue());
			}
			if(queryBuilder!=null){
				equalBool.must(queryBuilder);
			}
		}

		//模糊搜索
		if(StringUtils.isNotBlank(queryParam.getSearchText())) {
			likeBool = new BoolQueryBuilder();
			List<String> fields = queryParam.getFuzzyFields();
			for(String field:fields) {
				likeBool.should(QueryBuilders.matchQuery(field, queryParam.getSearchText()));
			}
		}
		if(likeBool!=null){
			mainBool.must(equalBool).must(likeBool);
		}else {
			mainBool.must(equalBool);
		}

		//排序
		Map<String, String> orderMap = queryParam.getOrderMap();
		if(orderMap != null && !orderMap.isEmpty()) {
			for(Map.Entry<String, String> entry : orderMap.entrySet()) {
				sourceBuilder.sort(new FieldSortBuilder(entry.getKey()).order(ASC.equals(entry.getValue().toLowerCase())?SortOrder.ASC:SortOrder.DESC));
			}
		}
		sourceBuilder.query(mainBool);
		sourceBuilder.timeout(new TimeValue(QUERY_TIMEOUT, TimeUnit.SECONDS)); //设置超时时间
        searchRequest.source(sourceBuilder);
        try {
            SearchResponse response = client.search(searchRequest, RequestOptions.DEFAULT);
            SearchHits hits = response.getHits();
            for (SearchHit hit : hits) {
                String sourceAsString = hit.getSourceAsString();
                list.add(JSON.parseObject(sourceAsString, getTClass()));
            }
        } catch (IOException e) {
        	try { //报IOException时， 重试一次
        		SearchResponse response = client.search(searchRequest, RequestOptions.DEFAULT);
                SearchHits hits = response.getHits();
                for (SearchHit hit : hits) {
                    String sourceAsString = hit.getSourceAsString();
                    list.add(JSON.parseObject(sourceAsString, getTClass()));
                }
        	} catch(IOException e1) {
        		e1.printStackTrace();
        		throw new BusinessException("根据 parammap 条件，查询全部记录索引失败，MSG："+e1.getMessage());
        	}
        }
        return list;
	}

	/**
     * 根据 map 条件，查询全部记录（并翻页）
     *
     * @param page         分页查询条件（可以为 RowBounds.DEFAULT）
     * @param queryParam  查询参数
     */
	@Override
	public IPage<T> selectPage(IPage<T> page, QueryParam queryParam) {
		//根据class对象获取该实体对应的表名
		TableInfo tableInfo = TableInfoTool.initTableInfo(getTClass());
		String index = tableInfo.getTableName();
		//查询当前页记录
		SearchRequest searchRequest = new SearchRequest(index);
		SearchSourceBuilder sourceBuilder = new SearchSourceBuilder();
		
		//查询参数
		Map<String, Parameter> paramMap = queryParam.getParams();
		BoolQueryBuilder mainBool = QueryBuilders.boolQuery();
		BoolQueryBuilder equalBool = QueryBuilders.boolQuery();
		BoolQueryBuilder likeBool = null;
		for(Map.Entry<String, Parameter> entry: paramMap.entrySet()){
			Parameter param = entry.getValue();
			QueryBuilder queryBuilder = null;
			if(QueryParam.EQ.equals(param.getType())) {
				queryBuilder = QueryBuilders.termQuery(entry.getKey(), param.getValue());
			} else if(QueryParam.LIKE.equals(param.getType())) {
				queryBuilder = QueryBuilders.matchQuery(entry.getKey(), param.getValue());
			} else if(QueryParam.IN.equals(param.getType())) {
				queryBuilder = QueryBuilders.termQuery(entry.getKey(), param.getValue().toString().split(","));
			} else if(QueryParam.BETWEEN.equals(param.getType())) {
				String[] dataArr = param.getValue().toString().split(",");
				queryBuilder = QueryBuilders.rangeQuery(entry.getKey()).from(dataArr[0]).to(dataArr[1]).includeLower(true).includeUpper(true);
			} else if(QueryParam.GT.equals(param.getType())) {
				queryBuilder = QueryBuilders.rangeQuery(entry.getKey()).gt(param.getValue());
			} else if(QueryParam.GE.equals(param.getType())){
				queryBuilder = QueryBuilders.rangeQuery(entry.getKey()).gte(param.getValue());
			} else if(QueryParam.LT.equals(param.getType())) {
				queryBuilder = QueryBuilders.rangeQuery(entry.getKey()).lt(param.getValue());
			} else if(QueryParam.LE.equals(param.getType())) {
				queryBuilder = QueryBuilders.rangeQuery(entry.getKey()).lte(param.getValue());
			}
			if(queryBuilder!=null){
				equalBool.must(queryBuilder);
			}
		}
		
		//模糊搜索
		if(StringUtils.isNotBlank(queryParam.getSearchText())) {
			likeBool = new BoolQueryBuilder();
			List<String> fields = queryParam.getFuzzyFields();
			for(String field:fields) {
				likeBool.should(QueryBuilders.matchQuery(field, queryParam.getSearchText()));
			}
		}
		if(likeBool!=null){
			mainBool.must(equalBool).must(likeBool);
		}else {
			mainBool.must(equalBool);
		}
		
		//排序
		Map<String, String> orderMap = queryParam.getOrderMap();
		if(orderMap != null && !orderMap.isEmpty()) {
			for(Map.Entry<String, String> entry : orderMap.entrySet()) {
				sourceBuilder.sort(new FieldSortBuilder(entry.getKey()).order(ASC.equals(entry.getValue().toLowerCase())?SortOrder.ASC:SortOrder.DESC));
			}
		}
		sourceBuilder.query(mainBool);
		sourceBuilder.from(queryParam.getPageIndex() <= 0 ? 0 : (queryParam.getPageIndex()-1)*queryParam.getPageSize());
        sourceBuilder.size(queryParam.getPageSize());
        sourceBuilder.trackTotalHits(true);
        sourceBuilder.timeout(new TimeValue(QUERY_TIMEOUT, TimeUnit.SECONDS)); //设置超时时间
        searchRequest.source(sourceBuilder);
        try {
        	List<T> list = new ArrayList<>();
            SearchResponse response = client.search(searchRequest, RequestOptions.DEFAULT);
            SearchHits hits = response.getHits();
            for (SearchHit hit : hits) {
                String sourceAsString = hit.getSourceAsString();
                list.add(JSON.parseObject(sourceAsString, getTClass()));
            }
            page.setRecords(list);
            page.setTotal(hits.getTotalHits().value);
        } catch (IOException e) {
        	try { //报IOException时， 重试一次
        		List<T> list = new ArrayList<>();
                SearchResponse response = client.search(searchRequest, RequestOptions.DEFAULT);
                SearchHits hits = response.getHits();
                for (SearchHit hit : hits) {
                    String sourceAsString = hit.getSourceAsString();
                    list.add(JSON.parseObject(sourceAsString, getTClass()));
                }
                page.setRecords(list);
                page.setTotal(hits.getTotalHits().value);
        	} catch (IOException e1) {
        		e1.printStackTrace();
        		throw new BusinessException("根据 parammap 条件，查询全部记录索引失败，MSG："+e1.getMessage());
        	}
        }
        return page;
	}
}
