package com.ejianc.business.store.service;

import com.ejianc.business.store.consts.InOutTypeEnum;
import com.ejianc.business.store.service.handler.IStoreManageHandler;
import com.ejianc.business.store.service.handler.StoreManageFactory;
import com.ejianc.business.store.vo.StoreManageVO;
import com.ejianc.business.store.vo.SurplusUpdateVO;
import com.ejianc.business.store.vo.UseCalculateVO;
import com.ejianc.framework.core.response.CommonResponse;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.List;

/**
 * @author songlx
 * @version 1.0
 * @description: 仓库管理服务类
 * @date 2022/1/25
 */
@Service
public class StoreManageService {

    @Autowired
    private StoreManageFactory storeManageFactory;

    @Autowired
    ISurplusService surplusService;

    @Autowired
    IInOutService inOutService;

    @Autowired
    IFlowService flowService;

    /**
     * @param storeManageVO
     * @description: 出入库业务处理
     * @return: com.ejianc.framework.core.response.CommonResponse<com.ejianc.business.store.vo.StoreManageVO>
     * @author songlx
     * @date: 2022/1/25
     */
    public CommonResponse<StoreManageVO> inOutStore(StoreManageVO storeManageVO) {
        CommonResponse<StoreManageVO> validateRes = this.validateParam(storeManageVO, false);
        if (!validateRes.isSuccess()) {
            return validateRes;
        }
        InOutTypeEnum inOutTypeEnum = storeManageVO.getInOutTypeEnum();
        IStoreManageHandler storeManageHandler = storeManageFactory.getHandler(inOutTypeEnum);
        CommonResponse<StoreManageVO> handle = storeManageHandler.handle(storeManageVO);
        return handle;
    }

    /**
     * @param storeManageVO
     * @description: 出入库业务逆向处理
     * @return: com.ejianc.framework.core.response.CommonResponse<com.ejianc.business.store.vo.StoreManageVO>
     * @author songlx
     * @date: 2022/1/25
     */
    public CommonResponse<StoreManageVO> inOutStoreRollback(StoreManageVO storeManageVO) {
        CommonResponse<StoreManageVO> validateRes = this.validateParam(storeManageVO, true);
        if (!validateRes.isSuccess()) {
            return validateRes;
        }
        InOutTypeEnum inOutTypeEnum = storeManageVO.getInOutTypeEnum();
        IStoreManageHandler storeManageHandler = storeManageFactory.getHandler(inOutTypeEnum);
        CommonResponse<StoreManageVO> handle = storeManageHandler.handleRollback(storeManageVO);
        return handle;
    }

    /**
     * @param storeManageVO
     * @description: 查询物资库占用存现存量库存均价
     * @return: com.ejianc.framework.core.response.CommonResponse<com.ejianc.business.store.vo.StoreManageVO>
     * @author songlx
     * @date: 2022/2/9
     */
    public CommonResponse<StoreManageVO> surplusMaterialPrice(StoreManageVO storeManageVO) {
        CommonResponse<SurplusUpdateVO> response = surplusService.validateStoreSurplusByOutLock(storeManageVO);
        if (!response.isSuccess()) {
            return CommonResponse.error(response.getMsg());
        }
        UseCalculateVO useCalculateVO = surplusService.useCalculate(storeManageVO);
        storeManageVO.setUseMaterialPriceVOList(useCalculateVO.getUseMaterialPriceVOList());
        return CommonResponse.success(storeManageVO);
    }


    /**
     * @param storeManageVO
     * @description: 周转材更新摊销
     * @return: com.ejianc.framework.core.response.CommonResponse<com.ejianc.business.store.vo.StoreManageVO>
     * @author songlx
     * @date: 2022/3/2
     */
    public CommonResponse<StoreManageVO> updateShareMny(StoreManageVO storeManageVO) {
        storeManageVO = flowService.updateShareMny(storeManageVO);
        return CommonResponse.success(storeManageVO);
    }


    /**
     * @param storeManageVO
     * @description: 公共的入参校验
     * @return: com.ejianc.framework.core.response.CommonResponse<com.ejianc.business.store.vo.StoreManageVO>
     * @author songlx
     * @date: 2022/1/26
     */
    private CommonResponse<StoreManageVO> validateParam(StoreManageVO storeManageVO, boolean isRollBack) {
        Long storeId = storeManageVO.getStoreId();
        Long sourceId = storeManageVO.getSourceId();
        InOutTypeEnum inOutTypeEnum = storeManageVO.getInOutTypeEnum();
        List<Long> sourceIdsForRollBack = storeManageVO.getSourceIdsForRollBack();

        if (storeId == null) {
            return CommonResponse.error("仓库ID不能为空!");
        }
        if (CollectionUtils.isEmpty(sourceIdsForRollBack) && sourceId == null) {
            return CommonResponse.error("业务单据主表ID不能为空!");
        }
        if (inOutTypeEnum == null) {
            return CommonResponse.error("出入库枚举类型不能为空!");
        }
     /*   if (isRollBack) {
            if (CollectionUtils.isEmpty(sourceIdsForRollBack)) {
                return CommonResponse.error("参数sourceIdsForRollBack不能为空!");
            }
        } else {
            if (!storeManageVO.getOutEffectiveON() && CollectionUtils.isEmpty(storeManageVO.getFlowVOList())) {
                return CommonResponse.error("出入库明细不能为空!");
            }
        }*/

        return CommonResponse.success();
    }

    /**
     * @param storeManageVO
     * @description: 周转材库存现存量库存均价
     * @return: com.ejianc.framework.core.response.CommonResponse<com.ejianc.business.store.vo.StoreManageVO>
     * @author songlx
     * @date: 2022/2/9
     */
    public CommonResponse<StoreManageVO> turnSurplusMaterialPrice(StoreManageVO storeManageVO) {
        CommonResponse<SurplusUpdateVO> response = surplusService.validateStoreSurplusByOutLock(storeManageVO);
        if (!response.isSuccess()) {
            return CommonResponse.error(response.getMsg());
        }
        UseCalculateVO useCalculateVO = surplusService.useTurnCalculate(storeManageVO);
        storeManageVO.setUseMaterialPriceVOList(useCalculateVO.getUseMaterialPriceVOList());
        return CommonResponse.success(storeManageVO);
    }


}
