package com.ejianc.business.store.service.handler;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.ejianc.business.store.bean.FlowEntity;
import com.ejianc.business.store.consts.InOutTypeEnum;
import com.ejianc.business.store.service.IFlowService;
import com.ejianc.business.store.service.IStoreService;
import com.ejianc.business.store.service.ISurplusService;
import com.ejianc.business.store.util.StoreLockUtil;
import com.ejianc.business.store.util.StoreManageUtil;
import com.ejianc.business.store.vo.FlowVO;
import com.ejianc.business.store.vo.StoreManageVO;
import com.ejianc.business.store.vo.SurplusUpdateVO;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @author songlx
 * @version 1.0
 * @description: 仓库入库
 * @date 2022/1/25
 */
@Service
public class InStoreHandler implements IStoreManageHandler {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    IFlowService flowService;

    @Autowired
    ISurplusService surplusService;
    @Autowired
    IStoreService storeService;


    /**
     * @param storeManageVO
     * @description: 仓库入库具体实现逻辑, 收料入库审批通过后调用
     * 正向入库
     * 1 直接保存入库流水
     * 2 更新库存 无则插入,有则增量更新
     * @return: com.ejianc.framework.core.response.CommonResponse<com.ejianc.business.store.vo.StoreManageVO>
     * @author songlx
     * @date: 2022/1/25
     */
    @Override
    public CommonResponse<StoreManageVO> handle(StoreManageVO storeManageVO) {
        //设置仓库属性
        storeManageVO = storeService.setStoreAttr(storeManageVO);
        Long storeId = storeManageVO.getStoreId();
        Long sourceId = storeManageVO.getSourceId();
        if (sourceId != null) {
            QueryWrapper<FlowEntity> queryWrapper = new QueryWrapper<>();
            queryWrapper.eq("source_id", sourceId);
            int count = flowService.count(queryWrapper);
            if (count > 0) {
                InOutTypeEnum inOutTypeEnum = storeManageVO.getInOutTypeEnum();
                String billTypeName = inOutTypeEnum != null ? inOutTypeEnum.getInOutTypeName() : "无";
                logger.error("---start---------重复校验ERROR---------------------------------");
                logger.error("仓库入库流水重复校验：已存在该单据入库流水！来源单据类型：{}，单据ID：{}", billTypeName, sourceId);
                logger.error("---end-----------重复校验ERROR---------------------------------");
                return CommonResponse.success("仓库入库流水重复校验：已存在该单据入库流水！来源单据ID：" + sourceId, storeManageVO);
            }
        }

        List<FlowVO> flowVOList = storeManageVO.getFlowVOList();
        SurplusUpdateVO surplusUpdateVO = StoreManageUtil.getSurplusUpdateVO(storeId, flowVOList, false);
        /*Boolean lock = StoreLockUtil.getLock(storeId);
        if (!lock) {
            throw new BusinessException("获取仓库锁网络异常，请刷新重试！");
        }*/

        try {
            // 入库初始化
            flowVOList.forEach(flowVO -> {
                flowVO.setOutNum(BigDecimal.ZERO);
                flowVO.setOutLockNum(BigDecimal.ZERO);
                flowVO.setSurplusNum(flowVO.getNum());
                flowVO.setSurplusMny(flowVO.getMny());
                flowVO.setSurplusTaxMny(flowVO.getTaxMny());
                flowVO.setEstimatePrice(flowVO.getPrice());
                flowVO.setEstimateTaxPrice(flowVO.getTaxPrice());
            });

            List<FlowEntity> flowEntities = BeanMapper.mapList(flowVOList, FlowEntity.class);
            boolean b = flowService.saveOrUpdateBatch(flowEntities);
            //入库成功后更新库存
            if (b) {
                SurplusUpdateVO surplusUpdateVORet = surplusService.updateStoreSurplus(surplusUpdateVO, true, true);
                //更新后将这些物资最新库存返回
                storeManageVO.setSurplusVOList(surplusUpdateVORet.getSurplusVOList());
            }
        } catch (Exception e) {
            e.printStackTrace();
            throw new BusinessException("操作异常，请刷新重试！");
        }
        /*finally {
            if (lock) {
                StoreLockUtil.releaseLock(storeId);
            }
        }*/
        return CommonResponse.success(storeManageVO);
    }

    /**
     * @param storeManageVO
     * @description: 仓库入库回滚实现逻辑, 收料入库弃审前调用
     * 1 校验入库是否被使用
     * 2 若使用,不能回滚
     * 若未使用则删除入库并逆向更新库存
     * @return: com.ejianc.framework.core.response.CommonResponse<com.ejianc.business.store.vo.StoreManageVO>
     * @author songlx
     * @date: 2022/1/25
     */
    @Override
    public CommonResponse<StoreManageVO> handleRollback(StoreManageVO storeManageVO) {
        Long storeId = storeManageVO.getStoreId();
        Long sourceId = storeManageVO.getSourceId();
        //校验入库单是否被使用，结算或者对账
        CommonResponse<List<FlowEntity>> res = flowService.validateIsUse(sourceId, storeId);
        if (!res.isSuccess()) {
            return CommonResponse.error(res.getMsg());
        }
        List<FlowEntity> instoreList = res.getData();
        List<FlowVO> flowVOList = BeanMapper.mapList(instoreList, FlowVO.class);
        SurplusUpdateVO surplusUpdateVO = StoreManageUtil.getSurplusUpdateVO(storeId, flowVOList, true);

        /*Boolean lock = StoreLockUtil.getLock(storeId);
        if (!lock) {
            throw new BusinessException("获取仓库锁网络异常，请刷新重试！");
        }*/
        try {
            SurplusUpdateVO surplusUpdateVORet = surplusService.updateStoreSurplus(surplusUpdateVO, true, true);
            //更新后将这些物资最新库存返回
            storeManageVO.setSurplusVOList(surplusUpdateVORet.getSurplusVOList());
            //删除入库流水
            List<Long> delListIds = flowVOList.stream().map(FlowVO::getId).collect(Collectors.toList());
            flowService.removeByIds(delListIds);
            return CommonResponse.success(storeManageVO);

        } catch (Exception e) {
            e.printStackTrace();
            throw new BusinessException("操作异常，请刷新重试！");
        } /*finally {
            if (lock) {
                StoreLockUtil.releaseLock(storeId);
            }
        }*/
    }
}
