package com.ejianc.foundation.billcode;

import java.io.IOException;
import java.util.HashMap;
import java.util.List;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.configuration.ConfigurationException;
import org.apache.commons.configuration.XMLConfiguration;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.io.DefaultResourceLoader;
import org.springframework.core.io.ResourceLoader;
import org.springframework.stereotype.Component;

import com.ejianc.foundation.billcode.elemproc.itf.IElemProcessor;
import com.ejianc.foundation.billcode.randomcode.IRandomCodeGenerator;
import com.ejianc.foundation.billcode.randomcode.impl.RandomCodeGenerator;
import com.ejianc.foundation.billcode.sngenerator.ISNGenerator;
import com.ejianc.foundation.billcode.sngenerator.impl.PureDigitalSNGenerator;
import com.ejianc.foundation.billcode.sysdate.ISysDateProvider;
import com.ejianc.foundation.billcode.sysdate.impl.DateProviderJavaDate;

@Component
public class BillCodeEngineContext {

    private static Logger logger = LoggerFactory.getLogger(BillCodeEngineContext.class);
    private static BillCodeEngineContext context = null;
    private static ISysDateProvider sysdateClss;
    private static HashMap<String, ISNGenerator> snGenerators;
    private static HashMap<Integer, IElemProcessor> processors;
    private static IRandomCodeGenerator rdmCodeClass;

    public static BillCodeEngineContext getInstance() {
        if(null == context) {
            context = getInstance("classpath:/billCodeEngineContext.xml");
        }

        return context;
    }

    public static BillCodeEngineContext getInstance(String xmlQualifiedPath) {
        if(null == context) {
            if(StringUtils.isNotBlank(xmlQualifiedPath)) {
                try {
                    ResourceLoader resourceLoader = new DefaultResourceLoader();

                    XMLConfiguration e = new XMLConfiguration();
                    e.load(resourceLoader.getResource(xmlQualifiedPath).getInputStream());
                    String sysDateQualifiedName = e.getString("sysdate");
                    List<Object> snGeneratorQualifiedNameLsit = e.getList("snGenerators.generator");
                    List<Object> elemProcQualifiedNameList = e.getList("elemProcessors.processor");
                    String rdmCodeQualifiedName = e.getString("randomCode");
                    initSysDateImp(sysDateQualifiedName);
                    initSNGnneratorMap(snGeneratorQualifiedNameLsit);
                    initElemProcesMap(elemProcQualifiedNameList);
                    intRdmcodeclass(rdmCodeQualifiedName);
                    context = new BillCodeEngineContext();
                } catch (ConfigurationException e1) {
                    logger.error("初始化单据编码生成器引擎异常，", e1);
                } catch (IOException e2) {
                    logger.error("初始化单据编码生成器引擎异常，", e2);
                }
            } else {
                context = getInstance();
            }
        }

        return context;
    }

    private static void intRdmcodeclass(String rdmCodeQualifiedName) {
        Object obj = getInstanceByQualifiedName(rdmCodeQualifiedName);
        if(null != obj && obj instanceof ISysDateProvider) {
            rdmCodeClass = (IRandomCodeGenerator) obj;
        } else {
            rdmCodeClass = new RandomCodeGenerator();
        }
    }

    private static void initElemProcesMap(List<Object> elemProcQualifiedNameList) {
        processors = new HashMap<>();
        if(CollectionUtils.isNotEmpty(elemProcQualifiedNameList)) {
            for(Object name : elemProcQualifiedNameList) {
                IElemProcessor processor = (IElemProcessor) getInstanceByQualifiedName(name.toString());
                processors.put(Integer.valueOf(processor.getCanProcElemType()), processor);
            }
        }
    }

    private static void initSNGnneratorMap(List<Object> snGeneratorQualifiedNameLsit) {
        snGenerators = new HashMap<>();
        if(CollectionUtils.isNotEmpty(snGeneratorQualifiedNameLsit)) {
            for(Object name : snGeneratorQualifiedNameLsit) {
                ISNGenerator generator = (ISNGenerator) getInstanceByQualifiedName(name.toString());
                snGenerators.put(generator.getSNGeneratorType(), generator);
            }
        }
    }

    private static void initSysDateImp(String sysDateQualifiedName) {
        Object obj = getInstanceByQualifiedName(sysDateQualifiedName);
        if(null != obj && obj instanceof ISysDateProvider) {
            sysdateClss = (ISysDateProvider) obj;
        } else {
            sysdateClss = new DateProviderJavaDate();
        }
    }

    private static Object getInstanceByQualifiedName(String sysDateQualifiedName) {
        Object instance = null;
        if(StringUtils.isNotBlank(sysDateQualifiedName)) {
            String className = sysDateQualifiedName.trim();

            try {
                instance = Class.forName(className).newInstance();
            } catch (IllegalAccessException e) {
                e.printStackTrace();
            } catch (InstantiationException e) {
                e.printStackTrace();
            } catch (ClassNotFoundException e) {
                e.printStackTrace();
            }
        }

        return instance;
    }

    public IElemProcessor getElemProcessor(String elemType) {
        return (IElemProcessor) processors.get(Integer.valueOf(elemType));
    }

    public ISNGenerator getSNGeneratorImpl(String elemValue) {
        return (ISNGenerator) (snGenerators.get(elemValue) != null
                ? (ISNGenerator) snGenerators.get(elemValue) : new PureDigitalSNGenerator());
    }

    public ISysDateProvider getSysDateImp() {
        return sysdateClss;
    }

    public IRandomCodeGenerator getRdmCodeClass() {
        return rdmCodeClass;
    }
}
