package com.ejianc.foundation.support.controller.api;

import com.alibaba.fastjson.JSON;
import com.ejianc.foundation.support.bean.ModuleEntity;
import com.ejianc.foundation.support.service.IModuleService;
import com.ejianc.foundation.support.vo.ModuleVO;
import com.ejianc.framework.cache.redis.CacheManager;
import com.ejianc.framework.cache.serializer.Serializer;
import com.ejianc.framework.cache.serializer.impl.DefaultJDKSerializer;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.skeleton.template.BaseVO;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.*;

import java.nio.charset.Charset;
import java.util.List;

@RestController
@RequestMapping("/api/module/")
public class ModuleApi {
	private Logger logger = LoggerFactory.getLogger(this.getClass());

	// 所有模块缓存
	private final static String MODEL_CACHE_KEY = "model_cache_key:";

	@Autowired
	private CacheManager cacheManager;

	@Autowired
	private IModuleService moduleService;

	public static final String DEFAULT_CHARSET = "UTF-8";

	@Value("${oms.tenantid}")
	private Long omsTenantId;

	private Serializer serializer = new DefaultJDKSerializer();

	@RequestMapping(value = "getListByPid", method=RequestMethod.GET)
	@ResponseBody
	public CommonResponse<List<ModuleVO>> queryModuleListByPid(@RequestParam(value="moduleId", required=true) Long moduleId) {
		try {
			List<ModuleVO> moduleList = moduleService.queryAllChildrenByPid(moduleId);
			return CommonResponse.success(moduleList);
		} catch (Exception e) {
			logger.error("系统异常："+e.getMessage());
			e.printStackTrace();
		}
		return CommonResponse.error("查询失败");
	}

	/**
	 * @Author mrsir_wxp
	 * @Date 2020/9/15 根据模块id查询模块信息
	 * @Description getModuleById
	 * @Param [moduleId]
	 * @Return com.ejianc.framework.core.response.CommonResponse<com.ejianc.foundation.support.vo.ModuleVO>
	 */
	@RequestMapping(value = "getModuleById", method=RequestMethod.GET)
	@ResponseBody
	public CommonResponse<ModuleVO> getModuleById(@RequestParam(value="moduleId", required=true) Long moduleId) {
		try {
			ModuleEntity moduleEntity = moduleService.selectById(moduleId);
			ModuleVO moduleVO = BeanMapper.map(moduleEntity, ModuleVO.class);
			return CommonResponse.success(moduleVO);
		} catch (Exception e) {
			logger.error("系统异常："+e.getMessage());
			e.printStackTrace();
		}
		return CommonResponse.error("查询失败");
	}

	@PostMapping(value = "refreshModuleCache")
	public CommonResponse<String> refreshModuleCache() {
		try {
			QueryParam queryParam = new QueryParam();
			queryParam.getParams().put("tenant_id", new Parameter(QueryParam.EQ, omsTenantId));
			queryParam.getParams().put("dr", new Parameter(QueryParam.EQ, BaseVO.DR_UNDELETE));
			queryParam.getOrderMap().put("sequence", "asc");
			List<ModuleEntity> moduleList = moduleService.queryList(queryParam, false);

			if(!moduleList.isEmpty()) {
				cacheManager.piplineExecute(pipeline -> {
					String key = MODEL_CACHE_KEY + omsTenantId;
					pipeline.multi();
					for(ModuleEntity m : moduleList) {
						byte[] keyBytes = key.getBytes(Charset.forName(DEFAULT_CHARSET));
						byte[] fieldBytes = m.getId().toString().getBytes(Charset.forName(DEFAULT_CHARSET));
						byte[] contentBytes = serializer.marshalToByte(JSON.toJSONString(m));
						pipeline.hset(keyBytes, fieldBytes, contentBytes);
					}
					pipeline.exec();
				});
			}
			return CommonResponse.success("模块缓存刷新成功！");
		} catch (Exception e) {
			logger.error("系统异常：", e);
		}
		return CommonResponse.error("模块缓存刷新失败");
	}
}
