package com.ejianc.foundation.billcode.lock;

import com.ejianc.foundation.billcode.BillCodeException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;
import redis.clients.jedis.Transaction;
import redis.clients.jedis.exceptions.JedisException;

import java.util.List;
import java.util.UUID;

@Component
public class RedisLock implements IBillCodeEngineLock {

    private Logger logger = LoggerFactory.getLogger(RedisLock.class);

    private final static ThreadLocal<String> idStore = new ThreadLocal<String>();

    private final String LOCK_KEY_PREFIX = "lock_ejc_billcode:";
    private final String LOCK_KEY_SEPERATOR = "_";

    @Autowired
    private JedisPool jedisPool;

    @Override
    public int getEngineLockType() {
        return IBillCodeEngineLock.ENGINELOCK_MEMOEYLOCK;
    }

    @Override
    public boolean lock(String pk_bilcodebase, String billCodeSNRefer, Long tenantId) throws BillCodeException {
        String lockKey = LOCK_KEY_PREFIX + pk_bilcodebase + LOCK_KEY_SEPERATOR + billCodeSNRefer.replaceAll("\\^", "") + LOCK_KEY_SEPERATOR + tenantId ;
        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();
            // 记录结束时间
            long end1 = System.currentTimeMillis();
            long acquireTimeout = 100000;
            long timeout = 500;
            // 获取锁的超时时间，超过这个时间则放弃获取锁
            long end = System.currentTimeMillis() + acquireTimeout;
            // 超时时间，上锁后超过此时间则自动释放锁

            while (System.currentTimeMillis() < end) {
                String identifier = UUID.randomUUID().toString();
                String result = jedis.set(lockKey, identifier, "NX", "PX", timeout);
                if ("OK".equals(result)) {
                    idStore.set(identifier);
                    logger.info(Thread.currentThread().getId() + "RedisLock获取RedisLock--"+System.currentTimeMillis()+"-----" + (System.currentTimeMillis() - end1));
                    return true;
                }
                try {
                    Thread.sleep(20);
                } catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                }
            }

        } catch (JedisException e) {
            throw new JedisException(e);
        } finally {
            if (jedis != null) {
                jedis.close();
            }
        }
        return false;

    }

    @Override
    public boolean unlock(String pk_bilcodebase, String billCodeSNRefer, Long tenantId) throws BillCodeException {
    	String lockKey = LOCK_KEY_PREFIX + pk_bilcodebase + LOCK_KEY_SEPERATOR + billCodeSNRefer.replaceAll("\\^", "") + LOCK_KEY_SEPERATOR + tenantId ;
        Jedis jedis = null;
        boolean retFlag = false;
        try {
            jedis = jedisPool.getResource();
            while (true) {
                // 监视lock，准备开始事务
                jedis.watch(lockKey);
                // 通过前面返回的value值判断是不是该锁，若是该锁，则删除，释放锁
                if (idStore.get().equals(jedis.get(lockKey))) {
                    Transaction transaction = jedis.multi();
                    transaction.del(lockKey);
                    List<Object> results = transaction.exec();
                    if (results == null) {
                        continue;
                    }
                    retFlag = true;
                }
                jedis.unwatch();
                break;
            }
        } catch (JedisException e) {
            throw new JedisException(e);
        } finally {
            if (jedis != null) {
                jedis.close();
                idStore.remove();
            }
        }
        return retFlag;
    }
}
