package com.ejianc.foundation.support.service.impl;

import java.net.URL;
import java.nio.file.Paths;
import java.time.Duration;

import com.alibaba.druid.util.StringUtils;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.aliyun.farui20240628.Client;
import com.aliyun.farui20240628.models.CreateTextFileAdvanceRequest;
import com.aliyun.farui20240628.models.CreateTextFileResponse;
import com.aliyun.teaopenapi.models.Config;
import com.aliyun.teautil.models.RuntimeOptions;

import com.aliyun.auth.credentials.Credential;
import com.aliyun.auth.credentials.provider.StaticCredentialProvider;
import com.aliyun.sdk.gateway.pop.Configuration;
import com.aliyun.sdk.gateway.pop.auth.SignatureVersion;
import com.aliyun.sdk.service.farui20240628.AsyncClient;
import com.aliyun.sdk.service.farui20240628.models.RunContractRuleGenerationRequest;
import com.aliyun.sdk.service.farui20240628.models.RunContractRuleGenerationResponseBody;
import com.aliyun.sdk.service.farui20240628.models.RunContractResultGenerationRequest;
import com.aliyun.sdk.service.farui20240628.models.RunContractResultGenerationResponseBody;
import darabonba.core.RequestConfiguration;
import darabonba.core.ResponseIterable;
import darabonba.core.ResponseIterator;
import darabonba.core.client.ClientOverrideConfiguration;

import com.ejianc.foundation.support.service.IFaruiService;
import com.ejianc.framework.core.response.CommonResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;


@Service
public class FaruiServiceImpl implements IFaruiService {

	private final Logger logger = LoggerFactory.getLogger(getClass());

	@Value("${farui.accessKeyId}")
	private String accessKeyId ;
	@Value("${farui.accessKeySecret}")
	private String accessKeySecret ;
	@Value("${farui.endpoint}")
	private String endpoint ;
	@Value("${farui.workspaceId}")
	private String workspaceId ;

	/**
	 * 合同审查-上传文档
	 * 该接口用于将文档上传到合同审查模块中，添加成功之后，
	 * 系统会自动启动文件的解析，并返回对应的文件id用于后续生成审查规则和审查结果。
	 * 文件解析有排队机制，如果队列较长，文件可能需要等待一段时间才能解析完成
	 * see  <a href="https://tyfarui.biz.aliyun.com/help/openApiJavaExample.url">...</a>
	 *
	 * @param filePath
	 */
	@Override
	public CommonResponse<String> createTextFile(String filePath) {
		logger.info("filePath :{} accessKeyId:{}  workspaceId:{}  endpoint:{}",filePath,accessKeyId,workspaceId,endpoint);
		Config config = new Config()
				.setAccessKeyId(accessKeyId)
				.setAccessKeySecret(accessKeySecret)
				.setEndpoint(endpoint)
				.setProtocol("HTTPS");

        Client client = null;
        try {
            client = new Client(config);
			CreateTextFileAdvanceRequest createFileAdvanceRequest = new CreateTextFileAdvanceRequest();
			URL url = new URL(filePath);
			// 获取文件名，假设URL的路径部分的最后一部分是文件名
			String fileName = Paths.get(url.getPath()).getFileName().toString();
			createFileAdvanceRequest.setTextFileName(fileName).setTextFileUrlObject(url.openStream());
			RuntimeOptions runtime = new RuntimeOptions();
			Map<String, String> headers = new HashMap<>();
			CreateTextFileResponse createFileResponse = client.createTextFileAdvance(workspaceId, createFileAdvanceRequest, headers, runtime);
			if (createFileResponse.getStatusCode().equals(200)) {
				logger.info("上传返回数据："+JSONObject.toJSONString(createFileResponse.getBody().getData()));
				if(createFileResponse.getBody().getData().getTextFileId()!=null){
					return CommonResponse.success("推送成功！",createFileResponse.getBody().getData().getTextFileId());
				}
			}else {
				logger.info("上传失败："+JSONObject.toJSONString(createFileResponse));
			}
        } catch (Exception e) {
			logger.error("上传失败："+e.getMessage(),e);
        }
		return CommonResponse.error("上传失败！");
	}

	/**
	 * 合同审查-生成规则
	 * 该接口用于合同审查模块的智能规则生成，调用大模型返回合同的审查规则和对应风险。传入合同审查文件的ID、审查立场，会通过sse的方式增量式返回模型生成的审查规则
	 * see  <a href="https://tyfarui.biz.aliyun.com/help/openApiJavaExample.url">...</a>
	 *
	 * @param fileId
	 */
	@Override
	public CommonResponse<JSONObject> generationContractRule(String fileId, String position) {
		logger.info("合同审查-生成规则参数：fileId :{} position:{}  accessKeyId:{} workspaceId:{}  endpoint:{}",fileId,position,accessKeyId,workspaceId,endpoint);
		try {
			StaticCredentialProvider provider = StaticCredentialProvider.create(
					Credential.builder() .accessKeyId(accessKeyId) .accessKeySecret(accessKeySecret) .build()
			);
			AsyncClient clientAs = AsyncClient.builder()// Region ID
					.credentialsProvider(provider).serviceConfiguration(Configuration.create().setSignatureVersion(SignatureVersion.V3))
					.overrideConfiguration(ClientOverrideConfiguration.create().setProtocol("HTTPS").setEndpointOverride(endpoint)).build();

			RunContractRuleGenerationRequest.MetaData metaData = RunContractRuleGenerationRequest.MetaData.builder().fileId(fileId).position(position).build();

			RunContractRuleGenerationRequest.Assistant ruleAssistant = RunContractRuleGenerationRequest.Assistant.builder()
					.type("contract_examime").version("1").metaData(metaData).build();

			RequestConfiguration requestConfiguration = RequestConfiguration.create()
					.setConnectTimeout(Duration.ofSeconds(3*60))
					.setResponseTimeout(Duration.ofSeconds(3*60));

			RunContractRuleGenerationRequest runContractRuleGenerationRequest = RunContractRuleGenerationRequest.builder()
					.appId("farui").stream(true).workspaceId(workspaceId).requestConfiguration(requestConfiguration).assistant(ruleAssistant).build();

			ResponseIterable<RunContractRuleGenerationResponseBody> x = clientAs.runContractRuleGenerationWithResponseIterable(runContractRuleGenerationRequest);

			ResponseIterator<RunContractRuleGenerationResponseBody> iterator = x.iterator();
			JSONObject jsonObject = new JSONObject();
			JSONArray rules = new JSONArray();
			while (iterator.hasNext()) {
				logger.info("--generationContractRule--event----");
				RunContractRuleGenerationResponseBody event = iterator.next();
				if(event.getSuccess()){
					logger.info("event.getOutput().getRules()=="+JSONObject.toJSONString(event.getOutput()));
					if (event.getOutput().getRuleTaskId()!=null) {
						jsonObject.put("ruleTaskId",event.getOutput().getRuleTaskId());
					}
					if (event.getOutput().getRules()!=null && !event.getOutput().getRules().isEmpty()) {
						JSONObject role = new JSONObject();
						role.put("ruleSequence",event.getOutput().getRules().get(0).getRuleSequence());
						role.put("ruleTag",event.getOutput().getRules().get(0).getRuleTag());
						role.put("ruleTitle",event.getOutput().getRules().get(0).getRuleTitle());
						role.put("riskLevel",event.getOutput().getRules().get(0).getRiskLevel());
						rules.add(role);
					}
				}else {
					logger.info("生成规则失败！event.getOutput().getRules()=="+JSONObject.toJSONString(event));
				}
			}
			if(!rules.isEmpty()){
				jsonObject.put("rules",rules);
				return CommonResponse.success(jsonObject);
			}
		} catch (Exception e) {
			return CommonResponse.error("生成规则失败！"+e.getMessage());
		}
		return CommonResponse.error("生成规则失败！");
	}

	/**
	 * 合同审查-生成结果
	 * 该接口用于合同审查模块的审查结果生成，调用大模型返回合同的审查结果、审查风险项、建议修改内容等。传入合同审查文件的ID、审查立场、审查规则，
	 * 会通过sse的方式增量式返回模型生成的审查结果
	 * see  <a href="https://tyfarui.biz.aliyun.com/help/openApiJavaExample.url">...</a>
	 *
	 * @param fileId
	 * @param ruleTasks
	 */
	@Override
	public CommonResponse<JSONObject> generationContractResult( String fileId, String position, JSONObject ruleTasks) {
		if(ruleTasks==null){
			return CommonResponse.error("生成结果失败！规则列表为空！");
		}
		try {
			String ruleTaskId = ruleTasks.getString("ruleTaskId");
			if(StringUtils.isEmpty(ruleTaskId)){
				return CommonResponse.error("生成结果失败！规则任务ID为空！");
			}
			StaticCredentialProvider provider = StaticCredentialProvider.create(
					Credential.builder()
							.accessKeyId(accessKeyId)
							.accessKeySecret(accessKeySecret)
							.build()
			);
			AsyncClient client = AsyncClient.builder()// Region ID
					.credentialsProvider(provider)
					.serviceConfiguration(Configuration.create()
							.setSignatureVersion(SignatureVersion.V3)
					)
					.overrideConfiguration(
							ClientOverrideConfiguration.create()
									.setProtocol("HTTPS")
									.setEndpointOverride(endpoint)
					)
					.build();

			//这里是 runContractRuleGeneration生成的规则，规则内容需要和生成时保持一致
			//规则也可以通过 RunContractResultGenerationRequest.MetaData.CustomRuleConfig 传入自定义规则
			JSONArray ruleArr = ruleTasks.getJSONArray("rules");
			if(ruleArr==null || ruleArr.isEmpty()){
				return CommonResponse.error("生成结果失败！规则列表为空！");
			}
			List<RunContractResultGenerationRequest.Rules> rules = new ArrayList<>();
			for (Object o : ruleArr) {
				JSONObject ruleObject = (JSONObject) o;
				RunContractResultGenerationRequest.Rules rule = RunContractResultGenerationRequest.Rules.builder()
						.ruleSequence(ruleObject.getString("ruleSequence"))
						.ruleTag(ruleObject.getString("ruleTag"))
						.ruleTitle(ruleObject.getString("ruleTitle"))
						.riskLevel(ruleObject.getString("riskLevel"))
						.build();
				rules.add(rule);
			}

			RunContractResultGenerationRequest.MetaData metaData = RunContractResultGenerationRequest.MetaData.builder()
					.fileId(fileId)
					.position(position)
					.ruleTaskId(ruleTaskId)
					.rules(rules)
					.build();

			RunContractResultGenerationRequest.Assistant assistant = RunContractResultGenerationRequest.Assistant.builder()
					.type("contract_examime")
					.metaData(metaData)
					.version("1")
					.build();

			RunContractResultGenerationRequest request = RunContractResultGenerationRequest.builder()
					.appId("farui")
					.stream(true)
					.workspaceId(workspaceId)
					.assistant(assistant)
					.build();

			ResponseIterable<RunContractResultGenerationResponseBody> x = client.runContractResultGenerationWithResponseIterable(request);

			ResponseIterator<RunContractResultGenerationResponseBody> iterator = x.iterator();
			while (iterator.hasNext()) {
				logger.info("----event----");
				RunContractResultGenerationResponseBody event = iterator.next();
				if (event.getSuccess()) {

					logger.info(event.toMap().toString());
				}
			}
		}catch (Exception e){
			throw new RuntimeException(e);
		}
		return null;
	}
}
