package com.ejianc.business.supsignature.signature.controller;

import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.ejianc.business.supsignature.signature.bean.AuthLogEntity;
import com.ejianc.business.supsignature.signature.service.IAuthLogService;
import com.ejianc.business.supsignature.signature.service.ISaasSignService;
import com.ejianc.foundation.usercenter.vo.UserVO;
import com.ejianc.framework.auth.session.UserContext;
import com.ejianc.framework.core.response.CommonResponse;
import com.qiyuesuo.sdk.v2.bean.Company;
import com.qiyuesuo.sdk.v2.bean.User;
import com.qiyuesuo.sdk.v2.response.SaaSCompanyAuthPageResult;
import com.qiyuesuo.sdk.v2.response.SaaSPrivilegeUrlResult;
import com.qiyuesuo.sdk.v2.response.SdkResponse;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.*;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

@RestController
@RequestMapping("/saasSign")
public class SaasSignController implements Serializable {

    private static final long serialVersionUID = -4172061559099874029L;
    private final Logger logger = LoggerFactory.getLogger(this.getClass());

    @Value("${qiyuesuoSaas.authCallbackUrl}")
    private String authCallbackUrl; // 企业认证回调地址
    @Autowired
    private ISaasSignService service;
    @Autowired
    private IAuthLogService authLogService;


    /**
     * 查询公司信息
     *
     * @return Company 公司信息
     */
    @PostMapping(value = "/queryCompanyDetail")
    public CommonResponse<SdkResponse<Company>> queryCompanyDetail() {
        return service.queryCompanyDetail(service.getEnterpriseByTenantId().getData().getFullName());
    }


    /**
     * 获取认证链接
     *
     * @return SaaSCompanyAuthPageResult 认证链接
     */
    @PostMapping(value = "/queryCompanyAuthPage")
    public CommonResponse<SdkResponse<SaaSCompanyAuthPageResult>> queryCompanyAuthPage() {

        CommonResponse<UserVO> user = service.getUserByUserId();
        Map<String, Object> applicantInfo = new HashMap<>();
        applicantInfo.put("name", user.getData().getUserName());
        applicantInfo.put("contact", user.getData().getUserMobile());
        applicantInfo.put("contactType", "MOBILE");

	    UserContext userContext = new UserContext();
	    if (StringUtils.isBlank(userContext.getEnterpriseName())) {
		    return CommonResponse.error("当前请求的组织名称不能为空！");
	    }
        // 从我的空间取组织
        return service.queryCompanyAuthPage(userContext.getEnterpriseName().trim(), JSONObject.toJSONString(applicantInfo), null, null, null, authCallbackUrl);
    }


    /**
     * 获取授权链接
     *
     * @param companyId 公司id
     * @return 授权链接
     */
    @GetMapping(value = "/queryCompanyPrivilegeUrl")
    public CommonResponse<SdkResponse<SaaSPrivilegeUrlResult>> queryCompanyPrivilegeUrl(@RequestParam(value = "companyId") String companyId) {

        // 获取当前用户信息
        CommonResponse<UserVO> userVO = service.getUserByUserId();

        User user = new User();
        user.setContact(userVO.getData().getUserMobile());
        user.setContactType("MOBILE");

        // 指定授权模块
        List<String> privilegeModules = new ArrayList<>();
        privilegeModules.add("BASE_INFO");
        privilegeModules.add("SEAL");
        privilegeModules.add("CONTRACT");
        privilegeModules.add("COMPANY_EMPLOYEE");

        return service.queryCompanyPrivilegeUrl(Long.valueOf(companyId), null, user, false, null, null, privilegeModules);
    }


    /**
     * 企业认证回调
     *
     * @param signature 加密后的签名
     * @param timestamp 契约锁消息发送的时间戳
     * @param content   加密后的业务数据体
     */
    @PostMapping(value = "/companyAuthCallback")
    public void companyAuthCallback(@RequestParam(value = "signature") String signature,
                                    @RequestParam(value = "timestamp") String timestamp,
                                    @RequestParam(value = "content") String content) {
        logger.info("企业认证回调,返回的加密参数：content--{}", content);

        // 解密
        String decrypt = service.decrypt(content, "企业认证");
        logger.info("企业认证回调,解密后的参数：decrypt--{}", decrypt);
        // {"companyId":"2936986411087299557","requestId":"f57b8c31-67fa-4d9f-b386-57b5b3bb2c63","companyName":"住建项目","legalPerson":"白朋言","actionEvent":7,"status":1,"registerNo":"12351753712"}

        AuthLogEntity authLogEntity = JSONObject.parseObject(decrypt, AuthLogEntity.class);
        authLogService.saveOrUpdate(authLogEntity, false);
        logger.info("企业认证回调,解密后的参数格式化：authLogEntity--{}", JSONObject.toJSONString(authLogEntity, SerializerFeature.PrettyFormat, SerializerFeature.WriteMapNullValue));

    }


    /**
     * 企业授权回调
     *
     * @param signature 加密后的签名
     * @param timestamp 契约锁消息发送的时间戳
     * @param content   加密后的业务数据体
     */
    @PostMapping(value = "/companyPrivilegeCallback")
    public void companyPrivilegeCallback(@RequestParam(value = "signature") String signature,
                                         @RequestParam(value = "timestamp") String timestamp,
                                         @RequestParam(value = "content") String content) {
        logger.info("企业授权回调,返回的加密参数：content--{}", content);

        // 解密
        String decrypt = service.decrypt(content, "企业授权");
        logger.info("企业授权回调,解密后的参数：decrypt--{}", decrypt);

        logger.info("企业授权回调,解密后的参数格式化：decrypt--{}", JSONObject.toJSONString(decrypt, SerializerFeature.PrettyFormat, SerializerFeature.WriteMapNullValue));


    }


    /**
     * 合同状态回调
     *
     * @param signature 加密后的签名
     * @param timestamp 契约锁消息发送的时间戳
     * @param content   加密后的业务数据体
     */
    @PostMapping(value = "/contractStatusCallback")
    public void contractStatusCallback(@RequestParam(value = "signature") String signature,
                                       @RequestParam(value = "timestamp") String timestamp,
                                       @RequestParam(value = "content") String content) {
        logger.info("合同状态回调,返回的加密参数：encrypt--{}", content);

        // 解密
        String decrypt = service.decrypt(content, "合同状态");
        logger.info("合同状态回调,解密后的参数：decrypt--{}", decrypt);

        logger.info("合同状态回调,解密后的参数格式化：decrypt--{}", JSONObject.toJSONString(decrypt, SerializerFeature.PrettyFormat, SerializerFeature.WriteMapNullValue));

    }


    /**
     * 生成单点登录链接
     *
     * @return url
     */
    @GetMapping(value = "/ssoLoginUrl")
    public CommonResponse<String> ssoLoginUrl(@RequestParam(value = "companyId") String companyId,
                                              @RequestParam(value = "saasClient") String saasClient,
                                              @RequestParam(value = "page", required = false, defaultValue = "CONTRACT_LIST_PAGE") String page) {
        return CommonResponse.success("生成单点登录链接，成功！", service.ssoLoginUrl(companyId, saasClient, page));
    }


    /**
     * CAS验证
     *
     * @param ticket 单点登录请求凭证，由SaaS服务商调用单点登录接口时传入
     * @return String 返回验证结果；用户信息
     */
    @PostMapping(value = "/casVerification")
    public String casVerification(@RequestParam(value = "ticket") String ticket) {
        logger.info("契约锁请求CAS验证接口--------入参：{}", ticket);
        String ticketDetail = service.casVerification(ticket);
        JSONObject ticketObject = JSONObject.parseObject(ticketDetail);
        logger.info("兑换的ticketDetail--------{}", ticketObject);
        Map<String, Object> map = new HashMap<>();
        map.put("result", ticketObject);
        map.put("code", 0);
        map.put("message", "SUCCESS");
        logger.info("契约锁请求CAS验证接口--------出参：{}", JSONObject.toJSONString(map, SerializerFeature.PrettyFormat, SerializerFeature.WriteMapNullValue));

        return JSONObject.toJSONString(map, SerializerFeature.PrettyFormat, SerializerFeature.WriteMapNullValue);
    }

}