package com.ejianc.business.tax.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.StringUtils;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.income.api.IIncomeContractApi;
import com.ejianc.business.income.vo.ContractVo;
import com.ejianc.business.tax.bean.InvoiceOpenEntity;
import com.ejianc.business.tax.bean.InvoiceReceiveEntity;
import com.ejianc.business.tax.mapper.InvoiceOpenMapper;
import com.ejianc.business.tax.service.IInvoiceOpenService;
import com.ejianc.business.tax.vo.InvoiceOpenRecordVO;
import com.ejianc.business.tax.vo.InvoiceOpenVO;
import com.ejianc.business.taxnew.bean.InvoiceOpenRegistEntity;
import com.ejianc.business.taxnew.mapper.InvoiceOpenRegistMapper;
import com.ejianc.business.taxnew.service.IInvoiceOpenRegistService;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.framework.cache.utils.RedisTool;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.collection.ListUtil;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.skeleton.template.BaseServiceImpl;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

/**
 * <p>
 * 合同收票表 服务实现类
 * </p>
 *
 * @author yqls
 * @since 2020-06-04
 */
@Service("InvoiceOpenService")
public class InvoiceOpenServiceImpl extends BaseServiceImpl<InvoiceOpenMapper, InvoiceOpenEntity> implements IInvoiceOpenService {
    private static final String TAX_INVOICE_OPEN_BILL_CODE = "TAX_OPEN";
    private final Logger logger = LoggerFactory.getLogger(getClass());
    @Autowired
    private IBillCodeApi billCodeApi;
    @Autowired
    private InvoiceOpenMapper invoiceOpenMapper;
    @Autowired
    private IOrgApi orgApi;
    @Autowired
    private JedisPool jedisPool;
    @Autowired
    private IIncomeContractApi iIncomeContractApi;
    @Autowired
    private InvoiceOpenRegistMapper invoiceOpenRegistMapper;

    private static final String BILL_CODE = "tax-open-regist-001";

    @Override
    public CommonResponse<InvoiceOpenVO> saveOrUpdate(InvoiceOpenVO invoiceOpenVo) {
        //保存时校验合同version是否一致
        if(invoiceOpenVo.getContractVersion()!=null&&invoiceOpenVo.getContractVersion()!=0){
            Jedis jedis = jedisPool.getResource();
            boolean locked = false;
            try{
                locked = RedisTool.tryLock(jedis, String.valueOf(invoiceOpenVo.getContractId()), "saveOrUpdate", 1000);
                logger.info("判断单据单据锁结果------"+locked);
                if(locked){
                    CommonResponse<ContractVo> contractVoCommonResponse = iIncomeContractApi.queryDetail(invoiceOpenVo.getContractId());
                    if (contractVoCommonResponse.isSuccess()){
                        ContractVo data = contractVoCommonResponse.getData();
                        Integer version = data.getVersion()==null?0:data.getVersion();
                        Integer conVersion = invoiceOpenVo.getContractVersion();
                        if(!version.equals(conVersion)){
                            return CommonResponse.error("该合同已被更新，请刷新后重做！");
                        }
                    }
                }else{
                    return CommonResponse.error("出现并发操作,请稍后重试！");
                }
            }catch (Exception e) {
                e.printStackTrace();
            } finally {
                if(locked) {
                    RedisTool.releaseLock(jedis, String.valueOf(invoiceOpenVo.getContractId()), "saveOrUpdate");
                }
                jedis.close();
            }
        }
        Long tenantId = InvocationInfoProxy.getTenantid();
        InvoiceOpenEntity entity = null;
        if(StringUtils.isEmpty(invoiceOpenVo.getBillCode())){
            //新增
            CommonResponse<String> billCode = billCodeApi.getCodeBatchByRuleCode(TAX_INVOICE_OPEN_BILL_CODE,tenantId);
            if(billCode.isSuccess()) {
                invoiceOpenVo.setBillCode(billCode.getData());
            }else{
                throw new BusinessException("网络异常， 编码生成失败， 请稍后再试");
            }
        }
        if(invoiceOpenVo.getId() != null && invoiceOpenVo.getId() > 0) {
            //修改  校验合同编号是否重复
            LambdaQueryWrapper<InvoiceOpenEntity> lambda = Wrappers.<InvoiceOpenEntity>lambdaQuery();
            lambda.eq(InvoiceOpenEntity::getBillCode, invoiceOpenVo.getBillCode());
            lambda.eq(InvoiceOpenEntity::getTenantId, tenantId);
            lambda.ne(InvoiceOpenEntity::getId, invoiceOpenVo.getId());
            List<InvoiceOpenEntity> entities = super.list(lambda);
            if(entities != null && entities.size() > 0) {
                throw new BusinessException("存在相同编码，不允许保存!");
            }
        }else{
            //校验合同编号是否重复
            LambdaQueryWrapper<InvoiceOpenEntity> lambda = Wrappers.<InvoiceOpenEntity>lambdaQuery();
            lambda.eq(InvoiceOpenEntity::getTenantId, tenantId);
            lambda.eq(InvoiceOpenEntity::getBillCode, invoiceOpenVo.getBillCode());
            List<InvoiceOpenEntity> entities = super.list(lambda);
            if(entities != null && entities.size() > 0) {
                throw new BusinessException("存在相同编码，不允许保存!");
            }
        }

        entity = BeanMapper.map(invoiceOpenVo, InvoiceOpenEntity.class);
        QueryParam param = new QueryParam();
        param.getParams().put("invoiceNumber",new Parameter(QueryParam.EQ,invoiceOpenVo.getInvoiceNumber()));
        if(invoiceOpenVo.getRowState().equals("add")){
            List<InvoiceOpenEntity> list = this.queryList(param);
            if(ListUtil.isNotEmpty(list)){
                throw new BusinessException("发票: 号码【"+invoiceOpenVo.getInvoiceNumber()+"】已存在！");
            }
        }

        if(!"del".equals(invoiceOpenVo.getRowState())){
            List<InvoiceOpenEntity> list = this.queryList(param);
            if(ListUtil.isNotEmpty(list) && list.get(0).getRegistId()!=null && !list.get(0).getRegistId().equals(entity.getId())){
                throw new BusinessException("号码【"+invoiceOpenVo.getInvoiceNumber()+"】已存在！");
            }
        }
        InvoiceOpenRegistEntity invoiceOpenRegistEntity = new InvoiceOpenRegistEntity();
        invoiceOpenRegistEntity.setRegistDate(invoiceOpenVo.getInvoiceDate());
        invoiceOpenRegistEntity.setSeller(invoiceOpenVo.getSupplierName());
        invoiceOpenRegistEntity.setSellerId(invoiceOpenVo.getSupplierId());
        invoiceOpenRegistEntity.setSellerAddrPhone(invoiceOpenVo.getSellerAddrPhone());
        invoiceOpenRegistEntity.setInvoiceAuantity(1);
        invoiceOpenRegistEntity.setHasContract(invoiceOpenVo.getType());
        invoiceOpenRegistEntity.setOrgName(invoiceOpenVo.getOrgName());
        invoiceOpenRegistEntity.setOrgId(invoiceOpenVo.getOrgId());
        invoiceOpenRegistEntity.setRegistState(0);
        invoiceOpenRegistEntity.setRegistOpenType(0);
        invoiceOpenRegistEntity.setCode(BILL_CODE);
        invoiceOpenRegistEntity.setTaxMny(invoiceOpenVo.getTaxMny());
        invoiceOpenRegistEntity.setInvoiceTaxMny(invoiceOpenVo.getInvoiceTaxMny());
        invoiceOpenRegistEntity.setBuyerId(invoiceOpenVo.getCustomerId());
        invoiceOpenRegistEntity.setBuyer(invoiceOpenVo.getCustomerName());
        invoiceOpenRegistEntity.setBuyerTaxId(invoiceOpenVo.getCustomerCreditCode());
        invoiceOpenRegistEntity.setContractId(invoiceOpenVo.getContractId());
        invoiceOpenRegistEntity.setContractName(invoiceOpenVo.getContractName());
        invoiceOpenRegistEntity.setProjectName(invoiceOpenVo.getProjectName());
        invoiceOpenRegistEntity.setProjectId(invoiceOpenVo.getProjectId());
        invoiceOpenRegistEntity.setBillState(0);
        invoiceOpenRegistEntity.setMemo(invoiceOpenVo.getMemo());
        invoiceOpenRegistEntity.setInvoiceMny(invoiceOpenVo.getInvoiceMny());
        invoiceOpenRegistEntity.setBuyerAddrPhone(invoiceOpenVo.getBuyerAddrPhone());
        invoiceOpenRegistEntity.setSellerBankAccount(invoiceOpenVo.getSellerBankAccount());
        invoiceOpenRegistEntity.setBuyerBankAccount(invoiceOpenVo.getBuyerBankAccount());

        if(invoiceOpenVo.getRegistId() != null){
            invoiceOpenRegistMapper.updateById(invoiceOpenRegistEntity);
        }else{
            invoiceOpenRegistMapper.insert(invoiceOpenRegistEntity);
        }
        entity.setRegistId(invoiceOpenRegistEntity.getId());
        entity.setRegistCode(invoiceOpenRegistEntity.getCode());
        super.saveOrUpdate(entity,false);

        return CommonResponse.success(BeanMapper.map(entity, InvoiceOpenVO.class));
    }
    @Override
    public InvoiceOpenVO queryDetail(Long id) {
        InvoiceOpenEntity entity = invoiceOpenMapper.selectById(id);
        if(entity != null) {
            InvoiceOpenVO invoiceOpenVo = BeanMapper.map(entity, InvoiceOpenVO.class);
            return invoiceOpenVo;
        }
        return null;
    }

    @Override
    public void deleteInvoiceOpen(List<InvoiceOpenVO> vos) {
        super.removeByIds(vos.stream().map(InvoiceOpenVO::getId).collect(Collectors.toList()),false);
    }

    @Override
    public BigDecimal getOpenMny(Long contractId, Long orgId) {
        BigDecimal openMny = invoiceOpenMapper.getOpenMny(contractId,orgId);
        return openMny;
    }
    @Override
    public InvoiceOpenRecordVO getInvoiceOpenRecord(Long id) {
        InvoiceOpenRecordVO invoiceOpenRecordVO = new InvoiceOpenRecordVO();
        invoiceOpenRecordVO.setContractId(id);
        List<Long> longList = orgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList());
        LambdaQueryWrapper<InvoiceOpenEntity> lambda = Wrappers.<InvoiceOpenEntity>lambdaQuery();
        lambda.eq(InvoiceOpenEntity::getContractId,id);
        lambda.in(InvoiceOpenEntity::getBillState, BillStateEnum.PASSED_STATE.getBillStateCode(),BillStateEnum.COMMITED_STATE.getBillStateCode());
        lambda.orderByDesc(InvoiceOpenEntity::getCreateTime);
        lambda.in(InvoiceOpenEntity::getOrgId, longList);
        List<InvoiceOpenEntity> entities = super.list(lambda);
        if(entities != null && entities.size() > 0){
            BigDecimal invoiceMny = BigDecimal.ZERO;//发票金额（不含税）
            BigDecimal invoiceTaxMny = BigDecimal.ZERO;//发票金额（含税）
            BigDecimal taxMny = BigDecimal.ZERO;//税金
            List<InvoiceOpenVO> invoiceOpenVOS = BeanMapper.mapList(entities, InvoiceOpenVO.class);
            for (InvoiceOpenVO invoiceOpenVO : invoiceOpenVOS){
                invoiceMny = invoiceMny.add(invoiceOpenVO.getInvoiceMny());
                invoiceTaxMny = invoiceTaxMny.add(invoiceOpenVO.getInvoiceTaxMny());
                if(invoiceOpenVO.getTaxMny()!=null){
                    taxMny = taxMny.add(invoiceOpenVO.getTaxMny());
                }
            }
            invoiceOpenRecordVO.setInvoiceMny(invoiceMny);
            invoiceOpenRecordVO.setInvoiceTaxMny(invoiceTaxMny);
            invoiceOpenRecordVO.setTaxMny(taxMny);
            invoiceOpenRecordVO.setInvoiceOpenVOList(invoiceOpenVOS);
        }
        return invoiceOpenRecordVO;
    }

    @Override
    public BigDecimal getNoContractOpenMny(Long projectId, Long customerId) {
        BigDecimal noContractMny = invoiceOpenMapper.getNoContractOpenMny(projectId,customerId);
        return noContractMny;
    }

    @Override
    public BigDecimal billingContract(Long projectId) {
        BigDecimal billingContract = BigDecimal.ZERO;
        List<Integer> billStatus = new ArrayList<>();
        billStatus.add(1);
        billStatus.add(3);
        LambdaQueryWrapper<InvoiceOpenEntity> lambda = Wrappers.<InvoiceOpenEntity>lambdaQuery();
        lambda.eq(InvoiceOpenEntity::getProjectId, projectId);
        lambda.eq(InvoiceOpenEntity::getDr, 0);
        lambda.eq(InvoiceOpenEntity::getType, 1);
        lambda.in(InvoiceOpenEntity::getBillState, billStatus);
        List<InvoiceOpenEntity> quoteEntities = super.list(lambda);
        for (InvoiceOpenEntity ce : quoteEntities) {
            billingContract = billingContract.add(ce.getInvoiceTaxMny());
        }
        return billingContract;
    }

    @Override
    public List<InvoiceOpenVO> queryInvoiceOpenList(QueryWrapper wrapper, Page<InvoiceOpenVO> page) {
        return baseMapper.queryInvoiceOpenList(wrapper,page);
    }

    /**
     * 获取billCode
     * @return
     */
    public String getBillCode(String code){
        CommonResponse<String> billCode = billCodeApi.getCodeBatchByRuleCode(code, InvocationInfoProxy.getTenantid());
        if(billCode.isSuccess()) {
            return billCode.getData();
        }else{
            throw new BusinessException("网络异常， 编码生成失败， 请稍后再试");
        }
    }
}
