package com.ejianc.business.tax.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.toolkit.StringUtils;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.ejianc.business.tax.bean.InvoiceReceiveEntity;
import com.ejianc.business.tax.mapper.InvoiceReceiveMapper;
import com.ejianc.business.tax.service.IInvoiceReceiveService;
import com.ejianc.business.tax.utils.ComputeUtil;
import com.ejianc.business.tax.vo.*;
import com.ejianc.business.taxnew.bean.InvoiceReceivePoolEntity;
import com.ejianc.business.taxnew.bean.InvoiceReceiveRegistEntity;
import com.ejianc.business.taxnew.service.IInvoiceReceiveRegistService;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.framework.cache.utils.RedisTool;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.collection.ListUtil;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.skeleton.fields.service.ICommenQueryFieldsService;
import com.ejianc.framework.skeleton.template.BaseServiceImpl;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * <p>
 * 有合同开票 服务实现类
 * </p>
 *
 * @author yqls
 * @since 2020-06-04
 */
@Service("InvoiceReceiveService")
public class InvoiceReceiveServiceImpl extends BaseServiceImpl<InvoiceReceiveMapper, InvoiceReceiveEntity> implements IInvoiceReceiveService {
    private static final String TAX_INVOICE_RECEIVE_BILL_CODE = "TAX_RECEIVE";
    private final Logger logger = LoggerFactory.getLogger(getClass());
    @Autowired
    private IBillCodeApi billCodeApi;
    @Autowired
    private InvoiceReceiveMapper invoiceReceiveMapper;
    @Autowired
    private IOrgApi orgApi;
    @Autowired
    private JedisPool jedisPool;
    @Autowired
    private ICommenQueryFieldsService commenQueryFieldsService;
    @Autowired
    private IInvoiceReceiveRegistService invoiceReceiveRegistService;
    private static final String BILL_CODE = "tax-receive-code-001";


    @Override
    public CommonResponse<InvoiceReceiveVO> saveOrUpdate(InvoiceReceiveVO invoiceReceiveVo) {

        //保存时校验合同version是否一致
        if (invoiceReceiveVo.getContractVersion() != null && invoiceReceiveVo.getContractVersion() != 0) {
            Jedis jedis = jedisPool.getResource();
            boolean locked = false;
            try {
                locked = RedisTool.tryLock(jedis, String.valueOf(invoiceReceiveVo.getContractId()), "saveOrUpdate", 1000);
                logger.info("判断单据单据锁结果------" + locked);
                if (locked) {
                    JSONObject contractEntity = commenQueryFieldsService.queryBillDetail(String.valueOf(invoiceReceiveVo.getContractId()), getBillTypeCode(invoiceReceiveVo.getContractType()));
                    String objVersion = contractEntity.getJSONObject("data").get("version").toString();
                    Integer version = contractEntity.getJSONObject("data").get("version") == null ? 0 : Integer.parseInt(objVersion.substring(0, objVersion.indexOf(".")));
                    Integer conVersion = invoiceReceiveVo.getContractVersion();
                    if (!version.equals(conVersion)) {
                        return CommonResponse.error("该合同已被更新，请刷新后重做！");
                    }
                } else {
                    return CommonResponse.error("出现并发操作,请稍后重试！");
                }
            } catch (Exception e) {
                e.printStackTrace();
            } finally {
                if (locked) {
                    RedisTool.releaseLock(jedis, String.valueOf(invoiceReceiveVo.getContractId()), "saveOrUpdate");
                }
                jedis.close();
            }
        }
        Long tenantId = InvocationInfoProxy.getTenantid();
        InvoiceReceiveEntity entity = null;
        if (StringUtils.isEmpty(invoiceReceiveVo.getBillCode())) {
            //新增
            CommonResponse<String> billCode = billCodeApi.getCodeBatchByRuleCode(TAX_INVOICE_RECEIVE_BILL_CODE, tenantId);
            if (billCode.isSuccess()) {
                invoiceReceiveVo.setBillCode(billCode.getData());
            } else {
                throw new BusinessException("网络异常， 编码生成失败， 请稍后再试");
            }
        }
        if (invoiceReceiveVo.getId() != null && invoiceReceiveVo.getId() > 0) {
            //修改  校验合同编号是否重复
            LambdaQueryWrapper<InvoiceReceiveEntity> lambda = Wrappers.<InvoiceReceiveEntity>lambdaQuery();
            lambda.eq(InvoiceReceiveEntity::getBillCode, invoiceReceiveVo.getBillCode());
            lambda.eq(InvoiceReceiveEntity::getTenantId, tenantId);
            lambda.ne(InvoiceReceiveEntity::getId, invoiceReceiveVo.getId());
            List<InvoiceReceiveEntity> entities = super.list(lambda);
            if (entities != null && entities.size() > 0) {
                throw new BusinessException("存在相同编码，不允许保存!");
            }
        } else {
            //校验合同编号是否重复
            LambdaQueryWrapper<InvoiceReceiveEntity> lambda = Wrappers.<InvoiceReceiveEntity>lambdaQuery();
            lambda.eq(InvoiceReceiveEntity::getTenantId, tenantId);
            lambda.eq(InvoiceReceiveEntity::getBillCode, invoiceReceiveVo.getBillCode());
            List<InvoiceReceiveEntity> entities = super.list(lambda);
            if (entities != null && entities.size() > 0) {
                throw new BusinessException("存在相同编码，不允许保存!");
            }
        }

        QueryParam param = new QueryParam();
        param.getParams().put("invoiceNumber",new Parameter(QueryParam.EQ,invoiceReceiveVo.getInvoiceNumber()));
        if(invoiceReceiveVo.getReceiveBillId() == null){
            List<InvoiceReceiveEntity> list = this.queryList(param);
            if(ListUtil.isNotEmpty(list)){
                throw new BusinessException("发票: 号码【"+invoiceReceiveVo.getInvoiceNumber()+"】已存在！");
            }
        }else{
            List<InvoiceReceiveEntity> list = this.queryList(param);
            if(ListUtil.isNotEmpty(list) && list.get(0).getReceiveBillId()!=null && !list.get(0).getReceiveBillId().equals(entity.getId())){
                throw new BusinessException("号码【"+invoiceReceiveVo.getInvoiceNumber()+"】已存在！");
            }
        }

        entity = BeanMapper.map(invoiceReceiveVo, InvoiceReceiveEntity.class);
        entity.setFlag(0);
        InvoiceReceiveRegistEntity receiveRegistEntity = new InvoiceReceiveRegistEntity();
        receiveRegistEntity.setBillState(0);
        receiveRegistEntity.setBuyerId(invoiceReceiveVo.getCustomerId());
        receiveRegistEntity.setBuyerName(invoiceReceiveVo.getCustomerName());
        receiveRegistEntity.setBuyerTaxNo(invoiceReceiveVo.getCustomerCreditCode());
        receiveRegistEntity.setCode(getBillCode(BILL_CODE));
        receiveRegistEntity.setContractId(invoiceReceiveVo.getContractId());
        receiveRegistEntity.setContractName(invoiceReceiveVo.getContractName());
        receiveRegistEntity.setMemo(invoiceReceiveVo.getMemo());
        receiveRegistEntity.setDependOnProject(Integer.valueOf(invoiceReceiveVo.getDependOnProject()!= null?invoiceReceiveVo.getDependOnProject():null));
        receiveRegistEntity.setInvoiceQuantity(1);
        receiveRegistEntity.setOrgId(invoiceReceiveVo.getOrgId());
        receiveRegistEntity.setOrgName(invoiceReceiveVo.getOrgName());
        receiveRegistEntity.setProjectId(invoiceReceiveVo.getProjectId());
        receiveRegistEntity.setProjectName(invoiceReceiveVo.getProjectName());
        receiveRegistEntity.setRegistDate(invoiceReceiveVo.getInvoiceDate());
        receiveRegistEntity.setSellerId(invoiceReceiveVo.getSupplierId());
        receiveRegistEntity.setSellerName(invoiceReceiveVo.getSupplierName());
        receiveRegistEntity.setSellerTaxNo(invoiceReceiveVo.getSupplierCreditCode());
        receiveRegistEntity.setType(invoiceReceiveVo.getType());
        receiveRegistEntity.setCurrentReceiveMny(invoiceReceiveVo.getInvoiceMny());
        receiveRegistEntity.setTaxMny(invoiceReceiveVo.getTaxMny());
        receiveRegistEntity.setProcessPersonId(InvocationInfoProxy.getUserid());
        if(invoiceReceiveVo.getReceiveBillId() != null){
            invoiceReceiveRegistService.updateById(receiveRegistEntity);
        }else{
            invoiceReceiveRegistService.save(receiveRegistEntity);
        }
        entity.setReceiveBillId(receiveRegistEntity.getId());
        entity.setReceiveBillCode(receiveRegistEntity.getCode());
        super.saveOrUpdate(entity, false);

        return CommonResponse.success(BeanMapper.map(entity, InvoiceReceiveVO.class));
    }

    @Override
    public InvoiceReceiveVO queryDetail(Long id) {
        InvoiceReceiveEntity entity = invoiceReceiveMapper.selectById(id);
        if (entity != null) {
            InvoiceReceiveVO invoiceReceiveVo = BeanMapper.map(entity, InvoiceReceiveVO.class);
            return invoiceReceiveVo;
        }
        return null;
    }

    @Override
    public void deleteInvoiceReceive(List<InvoiceReceiveVO> vos) {
        super.removeByIds(vos.stream().map(InvoiceReceiveVO::getId).collect(Collectors.toList()), false);
    }

    @Override
    public BigDecimal getReceiveMny(Long contractId, Long orgId) {
        BigDecimal receiveMny = invoiceReceiveMapper.getReceiveMny(contractId, orgId);
        return receiveMny;
    }

    @Override
    public InvoiceReceiveRecordVO getInvoiceReceiveRecord(Long id) {

        InvoiceReceiveRecordVO invoiceReceiveRecordVO = new InvoiceReceiveRecordVO();
        invoiceReceiveRecordVO.setContractId(id);
        List<Long> longList = orgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList());
        LambdaQueryWrapper<InvoiceReceiveEntity> lambda = Wrappers.<InvoiceReceiveEntity>lambdaQuery();
        lambda.eq(InvoiceReceiveEntity::getContractId, id);
        lambda.in(InvoiceReceiveEntity::getOrgId, longList);
        lambda.orderByDesc(InvoiceReceiveEntity::getCreateTime);
        List<InvoiceReceiveEntity> entitiesAllState = super.list(lambda);
        lambda.in(InvoiceReceiveEntity::getBillState, BillStateEnum.PASSED_STATE.getBillStateCode(), BillStateEnum.COMMITED_STATE.getBillStateCode());
        List<InvoiceReceiveEntity> entities = super.list(lambda);
        if (CollectionUtils.isNotEmpty(entities)) {
            List<InvoiceReceiveVO> list = BeanMapper.mapList(entities, InvoiceReceiveVO.class);
            BigDecimal invoiceMny = BigDecimal.ZERO;//发票金额（不含税）
            BigDecimal invoiceTaxMny = BigDecimal.ZERO;//发票金额（含税）
            BigDecimal taxMny = BigDecimal.ZERO;//税金
            for (InvoiceReceiveVO invoiceReceiveVO : list) {
                invoiceReceiveVO.setBillStateName(BillStateEnum.getEnumByStateCode(invoiceReceiveVO.getBillState()).getDescription());
                invoiceMny = ComputeUtil.safeAdd(invoiceMny,invoiceReceiveVO.getInvoiceMny());
                invoiceTaxMny = ComputeUtil.safeAdd(invoiceTaxMny,invoiceReceiveVO.getInvoiceTaxMny());
                if(invoiceReceiveVO.getTaxMny()!=null){
                    taxMny = ComputeUtil.safeAdd(taxMny,invoiceReceiveVO.getTaxMny());
                }
            }
            invoiceReceiveRecordVO.setInvoiceMny(invoiceMny);
            invoiceReceiveRecordVO.setInvoiceTaxMny(invoiceTaxMny);
            invoiceReceiveRecordVO.setTaxMny(taxMny);
            invoiceReceiveRecordVO.setInvoiceReceiveVOList(list);
        }
        if (CollectionUtils.isNotEmpty(entitiesAllState)) {
            List<InvoiceReceiveVO> listAllState = BeanMapper.mapList(entitiesAllState, InvoiceReceiveVO.class);
            for (InvoiceReceiveVO invoiceReceiveVO : listAllState) {
                invoiceReceiveVO.setBillStateName(BillStateEnum.getEnumByStateCode(invoiceReceiveVO.getBillState()).getDescription());
            }
            invoiceReceiveRecordVO.setInvoiceReceiveVOListAllState(listAllState);
        }
        return invoiceReceiveRecordVO;
    }

    @Override
    public Map<Long,InvoiceReceiveRecordVO> getInvoiceReceiveRecords(List<Long> ids) {
        QueryWrapper<InvoiceReceiveRegistEntity> wrapper = new QueryWrapper<>();
        wrapper.in("receive_type",4);
        wrapper.in("contract_id",ids);
        wrapper.in("bill_state",BillStateEnum.PASSED_STATE.getBillStateCode(),BillStateEnum.COMMITED_STATE.getBillStateCode());
        wrapper.orderByDesc("create_time");
        List<InvoiceReceiveRegistEntity> entitys = invoiceReceiveRegistService.list(wrapper);
        logger.info("entitys.size()"+entitys.size());
        Map<Long,InvoiceReceiveRecordVO> map = new HashMap<>();
        entitys.forEach(item->{
            InvoiceReceiveRecordVO invoiceReceiveRecordVO = map.get(item.getContractId());
            if(null == invoiceReceiveRecordVO){
                InvoiceReceiveRecordVO vo = new InvoiceReceiveRecordVO();
                vo.setContractId(item.getContractId());
                vo.setInvoiceMny(item.getCurrentReceiveMny());
                vo.setInvoiceTaxMny(item.getCurrentReceiveMnyTax());
                vo.setTaxMny(item.getTaxMny());
                map.put(item.getContractId(),vo);
            }else{
                invoiceReceiveRecordVO.setInvoiceMny(ComputeUtil.safeAdd(invoiceReceiveRecordVO.getInvoiceMny(), item.getCurrentReceiveMny()));
                invoiceReceiveRecordVO.setInvoiceTaxMny(ComputeUtil.safeAdd(invoiceReceiveRecordVO.getInvoiceTaxMny(), item.getCurrentReceiveMnyTax()));
                invoiceReceiveRecordVO.setTaxMny(ComputeUtil.safeAdd(invoiceReceiveRecordVO.getTaxMny(), item.getTaxMny()));
            }
        });
        return map;
    }

    @Override
    public CommonResponse<String> updateFlag(List<InvoiceReceiveFlagVO> invoiceReceiveFlagVOS) {
        List<InvoiceReceiveEntity> list = BeanMapper.mapList(invoiceReceiveFlagVOS, InvoiceReceiveEntity.class);
        List<Long> isLongList = new ArrayList<>();
        List<Long> notIslongList = new ArrayList<>();
        List<Long> notTaxCostList = new ArrayList<>();
        List<Long> TaxCostList = new ArrayList<>();
        for (InvoiceReceiveEntity invoiceReceiveEntity : list) {
            if (null!=invoiceReceiveEntity.getFlag()&&invoiceReceiveEntity.getFlag().equals(0)) {
                notIslongList.add(invoiceReceiveEntity.getId());
            }
            if (null!=invoiceReceiveEntity.getFlag()&&invoiceReceiveEntity.getFlag().equals(1)) {
                isLongList.add(invoiceReceiveEntity.getId());
            }
            if (null!=invoiceReceiveEntity.getTaxCostFlag()&&invoiceReceiveEntity.getTaxCostFlag().equals(0)) {
                notTaxCostList.add(invoiceReceiveEntity.getId());
            }
            if (null!=invoiceReceiveEntity.getTaxCostFlag()&&invoiceReceiveEntity.getTaxCostFlag().equals(1)) {
                TaxCostList.add(invoiceReceiveEntity.getId());
            }
        }
        //
        if (isLongList != null && isLongList.size() > 0) {
            LambdaUpdateWrapper<InvoiceReceiveEntity> updateWrapper = new LambdaUpdateWrapper<>();
            updateWrapper.in(InvoiceReceiveEntity::getId, isLongList);
            updateWrapper.set(InvoiceReceiveEntity::getFlag, 1);//(1:是，0：否)
            super.update(updateWrapper);
        }
        if (notIslongList != null && notIslongList.size() > 0) {
            LambdaUpdateWrapper<InvoiceReceiveEntity> updateWrapper = new LambdaUpdateWrapper<>();
            updateWrapper.in(InvoiceReceiveEntity::getId, notIslongList);
            updateWrapper.set(InvoiceReceiveEntity::getFlag, 0);//(1:是，0：否)
            super.update(updateWrapper);
        }
        if (TaxCostList != null && TaxCostList.size() > 0) {
            LambdaUpdateWrapper<InvoiceReceiveEntity> updateWrapper = new LambdaUpdateWrapper<>();
            updateWrapper.in(InvoiceReceiveEntity::getId, TaxCostList);
            updateWrapper.set(InvoiceReceiveEntity::getTaxCostFlag, 1);//(1:是，0：否)
            super.update(updateWrapper);
        }
        if (notTaxCostList != null && notTaxCostList.size() > 0) {
            LambdaUpdateWrapper<InvoiceReceiveEntity> updateWrapper = new LambdaUpdateWrapper<>();
            updateWrapper.in(InvoiceReceiveEntity::getId, notTaxCostList);
            updateWrapper.set(InvoiceReceiveEntity::getTaxCostFlag, 0);//(1:是，0：否)
            super.update(updateWrapper);
        }
        for (InvoiceReceiveEntity invoiceReceiveEntity : list) {

        }
        return CommonResponse.success("修改收票引用标识成功！");
    }

    @Override
    public Map<String, InvoiceMnyWarnVO> getInvoiceMnyByTenantIds(InvoicePubWarnQueryVO pubWarnQueryVO) {
        if (null == pubWarnQueryVO.getContractType()) {
            throw new BusinessException("合同类型不能为空！");
        }
        Map<String, InvoiceMnyWarnVO> map = new HashMap<>();
        List<InvoiceMnyWarnVO> list = baseMapper.getInvoiceMnyByTenantIds(pubWarnQueryVO);
        for (InvoiceMnyWarnVO invoiceMnyWarnVO : list) {
            map.put(invoiceMnyWarnVO.getTenantId() + "@" + invoiceMnyWarnVO.getOrgId() + "@" + invoiceMnyWarnVO.getContractId(), invoiceMnyWarnVO);
        }
        return map;
    }

    @Override
    public List<InvoiceReceiveVO> getTaxReceive(Long projectId) {
        List<Integer> billStatus = new ArrayList<>();
        billStatus.add(1);
        billStatus.add(3);
        LambdaQueryWrapper<InvoiceReceiveEntity> lambda = Wrappers.<InvoiceReceiveEntity>lambdaQuery();
        lambda.eq(InvoiceReceiveEntity::getProjectId, projectId);
        lambda.eq(InvoiceReceiveEntity::getDr, 0);
        lambda.in(InvoiceReceiveEntity::getBillState, billStatus);
        List<InvoiceReceiveEntity> entities = super.list(lambda);
        return BeanMapper.mapList(entities, InvoiceReceiveVO.class);
    }

    @Override
    public Map<String, InvoiceMnyWarnVO> getWarnInvoiceMny(InvoicePubWarnQueryVO pubWarnQueryVO) {
        if (null == pubWarnQueryVO.getContractType()) {
            throw new BusinessException("合同类型不能为空！");
        }
        Map<String, InvoiceMnyWarnVO> map = new HashMap<>();
        List<InvoiceMnyWarnVO> list = baseMapper.getWarnInvoiceMny(pubWarnQueryVO);
        for (InvoiceMnyWarnVO invoiceMnyWarnVO : list) {
            map.put(invoiceMnyWarnVO.getTenantId() + "@" + invoiceMnyWarnVO.getOrgId() + "@" + invoiceMnyWarnVO.getContractId(), invoiceMnyWarnVO);
        }
        return map;
    }

    @Override
    public CommonResponse<String> updateUsedInvoiceMny(List<InvoiceReceiveFlagVO> invoiceReceiveFlagVOS) {
        if (CollectionUtils.isNotEmpty(invoiceReceiveFlagVOS)) {
            List<InvoiceReceiveEntity> list = BeanMapper.mapList(invoiceReceiveFlagVOS, InvoiceReceiveEntity.class);
            List<Long> ids = list.stream().map(InvoiceReceiveEntity::getId).collect
                    (Collectors.toList());

            LambdaQueryWrapper<InvoiceReceiveEntity> lambda = Wrappers.<InvoiceReceiveEntity>lambdaQuery();
            lambda.in(InvoiceReceiveEntity::getId, ids);
            List<InvoiceReceiveEntity> entities = super.list(lambda);

            if (CollectionUtils.isNotEmpty(entities)) {
                Map<Long, InvoiceReceiveEntity> maps = entities.stream().collect(Collectors.toMap(InvoiceReceiveEntity::getId, Function.identity(), (key1, key2) -> key2));

                // 校验当钱可用金额是否足以被使用，有可能会有并发和时间差操作
                for (InvoiceReceiveFlagVO i : invoiceReceiveFlagVOS) {
                    Long id = i.getId();
                    InvoiceReceiveEntity e = maps.get(id);
                    if (e != null) {
                        BigDecimal use = ComputeUtil.safeAdd(e.getUsedInvoiceTaxMny(), i.getNowUesdInvoiceTaxMny());
                        //发票金额 小于已占用+ 当前占用
                        if (ComputeUtil.isLessThan(e.getInvoiceTaxMny(), use)) {
                            return CommonResponse.error("操作失败：发票【号码：" + e.getInvoiceNumber() + "】的剩余金额为：" + ComputeUtil.scaleTwo(ComputeUtil.safeSub(e.getInvoiceTaxMny(), e.getUsedInvoiceTaxMny())) + "，请刷新后重新选择！");
                        }
                        i.setNowUesdInvoiceTaxMny(use);
                        i.setFlag(ComputeUtil.equals(e.getInvoiceTaxMny(), use) ? 1 : 0);
                    }
                }
            }
            // 更新发票金额和占用状态
            for (InvoiceReceiveFlagVO i : invoiceReceiveFlagVOS) {
                LambdaUpdateWrapper<InvoiceReceiveEntity> updateWrapper = new LambdaUpdateWrapper<>();
                updateWrapper.eq(InvoiceReceiveEntity::getId, i.getId());
                updateWrapper.set(InvoiceReceiveEntity::getFlag, i.getFlag());//(1:是，0：否)
                updateWrapper.set(InvoiceReceiveEntity::getUsedInvoiceTaxMny, i.getNowUesdInvoiceTaxMny());
                super.update(updateWrapper);
            }
        }

        return CommonResponse.success("修改收票引用标识成功！");
    }

    public String getBillTypeCode(String contractType) {
        if ("1".equals(contractType)) {//分包合同
            return "BT200529000000001";
        } else if ("2".equals(contractType)) {//设备采购合同
            return "BT200528000000001";
        } else if ("3".equals(contractType)) {//设备租赁合同
            return "BT200528000000003";
        } else if ("4".equals(contractType)) {//物资采购合同
            return "BT200608000000001";
        } else if ("5".equals(contractType)) {//物资租赁合同
            return "BT200608000000001";
        }
        return "";
    }

    /**
     * 获取billCode
     * @return
     */
    public String getBillCode(String code){
        CommonResponse<String> billCode = billCodeApi.getCodeBatchByRuleCode(code, InvocationInfoProxy.getTenantid());
        if(billCode.isSuccess()) {
            return billCode.getData();
        }else{
            throw new BusinessException("网络异常， 编码生成失败， 请稍后再试");
        }
    }
}
