package com.ejianc.business.zdsstore.service.handler;

import cn.hutool.core.util.StrUtil;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.ejianc.business.zdsstore.bean.FlowEntity;
import com.ejianc.business.zdsstore.bean.InOutEntity;
import com.ejianc.business.zdsstore.consts.StoreCommonConsts;
import com.ejianc.business.zdsstore.service.IFlowService;
import com.ejianc.business.zdsstore.service.IInOutService;
import com.ejianc.business.zdsstore.service.IStoreService;
import com.ejianc.business.zdsstore.service.ISurplusService;
import com.ejianc.business.zdsstore.util.StoreLockUtil;
import com.ejianc.business.zdsstore.util.StoreManageUtil;
import com.ejianc.business.zdsstore.vo.*;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.util.ComputeUtil;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @author songlx
 * @version 1.0
 * @description: 退库处理器
 * @date 2022/1/25
 */
@Service
public class BackStoreHandler implements IStoreManageHandler {

    @Autowired
    IFlowService flowService;

    @Autowired
    ISurplusService surplusService;

    @Autowired
    IInOutService inOutService;
    @Autowired
    IStoreService storeService;

    /**
     * @param storeManageVO
     * @description: 物资退库逻辑, 审批通过(已提交)推送负数出库
     * 负数出库
     * 1 生成一条负数出库,用于出库数量抵消
     * 2 释放入库剩余数量
     * 3 更新出入库关系
     * 4 更新库存 增量
     * @return: com.ejianc.framework.core.response.CommonResponse<com.ejianc.business.store.vo.StoreManageVO>
     * @author songlx
     * @date: 2022/1/25
     */
    @Override
    public CommonResponse<StoreManageVO> handle(StoreManageVO storeManageVO) {
        //设置仓库属性
        storeManageVO= storeService.setStoreAttr(storeManageVO);
        Long storeId = storeManageVO.getStoreId();
        List<FlowVO> flowVOList = storeManageVO.getFlowVOList();
        SurplusUpdateVO surplusUpdateVO = StoreManageUtil.getSurplusUpdateVO(storeId, flowVOList, false);

        Boolean lock = StoreLockUtil.getLock(storeId);
        if (!lock) {
            throw new BusinessException("获取仓库锁网络异常，请刷新重试！");
        }
        try {
            List<Long> inOutIds = flowVOList.stream().map(FlowVO::getInOutId).collect(Collectors.toList());
            //获取出入库关系
            QueryWrapper<InOutEntity> inOutEntityQueryWrapper = new QueryWrapper<>();
            inOutEntityQueryWrapper.in("id", inOutIds);
            List<InOutEntity> inOutEntities = inOutService.list(inOutEntityQueryWrapper);

            Map<Long, List<InOutEntity>> inOutMap = inOutEntities.stream().collect(
                    Collectors.groupingBy(i -> i.getId(), Collectors.toList()));

            //查询入库占用
            List<Long> inFlowIds = inOutEntities.stream().map(InOutEntity::getInFlowId).collect(Collectors.toList());
            QueryWrapper<FlowEntity> inFlowQueryWrapper = new QueryWrapper<>();
            inFlowQueryWrapper.in("id", inFlowIds);
            List<FlowEntity> inFlowList = flowService.list(inFlowQueryWrapper);
            Map<Long, List<FlowEntity>> inFlowMap = inFlowList.stream().collect(Collectors.groupingBy(FlowEntity::getId));

            List<InOutEntity> updateInOutVOList = new ArrayList<>();
            List<FlowEntity> updateFlowVOList = new ArrayList<>();

            flowVOList.forEach(t -> {
                Long inOutId = t.getInOutId();
                List<InOutEntity> inOutEntities1 = inOutMap.get(inOutId);
                InOutEntity inOutEntity = inOutEntities1.get(0);
                List<FlowEntity> flowEntities = inFlowMap.get(inOutEntity.getInFlowId());
                FlowEntity flowEntity = flowEntities.get(0);

                BigDecimal returnStoreNum = t.getNum();
                inOutEntity.setReturnStoreNum(ComputeUtil.safeAdd(inOutEntity.getReturnStoreNum(), returnStoreNum));
                //出库数量= 原出库数量 - 退库数量
                //inOutEntity.setOutNum(ComputeUtil.safeSub(inOutEntity.getOutNum(), returnStoreNum));
                updateInOutVOList.add(inOutEntity);
                flowEntity.setReturnStoreFlag(StoreCommonConsts.YES);
                // 累计出库数量, 盈余数量, 盈余金额
                flowEntity.setOutNum(ComputeUtil.safeSub(flowEntity.getOutNum(), returnStoreNum));
                flowEntity.setSurplusNum(ComputeUtil.safeAdd(flowEntity.getSurplusNum(), returnStoreNum));
                flowEntity.setSurplusMny(ComputeUtil.safeMultiply(flowEntity.getSurplusNum(), flowEntity.getPrice()));
                flowEntity.setSurplusTaxMny(ComputeUtil.safeMultiply(flowEntity.getSurplusNum(), flowEntity.getTaxPrice()));
                flowEntity.setOutUseFlag(StoreCommonConsts.UseOutFlag.USEING);
                updateFlowVOList.add(flowEntity);
            });
            flowVOList.forEach(t -> {
                t.setNum(ComputeUtil.convertToMinusNumber(t.getNum()));
                t.setMny(ComputeUtil.safeMultiply(t.getNum(), t.getPrice()));
                t.setTaxMny(ComputeUtil.safeMultiply(t.getNum(), t.getTaxPrice()));
            });
            // 加上保存的负数的出库
            updateFlowVOList.addAll(BeanMapper.mapList(flowVOList, FlowEntity.class));
            boolean b = flowService.saveOrUpdateBatch(updateFlowVOList);
            //入库成功后更新库存
            if (b) {
                inOutService.saveOrUpdateBatch(updateInOutVOList);
                SurplusUpdateVO surplusUpdateVORet = surplusService.updateStoreSurplus(surplusUpdateVO, false, true);
                //更新后将这些物资最新库存返回
                storeManageVO.setSurplusVOList(surplusUpdateVORet.getSurplusVOList());
            }
        } catch (Exception e) {
            e.printStackTrace();
            throw new BusinessException("操作异常，请刷新重试！");
        } finally {
            if (lock) {
                StoreLockUtil.releaseLock(storeId);
            }
        }
        return CommonResponse.success(storeManageVO);
    }

    /**
     * @param storeManageVO
     * @description: 仓库退库回滚实现逻辑, 仓库退库弃审前调用
     * 1 校验对应入库盈余数量
     * 2 删除对应负数出库
     * 3 释放入库剩余数量
     * 4 更新出入库关系
     * 5 更新库存 增量
     * 若未使用则删除入库并逆向更新库存
     * @return: com.ejianc.framework.core.response.CommonResponse<com.ejianc.business.store.vo.StoreManageVO>
     * @author songlx
     * @date: 2022/1/25
     */
    @Override
    public CommonResponse<StoreManageVO> handleRollback(StoreManageVO storeManageVO) {
        Long sourceId = storeManageVO.getSourceId();
        Long storeId = storeManageVO.getStoreId();

        Boolean lock = StoreLockUtil.getLock(storeId);
        if (!lock) {
            throw new BusinessException("获取仓库锁网络异常，请刷新重试！");
        }
        try {
            QueryWrapper<FlowEntity> backFlowQueryWrapper = new QueryWrapper<>();
            backFlowQueryWrapper.eq("source_id", sourceId);
            backFlowQueryWrapper.in("store_id", storeId);
            List<FlowEntity> backFlowList = flowService.list(backFlowQueryWrapper);

            List<FlowVO> backFlowVOList = BeanMapper.mapList(backFlowList, FlowVO.class);
            SurplusUpdateVO surplusUpdateVO = StoreManageUtil.getSurplusUpdateVO(storeId, backFlowVOList, true);

            List<Long> inOutIds = backFlowVOList.stream().map(FlowVO::getInOutId).collect(Collectors.toList());

            //获取出入库关系
            QueryWrapper<InOutEntity> inOutEntityQueryWrapper = new QueryWrapper<>();
            inOutEntityQueryWrapper.in("id", inOutIds);
            List<InOutEntity> inOutEntities = inOutService.list(inOutEntityQueryWrapper);

            Map<Long, List<InOutEntity>> inOutMap = inOutEntities.stream().collect(
                    Collectors.groupingBy(i -> i.getId(), Collectors.toList()));

            //查询入库占用
            List<Long> inFlowIds = inOutEntities.stream().map(InOutEntity::getInFlowId).collect(Collectors.toList());
            QueryWrapper<FlowEntity> inFlowQueryWrapper = new QueryWrapper<>();
            inFlowQueryWrapper.in("id", inFlowIds);
            List<FlowEntity> inFlowList = flowService.list(inFlowQueryWrapper);
            Map<Long, List<FlowEntity>> inFlowMap = inFlowList.stream().collect(Collectors.groupingBy(FlowEntity::getId));
            List<InOutEntity> updateInOutVOList = new ArrayList<>();
            List<FlowEntity> updateFlowVOList = new ArrayList<>();

            HashMap<String, SurplusVO> surplusMap = new HashMap<>();
            for (FlowEntity t : backFlowList) {
                Long inOutId = t.getInOutId();
                List<InOutEntity> inOutEntities1 = inOutMap.get(inOutId);
                InOutEntity inOutEntity = inOutEntities1.get(0);
                List<FlowEntity> flowEntities = inFlowMap.get(inOutEntity.getInFlowId());
                FlowEntity flowEntity = flowEntities.get(0);

                BigDecimal surplusNum = flowEntity.getSurplusNum();
                BigDecimal returnStoreNumZS = ComputeUtil.safeSub(BigDecimal.ZERO ,t.getNum());
                //1 校验对应入库盈余数量
                if (ComputeUtil.isLessThan(surplusNum, returnStoreNumZS)) {
                    return CommonResponse.error("物资[名称:" + t.getMaterialName() + ",规格:" + StrUtil.emptyToDefault(t.getMaterialSpec(), "无") + ",品牌:" + StrUtil.emptyToDefault(t.getBrandName(), "无") +"]库存不足,请刷新后操作!");
                }


                inOutEntity.setReturnStoreNum(ComputeUtil.safeSub(inOutEntity.getReturnStoreNum(), returnStoreNumZS));
                //出库数量= 原出库数量 - 退库数量
                updateInOutVOList.add(inOutEntity);

                flowEntity.setReturnStoreFlag(ComputeUtil.isEmpty(inOutEntity.getReturnStoreNum()) ? StoreCommonConsts.NO : StoreCommonConsts.YES);
                // 累计出库数量, 盈余数量, 盈余金额
                flowEntity.setOutNum(ComputeUtil.safeAdd(flowEntity.getOutNum(), returnStoreNumZS));
                flowEntity.setSurplusNum(ComputeUtil.safeSub(flowEntity.getSurplusNum(), returnStoreNumZS));
                flowEntity.setSurplusMny(ComputeUtil.safeMultiply(flowEntity.getSurplusNum(), flowEntity.getPrice()));
                flowEntity.setSurplusTaxMny(ComputeUtil.safeMultiply(flowEntity.getSurplusNum(), flowEntity.getTaxPrice()));
                flowEntity.setOutUseFlag(StoreCommonConsts.UseOutFlag.USEING);
                updateFlowVOList.add(flowEntity);
                String key = t.getMaterialId() + "@" + t.getBrandId();
                SurplusVO vo = surplusMap.get(key);
                BigDecimal munusNum = ComputeUtil.convertToMinusNumber(returnStoreNumZS);
                BigDecimal munusMny = ComputeUtil.safeMultiply(munusNum, flowEntity.getPrice());
                BigDecimal munusTaxMny = ComputeUtil.safeMultiply(munusNum, flowEntity.getTaxPrice());
                if (vo == null) {
                    vo = new SurplusVO();
                    vo.setStoreId(storeId);
                    vo.setMaterialId(t.getMaterialId());
                    vo.setBrandId(t.getBrandId());
                    // 库存减少
                    vo.setSurplusNum(munusNum);
                    vo.setSurplusMny(munusMny);
                    vo.setSurplusTaxMny(munusTaxMny);

                } else {
                    vo.setSurplusNum(ComputeUtil.safeAdd(vo.getSurplusNum(), munusNum));
                    vo.setSurplusMny(ComputeUtil.safeAdd(vo.getSurplusMny(), munusMny));
                    vo.setSurplusTaxMny(ComputeUtil.safeAdd(vo.getSurplusTaxMny(), munusTaxMny));
                }
                surplusMap.put(key, vo);
            }
            //3 释放入库剩余数量
            boolean b = flowService.saveOrUpdateBatch(updateFlowVOList);
            //入库成功后更新库存
            if (b) {
                // 2 删除对应负数出库
                List<Long> backFlowIds = backFlowVOList.stream().map(FlowVO::getId).collect(Collectors.toList());
                flowService.removeByIds(backFlowIds);
                // 4 更新出入库关系
                inOutService.saveOrUpdateBatch(updateInOutVOList);
                List<Long> materialList = new ArrayList<>();
                List<Long> brandList = new ArrayList<>();
                for(String key : surplusMap.keySet()){
                    String[] arr = key.split("@");
                    materialList.add(Long.valueOf(arr[0]));
                    brandList.add(Long.valueOf(arr[1]));
                }
                surplusUpdateVO.setMaterialIds(materialList);
                surplusUpdateVO.setBrandIds(brandList);
                surplusUpdateVO.setSurplusVOList(new ArrayList<>(surplusMap.values()));
                // 5 更新库存 增量
                SurplusUpdateVO surplusUpdateVORet = surplusService.updateStoreSurplus(surplusUpdateVO, false, true);
                //更新后将这些物资最新库存返回
                storeManageVO.setSurplusVOList(surplusUpdateVORet.getSurplusVOList());
            }
        } catch (Exception e) {
            e.printStackTrace();
            throw new BusinessException("操作异常，请刷新重试！");
        } finally {
            if (lock) {
                StoreLockUtil.releaseLock(storeId);
            }
        }
        return CommonResponse.success(storeManageVO);
    }
}
