package com.ejianc.business.zdsstore.controller;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.zdsstore.bean.StoreEntity;
import com.ejianc.business.zdsstore.bean.SurplusEntity;
import com.ejianc.business.zdsstore.consts.StoreCommonConsts;
import com.ejianc.business.zdsstore.service.IStoreService;
import com.ejianc.business.zdsstore.service.ISurplusService;
import com.ejianc.business.zdsstore.vo.StoreExportVO;
import com.ejianc.business.zdsstore.vo.StoreVO;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.foundation.support.api.IBillTypeApi;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.collection.ListUtil;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.kit.time.DateFormatUtil;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.ExcelExport;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.io.Serializable;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 仓库表
 *
 * @author generator
 */
@Controller
@RequestMapping("store")
public class StoreController implements Serializable {
    private static final long serialVersionUID = 1L;

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private IBillTypeApi billTypeApi;
    @Autowired
    private IBillCodeApi billCodeApi;
    @Autowired
    private IOrgApi iOrgApi;

    //private static final String BILL_CODE = "Invoice_Open_Apply_Code";//此处需要根据实际修改

    @Autowired
    private IStoreService service;

    @Autowired
    private ISurplusService surplusService;

    /**
     * @Description saveOrUpdate 新增或者修改
     */
    @RequestMapping(value = "/saveOrUpdate", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<StoreVO> saveOrUpdate(@RequestBody StoreVO saveOrUpdateVO) {
        StoreEntity entity = BeanMapper.map(saveOrUpdateVO, StoreEntity.class);
        //当前项目仓库如果改为默认，则该项目下其他默认仓库改为非默认，一个项目下只能由一个默认仓库
        if(entity.getId()==null){
            QueryWrapper<StoreEntity> queryWrapper = new QueryWrapper<>();
            queryWrapper.eq("project_id", entity.getProjectId());
            queryWrapper.eq("name", entity.getName());
            StoreEntity one = service.getOne(queryWrapper);
            if(one != null){
                return CommonResponse.error("保存失败，该项目下已存在当前仓库名称，请勿重复添加！");
            }
        }
        if(entity.getState() == 0 && entity.getDefaultFlag() == 1){
            return CommonResponse.error("该仓库是默认仓库！");
        }
        if (entity.getDefaultFlag() == 1) {
            QueryWrapper<StoreEntity> queryWrapper = new QueryWrapper<>();
            queryWrapper.eq("project_id", entity.getProjectId());
            queryWrapper.eq("default_flag", "1");
            StoreEntity one = service.getOne(queryWrapper);
            if (one != null) {
                UpdateWrapper<StoreEntity> updateWrapper = new UpdateWrapper<>();
                updateWrapper.eq("id", one.getId());
                updateWrapper.set("default_flag", "0");
                service.update(updateWrapper);
            }
        }
        service.saveOrUpdate(entity, false);
        StoreVO vo = BeanMapper.map(entity, StoreVO.class);
        return CommonResponse.success("保存或修改单据成功！", vo);
    }

    /**
     * @param id
     * @Description queryDetail 查询详情
     */
    @RequestMapping(value = "/queryDetail", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<StoreVO> queryDetail(Long id) {
        StoreEntity entity = service.selectById(id);
        QueryWrapper<SurplusEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.select("IFNULL(sum(surplus_tax_mny), 0) as surplusTaxMny");
        queryWrapper.eq("store_id", id);
        List<SurplusEntity> list = surplusService.list(queryWrapper);
        if (CollectionUtils.isNotEmpty(list)) {
            entity.setTotalMny(list.get(0).getSurplusTaxMny());
        }

        StoreVO vo = BeanMapper.map(entity, StoreVO.class);
        return CommonResponse.success("查询详情数据成功！", vo);
    }

    /**
     * @Description delete 批量删除单据
     * @Param [ids]
     */
    @RequestMapping(value = "/delete", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<String> delete(@RequestBody List<StoreVO> vos) {
        if (ListUtil.isNotEmpty(vos)) {
            for (StoreVO vo : vos) {
                 CommonResponse<String> resp = service.checkStoreMaterialList(vo.getId());
                if(!resp.isSuccess()){
                    return CommonResponse.error("删除失败！"+resp.getMsg());
                }
            }
        }
        service.removeByIds(vos.stream().map(StoreVO::getId).collect(Collectors.toList()), true);
        return CommonResponse.success("删除成功！");
    }

    /**
     * @param param
     * @Description queryList 查询列表
     * @Return com.ejianc.framework.core.response.CommonResponse<java.lang.String>
     */
    @RequestMapping(value = "/queryList", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<IPage<StoreVO>> queryList(@RequestBody QueryParam param) {

        /** 模糊搜索配置字段示例 */
        List<String> fuzzyFields = param.getFuzzyFields();
        fuzzyFields.add("name");
        fuzzyFields.add("projectName");
        /** 租户隔离 */
        param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));

        if(OrgVO.ORG_TYPE_DEPARTMENT.toString().equals(InvocationInfoProxy.getOrgType())){
            param.getParams().put("orgId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getOrgId()));
        } else {
            param.getParams().put("parentOrgId", new Parameter(QueryParam.IN, iOrgApi.findChildrenByParentIdWithoutProjectDept(
                        InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        }

        IPage<StoreEntity> page = service.queryPage(param, false);
        IPage<StoreVO> pageData = new Page<>(page.getCurrent(), page.getSize(), page.getTotal());
        page.getRecords().forEach(item->{
            //查询仓库下含税库存余额
            QueryWrapper<SurplusEntity> queryWrapper = new QueryWrapper<>();
            queryWrapper.select("IFNULL(sum(surplus_tax_mny), 0) as surplusTaxMny");
            queryWrapper.eq("store_id", item.getId());
            List<SurplusEntity> list = surplusService.list(queryWrapper);
            if (CollectionUtils.isNotEmpty(list)) {
                item.setTotalMny(list.get(0).getSurplusTaxMny());
            }
        });
        pageData.setRecords(BeanMapper.mapList(page.getRecords(), StoreVO.class));

        return CommonResponse.success("查询列表数据成功！", pageData);
    }

    /**
     * 获取RPC数据
     * resp 返回值
     * isMustSuc 是否必须成功
     * errMsg 失败提示
     */
    private Object getRespData(CommonResponse<?> resp, boolean isMustSuc, String errMsg) {
        if (isMustSuc && !resp.isSuccess()) {
            throw new BusinessException(StringUtils.isNoneBlank(errMsg) ? errMsg : "调用Rpc服务失败");
        }
        return resp.getData();
    }


    /**
     * @param param
     * @Description 导出
     * @Return void
     */
    @RequestMapping(value = "/excelExport", method = RequestMethod.POST)
    @ResponseBody
    public void excelExport(@RequestBody QueryParam param, HttpServletResponse response) {
        /** 模糊搜索配置字段示例 */
        List<String> fuzzyFields = param.getFuzzyFields();
        fuzzyFields.add("name");
        fuzzyFields.add("projectName");
        param.getParams().put("s.tenant_id", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        param.setPageIndex(1);
        param.setPageSize(-1);
        /** 数据隔离 本下 没有组织orgId的删除下面代码 */
        param.getParams().put("s.orgId", new Parameter(QueryParam.IN, iOrgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        List<StoreEntity> list = service.queryStoreList(param);
        List<StoreExportVO> exportVOS = new ArrayList<>();
        list.forEach(vo -> {
            StoreExportVO exportVO = BeanMapper.map(vo, StoreExportVO.class);
            exportVO.setCreateTime(DateFormatUtil.formatDate("yyyy-MM-dd", vo.getCreateTime()));
            if (vo.getAttrFlag() == 1) {
                exportVO.setAttrFlagName("自购材");
            }
            if (vo.getAttrFlag() == 2) {
                exportVO.setAttrFlagName("甲供材");
            }
            if (StoreCommonConsts.YES.equals(vo.getState())) {
                exportVO.setStateName("启用");
            }
            if (StoreCommonConsts.NO.equals(vo.getState())) {
                exportVO.setStateName("停用");
            }
            if (StoreCommonConsts.YES.equals(vo.getDefaultFlag())) {
                exportVO.setDefaultFlagName("是");
            }
            if (StoreCommonConsts.NO.equals(vo.getDefaultFlag())) {
                exportVO.setDefaultFlagName("否");
            }
            exportVOS.add(exportVO);
        });
        Map<String, Object> beans = new HashMap<>();
        beans.put("records", exportVOS);
        ExcelExport.getInstance().export("storeListExport.xlsx", beans, response);
    }


    @RequestMapping(value = "/queryStoreByProjectId", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<StoreVO> queryStoreByProjectId(Long projectId) {
        StoreVO vo = new StoreVO();
        if (projectId != null) {
            QueryWrapper<StoreEntity> queryWrapper = new QueryWrapper<>();
            queryWrapper.eq("project_id", projectId);
            queryWrapper.eq("default_flag", StoreCommonConsts.YES);
            StoreEntity entity = service.getOne(queryWrapper);
            if (entity != null) {
                vo = BeanMapper.map(entity, StoreVO.class);
            }
        }
        return CommonResponse.success("查询详情数据成功！", vo);
    }


    /**
     * @param
     * @Description 参照
     * @Return void
     */
    @RequestMapping(value = "/refStoreData", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<IPage<StoreVO>> refStoreData(@RequestParam Integer pageNumber, @RequestParam Integer pageSize,
                                                       String condition,
                                                       String searchObject,
                                                       String searchText) {
        QueryParam param = new QueryParam();
        param.getFuzzyFields().add("name");
        param.setPageSize(pageSize);
        param.setPageIndex(pageNumber);
        param.setSearchText(searchText);
        param.setSearchObject(searchObject);
        /** 租户隔离 */
        param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        param.getParams().put("state", new Parameter(QueryParam.EQ, StoreCommonConsts.YES));

        //查询本下范围内日的合同
        Long orgId = InvocationInfoProxy.getOrgId();

        Long projectId = null;
        Long parentOrgId = null;


        if (StringUtils.isNotEmpty(condition)) {
            /** 处理condition */
            JSONObject _con = JSONObject.parseObject(condition);
            if (null != _con.get("projectId")) {
                projectId = Long.valueOf(_con.get("projectId").toString());
            }
            if(null != _con.get("orgId")){
                orgId = Long.valueOf(String.valueOf(_con.get("orgId")));
            }
            if(null != _con.get("parentOrgId")){
                parentOrgId = _con.getLongValue("parentOrgId");
            }
            if(null != _con.get("attrFlag")){
                param.getParams().put("attrFlag", new Parameter(QueryParam.IN, Arrays.asList(_con.getString("attrFlag").split(","))));
            }
        }

        if (null != projectId) {
            param.getParams().put("projectId", new Parameter(QueryParam.EQ, projectId));
            service.autoBuid(projectId, null);
        } else if (null != parentOrgId) {
            param.getParams().put("parentOrgId", new Parameter(QueryParam.EQ, parentOrgId));
            service.autoBuid(null, parentOrgId);
        } else {
            CommonResponse<List<OrgVO>> orgResp = iOrgApi.findChildrenByParentId(orgId);
            if (!orgResp.isSuccess()) {
                logger.error("分页查询失败，获取当前本下组织信息失败, {}", orgResp.getMsg());
                return CommonResponse.error("查询失败，获取组织信息失败！");
            }
            List<Long> orgIdList = orgResp.getData().stream().map(OrgVO::getId).collect(Collectors.toList());
            param.getParams().put("orgId", new Parameter(QueryParam.IN, orgIdList));
        }

        IPage<StoreEntity> page = service.queryPage(param, false);
        IPage<StoreVO> pageData = new Page<>(page.getCurrent(), page.getSize(), page.getTotal());
        pageData.setRecords(BeanMapper.mapList(page.getRecords(), StoreVO.class));

        return CommonResponse.success("查询参照数据成功！", pageData);
    }

    /**
     * @Description 参照
     * @param
     * @Return void
     */
    @RequestMapping(value = "/refStore", method = RequestMethod.GET)
    @ResponseBody
    public List<StoreVO> refStore(@RequestParam(required = false) String searchText,
                                  @RequestParam(required = false) String condition) {
        QueryParam param = new QueryParam();
        param.getFuzzyFields().add("name");
        param.setSearchText(searchText);
        /** 租户隔离 */
        param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        param.getParams().put("state", new Parameter(QueryParam.EQ, StoreCommonConsts.YES));

        //查询本下范围内日的合同
        Long orgId = InvocationInfoProxy.getOrgId();

        Long projectId = null;
        Long parentOrgId = null;


        if (StringUtils.isNotEmpty(condition)) {
            /** 处理condition */
            JSONObject _con = JSONObject.parseObject(condition);
            if (null != _con.get("projectId")) {
                projectId = Long.valueOf(_con.get("projectId").toString());
            }
            if(null != _con.get("orgId")){
                orgId = Long.valueOf(String.valueOf(_con.get("orgId")));
            }
            if(null != _con.get("parentOrgId")){
                parentOrgId = _con.getLongValue("parentOrgId");
            }
            if(null != _con.get("attrFlag")){
                param.getParams().put("attrFlag", new Parameter(QueryParam.IN, Arrays.asList(_con.getString("attrFlag").split(","))));
            }
        }

        if (null != projectId) {
            param.getParams().put("projectId", new Parameter(QueryParam.EQ, projectId));
            service.autoBuid(projectId, null);
        } else if (null != parentOrgId) {
            param.getParams().put("parentOrgId", new Parameter(QueryParam.EQ, parentOrgId));
            service.autoBuid(null, parentOrgId);
        } else {
            CommonResponse<List<OrgVO>> orgResp = iOrgApi.findChildrenByParentId(orgId);
            if (!orgResp.isSuccess()) {
                logger.error("分页查询失败，获取当前本下组织信息失败, {}", orgResp.getMsg());
                return null;
            }
            List<Long> orgIdList = orgResp.getData().stream().map(OrgVO::getId).collect(Collectors.toList());
            param.getParams().put("orgId", new Parameter(QueryParam.IN, orgIdList));
        }
        List<StoreEntity> storeEntities = service.queryList(param, false);
        return BeanMapper.mapList(storeEntities,StoreVO.class);
    }
}
