package com.ejianc.business.zdsstore.service.handler;

import cn.hutool.json.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.ejianc.business.zdsstore.bean.FlowEntity;
import com.ejianc.business.zdsstore.consts.InOutTypeEnum;
import com.ejianc.business.zdsstore.consts.StoreCommonConsts;
import com.ejianc.business.zdsstore.service.IFlowService;
import com.ejianc.business.zdsstore.service.IStoreService;
import com.ejianc.business.zdsstore.service.ISurplusService;
import com.ejianc.business.zdsstore.util.StoreLockUtil;
import com.ejianc.business.zdsstore.util.StoreManageUtil;
import com.ejianc.business.zdsstore.vo.FlowVO;
import com.ejianc.business.zdsstore.vo.StoreManageVO;
import com.ejianc.business.zdsstore.vo.SurplusUpdateVO;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @author songlx
 * @version 1.0
 * @description: 周转材入库
 * @date 2022/1/25
 */
@Service
public class TurnInStoreHandler implements IStoreManageHandler {

    @Autowired
    IFlowService flowService;

    @Autowired
    ISurplusService surplusService;
    @Autowired
    IStoreService storeService;


    /**
     * @param storeManageVO
     * @description: 周转材入库具体实现逻辑, 收料入库审批通过后调用
     * 正向入库
     * 1 直接保存入库流水
     * 2 更新库存 无则插入,有则增量更新
     * @return: com.ejianc.framework.core.response.CommonResponse<com.ejianc.business.store.vo.StoreManageVO>
     * @author songlx
     * @date: 2022/1/25
     */
    @Override
    public CommonResponse<StoreManageVO> handle(StoreManageVO storeManageVO) {
        Long storeId = storeManageVO.getStoreId();
        InOutTypeEnum inOutTypeEnum = storeManageVO.getInOutTypeEnum();
        List<FlowVO> flowVOList = storeManageVO.getFlowVOList();
        SurplusUpdateVO surplusUpdateVO = StoreManageUtil.getSurplusUpdateVO(storeId, flowVOList, false);
        Boolean lock = StoreLockUtil.getLock(storeId);
        if (!lock) {
            throw new BusinessException("获取仓库锁网络异常，请刷新重试！");
        }

        try {
            // 入库初始化
            flowVOList.forEach(flowVO -> {
                flowVO.setOutNum(BigDecimal.ZERO);
                flowVO.setOutLockNum(BigDecimal.ZERO);
                flowVO.setSurplusNum(flowVO.getNum());
                // 收料入库净值 = 原值
                flowVO.setNetMny(flowVO.getMny());
                flowVO.setNetTaxMny(flowVO.getTaxMny());
                flowVO.setShareMny(BigDecimal.ZERO);
                flowVO.setShareTaxMny(BigDecimal.ZERO);
                flowVO.setOutNetMny(BigDecimal.ZERO);
                flowVO.setOutNetTaxMny(BigDecimal.ZERO);
                // 收料入库原值单价 = 原值单价
                if(InOutTypeEnum.周转材收料入库.equals(inOutTypeEnum)){
                    flowVO.setPurchasePrice(flowVO.getPrice());
                    flowVO.setPurchaseTaxPrice(flowVO.getTaxPrice());
                    flowVO.setEstimatePrice(flowVO.getPrice());
                    flowVO.setEstimateTaxPrice(flowVO.getTaxPrice());
                }

            });

            List<FlowEntity> flowEntities = BeanMapper.mapList(flowVOList, FlowEntity.class);
            boolean b = flowService.saveOrUpdateBatch(flowEntities);
            //入库成功后更新库存
            if (b) {
                SurplusUpdateVO surplusUpdateVORet = surplusService.updateStoreSurplus(surplusUpdateVO, true, true);
                //更新后将这些物资最新库存返回
                storeManageVO.setSurplusVOList(surplusUpdateVORet.getSurplusVOList());
            }
        } catch (Exception e) {
            e.printStackTrace();
            throw new BusinessException("操作异常，请刷新重试！");
        } finally {
            if (lock) {
                StoreLockUtil.releaseLock(storeId);
            }
        }
        return CommonResponse.success(storeManageVO);
    }

    /**
     * @param storeManageVO
     * @description: 仓库入库回滚实现逻辑, 收料入库弃审前调用
     * 1 校验周传材料入库弃审校验是否出库或被摊销
     * 2 若使用,不能回滚
     * 若未使用则删除入库并逆向更新库存
     * @return: com.ejianc.framework.core.response.CommonResponse<com.ejianc.business.store.vo.StoreManageVO>
     * @author songlx
     * @date: 2022/1/25
     */
    @Override
    public CommonResponse<StoreManageVO> handleRollback(StoreManageVO storeManageVO) {
        Long storeId = storeManageVO.getStoreId();
        Long sourceId = storeManageVO.getSourceId();
        QueryWrapper<FlowEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("source_id", sourceId);
        queryWrapper.eq("in_out_flag", StoreCommonConsts.IN_OUT_TYPE_IN);
        List<FlowEntity> instoreList = flowService.list(queryWrapper);
        if(CollectionUtils.isEmpty(instoreList)){
            return CommonResponse.error("未查询到该入库单流水信息！");
        }

        List<FlowVO> flowVOList = BeanMapper.mapList(instoreList, FlowVO.class);
        SurplusUpdateVO surplusUpdateVO = StoreManageUtil.getSurplusUpdateVO(storeId, flowVOList, true);

        Boolean lock = StoreLockUtil.getLock(storeId);
        if (!lock) {
            throw new BusinessException("获取仓库锁网络异常，请刷新重试！");
        }
        //周传材料入库弃审校验是否出库或被摊销
        CommonResponse<JSONObject> res = null;
        try {
            res = flowService.validateTurnIsUseOutFlag(sourceId, storeId);
            //入库成功后更新库存
            if (res.isSuccess()) {
                SurplusUpdateVO surplusUpdateVORet = surplusService.updateStoreSurplus(surplusUpdateVO, true, true);
                //更新后将这些物资最新库存返回
                storeManageVO.setSurplusVOList(surplusUpdateVORet.getSurplusVOList());
                //删除入库流水
                List<Long> delListIds = flowVOList.stream().map(FlowVO::getId).collect(Collectors.toList());
                flowService.removeByIds(delListIds);
                return CommonResponse.success(storeManageVO);
            }
        } catch (Exception e) {
            e.printStackTrace();
            throw new BusinessException("操作异常，请刷新重试！");
        } finally {
            if (lock) {
                StoreLockUtil.releaseLock(storeId);
            }
        }
        return CommonResponse.error(res.getMsg());
    }
}
