package com.ejianc.business.zdsstore.service.handler;

import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.ejianc.business.zdsstore.bean.FlowEntity;
import com.ejianc.business.zdsstore.bean.InOutEntity;
import com.ejianc.business.zdsstore.bean.SurplusEntity;
import com.ejianc.business.zdsstore.consts.StoreCommonConsts;
import com.ejianc.business.zdsstore.service.*;
import com.ejianc.business.zdsstore.vo.*;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.util.ComputeUtil;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author songlx
 * @version 1.0
 * @description: 直接出库
 * @date 2022/1/25
 */
@Service
public class StraightOutStoreHandler implements IStoreManageHandler {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    IFlowService flowService;

    @Autowired
    IInOutService inOutService;

    @Autowired
    ISurplusService surplusService;
    @Autowired
    IStoreService storeService;

    @Autowired
    StoreManageService storeManageService;

    /**
     * @param storeManageVO
     * @description: 直接出库具体实现逻辑
     * 1 没有保存，直接生效
     * @return: com.ejianc.framework.core.response.CommonResponse<com.ejianc.business.store.vo.StoreManageVO>
     */
    @Override
    public CommonResponse<StoreManageVO> handle(StoreManageVO storeManageVO) {
        //设置仓库属性
        storeManageVO = storeService.setStoreAttr(storeManageVO);
        Long sourceId = storeManageVO.getSourceId();
        Long storeId = storeManageVO.getStoreId();

        List<FlowEntity> inFlowEntityList = new ArrayList<>();
        List<FlowVO> flowVOList = storeManageVO.getFlowVOList();
        CommonResponse<SurplusUpdateVO> res;
        try {
            //1 校验库存量 要依据正数的变化量来和库存余量校验
            res = surplusService.validateStoreSurplusByOutLock(storeManageVO);
            if (res.isSuccess()) {
                //先进先出占用计算
                UseCalculateVO useCalculateVO = surplusService.useCalculate(storeManageVO);
                if (useCalculateVO == null) {
                    return CommonResponse.success("材料出库明细数量没有变化,仓库无需处理!");
                }
                // 查询是否已有对应的出库流水, 有则覆盖,无则插入
                List<FlowVO> updateFlowVOList = useCalculateVO.getUpdateFlowVOList();
                QueryWrapper<FlowEntity> outFlowWrapper = new QueryWrapper<>();
                outFlowWrapper.eq("source_id", sourceId);
                outFlowWrapper.eq("in_out_type", storeManageVO.getInOutTypeEnum().getInOutType());
                List<FlowEntity> outstoreList = flowService.list(outFlowWrapper);
                if (CollectionUtils.isNotEmpty(outstoreList)) {
                    Map<Long, List<FlowEntity>> outstoreMap = outstoreList.stream().collect(Collectors.groupingBy(FlowEntity::getSourceDetailId));
                    flowVOList.forEach(
                            t -> {
                                Long sourceDetailId = t.getSourceDetailId();
                                List<FlowEntity> flowEntityList = outstoreMap.get(sourceDetailId);
                                if (CollectionUtils.isNotEmpty(flowEntityList)) {
                                    FlowEntity flowEntity = flowEntityList.get(0);
                                    t.setId(flowEntity.getId());
                                    t.setCreateTime(flowEntity.getCreateTime());
                                    t.setCreateUserCode(flowEntity.getCreateUserCode());
                                    t.setTenantId(flowEntity.getTenantId());
                                    t.setVersion(flowEntity.getVersion());
                                }
                            }
                    );
                }

                // 出库流水合并进来
                updateFlowVOList.addAll(flowVOList);
                if (CollectionUtils.isNotEmpty(updateFlowVOList)) {
                    List<FlowEntity> flowEntities = BeanMapper.mapList(updateFlowVOList, FlowEntity.class);
                    ArrayList<FlowEntity> updateFlowList = new ArrayList<>();
                    ArrayList<FlowEntity> delFlowList = new ArrayList<>();
                    flowEntities.forEach(n -> {
                        if ("del".equals(n.getRowState())) {
                            delFlowList.add(n);
                        } else {
                            updateFlowList.add(n);
                        }
                    });
                    if (CollectionUtils.isNotEmpty(delFlowList)) {
                        flowService.removeByIds(delFlowList.stream().map(FlowEntity::getId).collect(Collectors.toList()));
                    }
                    if (CollectionUtils.isNotEmpty(updateFlowList)) {
                        for(FlowEntity flowEntity : updateFlowList){
                            flowService.saveOrUpdate(flowEntity);
                        }
                    }
                }
                List<InOutVO> updateInOutVOList = useCalculateVO.getUpdateInOutVOList();
                if (CollectionUtils.isNotEmpty(updateInOutVOList)) {
                    List<InOutEntity> inOutEntities = BeanMapper.mapList(updateInOutVOList, InOutEntity.class);
                    //根据dr分组,因为有的需要删除,有的是修改
                    Map<Integer, List<InOutEntity>> list = inOutEntities.stream().collect(Collectors.groupingBy(InOutEntity::getDr));
                    List<InOutEntity> delList = list.get(1);
                    List<InOutEntity> saveOrUpdList = list.get(0);
                    if (CollectionUtils.isNotEmpty(delList)) {
                        inOutService.removeByIds(delList.stream().map(InOutEntity::getId).collect(Collectors.toList()));
                    }
                    if (CollectionUtils.isNotEmpty(saveOrUpdList)) {
                        for(InOutEntity inOutEntity : saveOrUpdList){
                            inOutService.saveOrUpdate(inOutEntity);
                        }
                    }

                }
                List<SurplusVO> updateSurplusVOList = useCalculateVO.getUpdateSurplusVOList();
                if (CollectionUtils.isNotEmpty(updateSurplusVOList)) {
                    List<SurplusEntity> surplusEntities = BeanMapper.mapList(updateSurplusVOList, SurplusEntity.class);
                    for(SurplusEntity surplusEntity : surplusEntities){
                        surplusService.saveOrUpdate(surplusEntity);
                    }
                }
            }else{
                return CommonResponse.error(res.getMsg());
            }
        } catch (Exception e) {
            logger.error(JSONObject.toJSONString(e));
            throw new BusinessException("操作异常，请刷新重试！");
        } finally {
          /*  if (lock) {
                StoreLockUtil.releaseLock(storeId);
            }*/
        }

        QueryWrapper<InOutEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("out_bill_id", sourceId);
        //生效状态 未生效
        queryWrapper.eq("effective_state", StoreCommonConsts.NO);
        List<InOutEntity> inOutEntityList = inOutService.list(queryWrapper);
        if (CollectionUtils.isNotEmpty(inOutEntityList)) {
            List<Long> inFlowList = inOutEntityList.stream().map(InOutEntity::getInFlowId).collect(Collectors.toList());
            List<FlowEntity> flowEntities = (List<FlowEntity>) flowService.listByIds(inFlowList);
            Map<Long, List<FlowEntity>> inFlowMap = flowEntities.stream().collect(Collectors.groupingBy(FlowEntity::getId));

            Map<String, SurplusEntity> surplusUpdateMap = new HashMap<>();
            QueryWrapper<SurplusEntity> surplusEntityQueryWrapper = new QueryWrapper<>();
            surplusEntityQueryWrapper.eq("store_id", storeId);
            surplusEntityQueryWrapper.and(q -> {
                for(InOutEntity f : inOutEntityList) {
                    q.or(iq -> iq.eq("brand_id", f.getBrandId()).eq("material_id", f.getMaterialId()));
                }
                return q;
            });

            List<SurplusEntity> surplusEntities = surplusService.list(surplusEntityQueryWrapper);
            //原来的库存
            surplusEntities.forEach(t -> surplusUpdateMap.put(t.getMaterialId() + "@" + t.getBrandId(), t));
            List<SurplusEntity> updateSurplusList = new ArrayList<>();
            inOutEntityList.forEach(t -> {
                //原锁定数量\金额
                BigDecimal effectOutNum = t.getOutLockNum();
                BigDecimal outNetMny = t.getOutNetMny();
                BigDecimal outNetTaxMny = t.getOutNetTaxMny();
                t.setEffectiveDate(new Date());
                t.setEffectiveState(StoreCommonConsts.YES);
                //生效后占用数量归集到出库数量,占用数量归于0
                t.setOutNum(effectOutNum);
                t.setOutLockNum(BigDecimal.ZERO);
                t.setOutUseFlag(StoreCommonConsts.UseOutFlag.USE_FINISH);
                //更新库存锁定,生效后 锁定数量减少
                SurplusEntity surplusVO = surplusUpdateMap.get(t.getMaterialId() + "@" + t.getBrandId());
                BigDecimal outLockNum = ComputeUtil.safeSub(surplusVO.getOutLockNum(), effectOutNum);
                surplusVO.setOutLockNum(outLockNum);
                BigDecimal outLockMny = ComputeUtil.safeSub(surplusVO.getOutLockMny(), outNetMny);
                surplusVO.setOutLockMny(outLockMny);
                BigDecimal outLockTaxMny = ComputeUtil.safeSub(surplusVO.getOutLockTaxMny(), outNetTaxMny);
                surplusVO.setOutLockMny(outLockTaxMny);

                updateSurplusList.add(surplusVO);
                //入库的锁定变成出库
                List<FlowEntity> flowVOS = inFlowMap.get(t.getInFlowId());
                FlowEntity flowEntity = flowVOS.get(0);
                BigDecimal _outLockNum = ComputeUtil.safeSub(flowEntity.getOutLockNum(), effectOutNum);
                flowEntity.setOutLockNum(_outLockNum);
                BigDecimal _outNum = ComputeUtil.safeAdd(flowEntity.getOutNum(), effectOutNum);
                flowEntity.setOutNum(_outNum);
                inFlowEntityList.add(flowEntity);
            });

            //出库单生效.则出库单生效状态由占用态变成生效态
            QueryWrapper<FlowEntity> outFlowQueryWrapper = new QueryWrapper<>();
            outFlowQueryWrapper.eq("source_id", sourceId);
            List<FlowEntity> outFlowList = flowService.list(outFlowQueryWrapper);
            outFlowList.forEach(t -> {
                t.setEffectiveDate(new Date());
                t.setEffectiveState(StoreCommonConsts.YES);
                inFlowEntityList.add(t);
            });

            flowService.saveOrUpdateBatch(inFlowEntityList);
            inOutService.saveOrUpdateBatch(inOutEntityList);
            surplusService.saveOrUpdateBatch(updateSurplusList);
        }
        return CommonResponse.success(storeManageVO);
    }

    /**
     * @param storeManageVO
     * @description: 仓库出库具体实现逻辑逆向实现, 主要调用:出库单删除时
     * @return: com.ejianc.framework.core.response.CommonResponse<com.ejianc.business.store.vo.StoreManageVO>
     * @author songlx
     * @date: 2022/1/25
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public CommonResponse<StoreManageVO> handleRollback(StoreManageVO storeManageVO) {
        CommonResponse<StoreManageVO> checkResponse = handleRollback(storeManageVO, true);
        if (!checkResponse.isSuccess()) {
            return CommonResponse.error(checkResponse.getMsg());
        }
        return handleRollback(storeManageVO, false);
    }

    @Transactional(rollbackFor = Exception.class)
    public CommonResponse<StoreManageVO> handleRollback(StoreManageVO storeManageVO, boolean outEffectiveON) {
        try {
            if (outEffectiveON) {
                // 出库单撤回校验：出库单与对应入库单单价不一致，出库单不允许撤回；
                CommonResponse<StoreManageVO> checkResponse = outStoreRollbackCheck(storeManageVO);
                if (!checkResponse.isSuccess()) {
                    return CommonResponse.error(checkResponse.getMsg());
                }
            }
            UseCalculateVO useCalculateVO = surplusService.outRollBackByInOut(storeManageVO, outEffectiveON);
            //3 出库后更新入库\出入关系\库存
            List<FlowVO> updateFlowVOList = useCalculateVO.getUpdateFlowVOList();
            if (CollectionUtils.isNotEmpty(updateFlowVOList)) {
                List<FlowEntity> flowEntities = BeanMapper.mapList(updateFlowVOList, FlowEntity.class);
                for(FlowEntity f : flowEntities) {
                    flowService.saveOrUpdate(f);
                }
                // 出库单的处理
                List<Long> sourceIdsForRollBack = storeManageVO.getSourceIdsForRollBack();
                QueryWrapper<FlowEntity> outFlowQueryWrapper = new QueryWrapper<>();
                outFlowQueryWrapper.in("source_id", sourceIdsForRollBack);
                List<FlowEntity> outFlowList = flowService.list(outFlowQueryWrapper);

                if (outEffectiveON) {
                    outFlowList.forEach(t -> {
                        t.setEffectiveDate(new Date());
                        t.setEffectiveState(StoreCommonConsts.NO);
                    });
                    //出库单生效弃审回滚.则出库单生效状态变成占用态
                    for(FlowEntity f : outFlowList) {
                        flowService.saveOrUpdate(f);
                    }
//                    flowService.saveOrUpdateBatch(outFlowList);
                } else {
                    //出库单自由态删除回滚.则删除出库单
                    List<Long> delListIds = outFlowList.stream().map(FlowEntity::getId).collect(Collectors.toList());
                    flowService.removeByIds(delListIds);
                }
            }
            List<InOutVO> updateInOutVOList = useCalculateVO.getUpdateInOutVOList();
            if (CollectionUtils.isNotEmpty(updateInOutVOList)) {
                List<InOutEntity> inOutEntities = BeanMapper.mapList(updateInOutVOList, InOutEntity.class);
                if (outEffectiveON) {
                    inOutEntities.forEach(t -> {
                        t.setEffectiveDate(new Date());
                        t.setEffectiveState(StoreCommonConsts.NO);
                    });
                    for(InOutEntity io : inOutEntities) {
                        inOutService.saveOrUpdate(io);
                    }
//                    inOutService.saveOrUpdateBatch(inOutEntities);
                } else {
                    List<Long> delListIds = inOutEntities.stream().map(InOutEntity::getId).collect(Collectors.toList());
                    inOutService.removeByIds(delListIds);
                }
            }
            List<SurplusVO> updateSurplusVOList = useCalculateVO.getUpdateSurplusVOList();
            if (CollectionUtils.isNotEmpty(updateSurplusVOList)) {
                List<SurplusEntity> surplusEntities = BeanMapper.mapList(updateSurplusVOList, SurplusEntity.class);
                for(SurplusEntity s : surplusEntities) {
                    surplusService.saveOrUpdate(s);
                }
//                surplusService.saveOrUpdateBatch(surplusEntities);
            }

            return CommonResponse.success(storeManageVO);
        } catch (Exception e) {
            logger.error("操作异常：", e);
            throw new BusinessException("操作异常，请刷新重试！");
        } finally {
//            if (lock) {
//                StoreLockUtil.releaseLock(storeId);
//            }
        }
    }

    /**
     * @param storeManageVO
     * @description: 出库单撤回校验：出库单与对应入库单单价不一致，出库单不允许撤回；
     * @return
     */
    public CommonResponse<StoreManageVO> outStoreRollbackCheck(StoreManageVO storeManageVO) {
        logger.info("storeManageVO: {}", JSONObject.toJSONString(storeManageVO, SerializerFeature.PrettyFormat));
        List<Long> sourceIdsForRollBack = storeManageVO.getSourceIdsForRollBack();
        QueryWrapper<InOutEntity> inOutEntityQueryWrapper = new QueryWrapper<>();
        inOutEntityQueryWrapper.in("out_bill_id", sourceIdsForRollBack);
        //根据出库单id查询出入库关系
        List<InOutEntity> inOutEntities = inOutService.list(inOutEntityQueryWrapper);
        for (InOutEntity inOut : inOutEntities) {
            //根据出库单对应的入库流水id查询入库流水
            QueryWrapper<FlowEntity> flowEntityQueryWrapper = new QueryWrapper<>();
            flowEntityQueryWrapper.eq("id", inOut.getInFlowId());
            FlowEntity one = flowService.getOne(flowEntityQueryWrapper);
            if (one != null) {
                //入库流水的单价与出库单单价不一致，不允许撤回
                if (!one.getTaxPrice().equals(inOut.getTaxPrice())) {
                    CommonResponse.error("该出库单与对应入库单单价不一致，不允许撤回!");
                }
            }
        }
        return CommonResponse.success();
    }

}
