package com.ejianc.business.zdssupplier.utils.enterprise.processor.impl;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.ejianc.business.zdssupplier.utils.enterprise.processor.ICompanyCheckProcessor;
import com.ejianc.business.zdssupplier.utils.enterprise.vo.BaseCompanyInfoVO;
import com.ejianc.business.zdssupplier.utils.enterprise.vo.tyc.TYCCertificateVO;
import com.ejianc.business.zdssupplier.utils.enterprise.vo.tyc.TYCEnterpriseInfoVO;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.util.HttpTookit;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import java.util.HashMap;
import java.util.Map;

/**
 * 天眼查工具类
 *
 * @author CJ
 * @Description:
 * @date 2024/3/13 17:03
 */
@Component
public class TYCProcessor implements ICompanyCheckProcessor {

    private static Logger logger = LoggerFactory.getLogger(TYCProcessor.class);

    /**
     * 默认企业基础信息获取地址
     */
    private static final String DEFAULT_BASE_INFO_URL = "http://open.api.tianyancha.com/services/open/ic/baseinfo/normal";
    /**
     * 默认企业资质证书获取地址
     */
    private static final String DEFAULT_CERTIFICATE_URL = "http://open.api.tianyancha.com/services/open/bq/qualification/2.0";

    private static final Integer DEFAULT_REQ_PAGE_SIZE = 20;

    @Value("${enterpriseCheck.tycToken:NULL}")
    private String tycToken;

    @Value("${enterpriseCheck.baseInfoReqUrl:"+DEFAULT_BASE_INFO_URL+"}")
    private String baseInfoReqUrl;

    @Value("${enterpriseCheck.certiReqUrl:"+DEFAULT_CERTIFICATE_URL+"}")
    private String certiReqUrl;


    private static void handlePageData(TYCEnterpriseInfoVO companyInfo, JSONArray items) {
        TYCCertificateVO tmpCerti = null;
        for(Object itemObj : items) {
            tmpCerti = new TYCCertificateVO();
            JSONObject certiJson = (JSONObject) itemObj;
            tmpCerti.transfer(tmpCerti, certiJson);
            companyInfo.getCertiList().add(tmpCerti);
        }
    }

    @Override
    public BaseCompanyInfoVO getCompanyInfo(String companyName, JSONObject extParams) {
        TYCEnterpriseInfoVO companyInfo = null;
        Map<String, String> headers = new HashMap<>();
        if(StringUtils.isBlank(tycToken)) {
            throw new BusinessException("请检查是否配置天眼查token，当前获取token为空！");
        }
        headers.put("Authorization", tycToken);


        Integer pageNum = 1;
        Map<String, Object> params = new HashMap<>();
        params.put("keyword", companyName);
        boolean hasNext = true;
        try {
            //查询企业基本信息
            String resultStr =  HttpTookit.get(baseInfoReqUrl, params, headers);
            JSONObject resultJson = JSONObject.parseObject(resultStr);
            companyInfo = new TYCEnterpriseInfoVO();
            if("ok".equals(resultJson.getString("reason"))) {
                JSONObject baseInfo = resultJson.getJSONObject("result");
                TYCEnterpriseInfoVO.transfer(companyInfo, baseInfo);
            } else {
                logger.info("请求企业【{}】企业基本信息失败，{}", companyName, JSONObject.toJSONString(resultJson, SerializerFeature.PrettyFormat));
                hasNext = false;
            }

            //查询企业建筑资质信息
            params.put("pageSize", DEFAULT_REQ_PAGE_SIZE);
            while (hasNext) {
                params.put("pageNum", pageNum);
                String certificateRs = HttpTookit.get(DEFAULT_CERTIFICATE_URL, params, headers);
                JSONObject certificateJson = JSONObject.parseObject(certificateRs);
                if("ok".equals(certificateJson.getString("reason"))) {
                    JSONObject itemResult = certificateJson.getJSONObject("result");
                    Integer totalItems =itemResult.getInteger("total");
                    JSONArray items = itemResult.getJSONArray("items");
                    if(null != items && items.size() > 0) {
                        logger.info("查询企业【{}】企业资质信息，处理第{}页, {}条数据", companyName, pageNum, items.size());
                        handlePageData(companyInfo, items);
                    } else {
                        logger.info("企业【{}】企业资质信息 处理数据完成！！！总共{}页数据！", companyName, pageNum);
                        hasNext = false;
                    }
                    if(pageNum * DEFAULT_REQ_PAGE_SIZE >= totalItems) {
                        hasNext = false;
                    }
                    pageNum++;
                } else {
                    hasNext = false;
                    logger.error("查询企业【{}】企业资质信息失败，{}",companyName,
                            JSONObject.toJSONString(certificateRs, SerializerFeature.PrettyFormat));
                }
            }
        } catch (Exception e) {
            logger.error("查询企业【{}】企业资质信息异常，", companyName, e);
        }

        return companyInfo;
    }
}
