package com.ejianc.business.plan.handler;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.ejianc.business.plan.bean.ExecPlanDetailEntity;
import com.ejianc.business.plan.bean.ExecPlanEntity;
import com.ejianc.business.plan.bean.TotalPlanDetailEntity;
import com.ejianc.business.plan.bean.TotalPlanEntity;
import com.ejianc.business.plan.cons.PlanConstant;
import com.ejianc.business.plan.service.IExecPlanDetailService;
import com.ejianc.business.plan.service.IExecPlanService;
import com.ejianc.business.plan.service.ITotalPlanDetailService;
import com.ejianc.business.plan.service.ITotalPlanService;
import com.ejianc.business.plan.utils.DateUtil;
import com.ejianc.business.progress.utils.TreeHelper2;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author yqls
 * @version 1.0
 * @description: 总计划推送执行计划
 * @date 2022/6/16
 */
@Service
public class TotalHandler implements IPlanHandler {
    private final Logger logger = LoggerFactory.getLogger(getClass());

    private static final String BILL_CODE = "ZJKJ_EXEC_PLAN";

    @Autowired
    private IBillCodeApi billCodeApi;

    @Autowired
    private ITotalPlanService service;

    @Autowired
    private ITotalPlanDetailService detailService;

    @Autowired
    private IExecPlanService execService;

    @Autowired
    private IExecPlanDetailService execDetailService;

    @Autowired
    private CommonHelper commonHelper;

    /**
     * 计划推送具体实现逻辑, 总计划审批通过后调用
     * 正向推送
     * 1 直接生成执行计划
     * 2 更新执行计划 无则插入,有则增量更新
     * @param id
     * @return
     */
    @Override
    public ExecPlanEntity handle(Long id) {
        TotalPlanEntity source = service.selectById(id);
        if(source == null) {
            throw new BusinessException("未查询到总计划！");
        }
        QueryWrapper<TotalPlanDetailEntity> ew = new QueryWrapper<TotalPlanDetailEntity>()
                .eq("progress_id", id).orderByAsc("tid");
        List<TotalPlanDetailEntity> sourceList = detailService.list(ew);
        // 重置主键和父级主键
        sourceList = TreeUtils.restParentId(sourceList);
        // 执行计划
        ExecPlanEntity data = execService.getOne(new QueryWrapper<ExecPlanEntity>()
                .eq("project_id", source.getProjectId()));
        Long execPlanId = data != null ? data.getId() : null;// 执行计划主键
        List<ExecPlanDetailEntity> detailList = new ArrayList<>();
        if(data != null){
            detailList = execDetailService.list(new QueryWrapper<ExecPlanDetailEntity>()
                    .eq("progress_id", execPlanId).orderByAsc("tid"));
        }
        // 加锁
        Boolean lock = PlanLockUtil.getLock(source.getProjectId());
        if (!lock) {
            throw new BusinessException("获取执行计划锁网络异常，请刷新重试！");
        }
        // 主表新增/更新
        ExecPlanEntity entity = new ExecPlanEntity();
        try {
            entity = BeanMapper.map(source, ExecPlanEntity.class);
            // 处理主表
            this.transferHVO(data, entity);
            execService.saveOrUpdate(entity);
            execPlanId = entity.getId();
            // 处理子表
            List<ExecPlanDetailEntity> saveList = this.transferSaveList(sourceList, detailList, execPlanId, source.getProjectId());
            if(CollectionUtils.isNotEmpty(saveList)){
                execDetailService.saveOrUpdateBatch(saveList);
            }
            // 处理删除
            List<Long> delIds = commonHelper.transferDelIds(sourceList, detailList, entity.getProjectId(), PlanConstant.TOTAL_PLAN);
            if(CollectionUtils.isNotEmpty(delIds)){
                execDetailService.removeByIds(delIds);
            }
            // 回写来源单据记录执行计划主键
            LambdaUpdateWrapper<TotalPlanEntity> lambd = new LambdaUpdateWrapper<>();
            lambd.set(TotalPlanEntity::getSourceId, execPlanId);
            lambd.eq(TotalPlanEntity::getId, id);
            service.update(lambd);
        } catch (Exception e) {
            logger.error("推送总计划id-{}给执行计划id-{} 异常，", id, execPlanId, e);
            throw new BusinessException(e.getMessage());
        } finally {
            if (lock) {
                PlanLockUtil.releaseLock(source.getProjectId());
            }
        }
        return entity;
    }

    /**
     * 子表转成新增/编辑集合
     * @param sourceList 来源单据明细
     * @param detailList 执行计划明细
     * @return
     */
    private List<ExecPlanDetailEntity> transferSaveList(List<TotalPlanDetailEntity> sourceList,
                List<ExecPlanDetailEntity> detailList, Long progressId, Long projectId) {
        List<ExecPlanDetailEntity> saveList = new ArrayList<>();
        Map<String, ExecPlanDetailEntity> detailMap = detailList.stream().filter(x->StringUtils.isNotEmpty(x.getStructCode()))
                .collect(Collectors.toMap(ExecPlanDetailEntity::getStructCode, x->x));
        // 获取来源和执行计划主键对应关系
        Map<Long, Long> idMap = commonHelper.getIdMap(sourceList, detailList);
        if(CollectionUtils.isEmpty(detailList)) {// 新增
            sourceList.forEach(source->{
                ExecPlanDetailEntity vo = BeanMapper.map(source, ExecPlanDetailEntity.class);
//                vo.setId(null);
                vo.setVersion(null);
                vo.setCreateUserCode(null);
                vo.setCreateTime(null);
                vo.setUpdateUserCode(null);
                vo.setUpdateTime(null);
                vo.setProgressId(progressId);
                vo.setSourceId(source.getProgressId());// 来源单据主键
                vo.setSourceBid(source.getSourceBid());// 来源单据明细主键
                vo.setPlanState(PlanConstant.TOTAL_PLAN);// 任务标识
                vo.setPlanStart(source.getStart());// 计划开始
                vo.setPlanFinish(source.getFinish());// 计划结束
                vo.setPredictStart(source.getStart());// 预测开始
                vo.setPredictFinish(source.getFinish());// 预测结束
                vo.setProjectId(projectId);// 项目主键
                // 限制日期默认等于开始日期
                if(vo.getConstraintDate() != null){
                    vo.setConstraintDate(DurationUtil.getConstraintDate(vo.getStart(), vo.getFinish(), vo.getConstraintType()));
                }
                // 如果限制类型是越早越好的，默认改为限制不得早于...开始
                if(vo.getConstraintType() == 0){
                    vo.setConstraintType(4);
                    vo.setConstraintDate(vo.getStart());
                }
                vo.setFixedDate(1);
                saveList.add(vo);
            });
        } else {// 编辑
            sourceList.forEach(source->{
                // 特殊的，结构码重复造成执行计划被覆盖
                Boolean same = this.dealSourceList(detailMap, source, sourceList);
                ExecPlanDetailEntity vo = BeanMapper.map(source, ExecPlanDetailEntity.class);
                if(detailMap.containsKey(source.getStructCode()) && !same){// 编辑覆盖
                    ExecPlanDetailEntity detail = detailMap.get(source.getStructCode());
                    vo.setId(detail.getId());
                    vo.setParentId(detail.getParentId());
                    vo.setActualStart(detail.getActualStart());
                    vo.setActualFinish(detail.getActualFinish());
                    vo.setEstimateFinish(detail.getEstimateFinish());
                    vo.setPredictStart(detail.getPredictStart());// 预测开始
                    vo.setPredictFinish(detail.getPredictFinish());// 预测结束
                    vo.setPercentComplete(detail.getPercentComplete());
                    vo.setFinishNum(detail.getFinishNum());
                    vo.setFinishState(detail.getFinishState());
                    vo.setLightType(detail.getLightType());
                    vo.setDiffValue(detail.getDiffValue());
                    vo.setDiffNum(detail.getDiffNum());
                    vo.setDiffType(detail.getDiffType());
                    vo.setDiffResson(detail.getDiffResson());
                    vo.setResourceNum(detail.getResourceNum());
                    vo.setCreateTime(detail.getCreateTime());
                    vo.setCreateUserCode(detail.getCreateUserCode());
                    vo.setVersion(detail.getVersion());
//                    vo.setBimIds(detail.getBimIds());
//                    vo.setBimDetailIds(detail.getBimDetailIds());
//                    vo.setState(detail.getState());
                } else {
                    vo.setParentId(idMap.get(vo.getParentId()));
                    vo.setCreateUserCode(null);
                    vo.setCreateTime(null);
                    vo.setVersion(null);
                }
                vo.setUpdateUserCode(null);
                vo.setUpdateTime(null);
                vo.setProgressId(progressId);
                vo.setSourceId(source.getProgressId());// 来源单据主键
                vo.setSourceBid(source.getSourceBid());// 来源单据明细主键
                vo.setPlanState(PlanConstant.TOTAL_PLAN);// 任务标识
                vo.setPlanStart(source.getStart());// 计划开始
                vo.setPlanFinish(source.getFinish());// 计划结束
                if(vo.getActualStart() == null){
                    vo.setPredictStart(source.getStart());// 预测开始
                    vo.setPredictFinish(source.getFinish());// 预测结束
                }
                JSONArray predecessorLink = JSONArray.parseArray(source.getPredecessorLink());
                if (predecessorLink != null && predecessorLink.size() > 0) {
                    for (int j = 0; j < predecessorLink.size(); j++) {
                        JSONObject json = predecessorLink.getJSONObject(j);
                        Long PredecessorUID = json.getLong("PredecessorUID");
                        if (idMap.containsKey(PredecessorUID)) {
                            json.put("PredecessorUID", idMap.get(PredecessorUID));
                        }
                        Long TaskUID = json.getLong("TaskUID");
                        if (idMap.containsKey(TaskUID)) {
                            json.put("TaskUID", idMap.get(TaskUID));
                        }
                    }
                    vo.setPredecessorLink(String.valueOf(predecessorLink));
                }
                JSONArray assignments = JSONArray.parseArray(source.getAssignments());
                if (assignments != null && assignments.size() > 0) {
                    for (int j = 0; j < assignments.size(); j++) {
                        JSONObject json = assignments.getJSONObject(j);
                        Long TaskUID = json.getLong("TaskUID");
                        if (idMap.containsKey(TaskUID)) {
                            json.put("TaskUID", idMap.get(TaskUID));
                        }
                    }
                    vo.setAssignments(String.valueOf(assignments));
                }
                vo.setProjectId(projectId);// 项目主键
                // 限制日期默认等于开始日期
                if(vo.getConstraintDate() != null){
                    vo.setConstraintDate(DurationUtil.getConstraintDate(vo.getStart(), vo.getFinish(), vo.getConstraintType()));
                }
                // 如果限制类型是越早越好的，默认改为限制不得早于...开始
                if(vo.getConstraintType() == 0){
                    vo.setConstraintType(4);
                    vo.setConstraintDate(vo.getStart());
                }
                vo.setFixedDate(1);
                // 偏差时间=预测完成时间-完成时间
                vo.setDiffValue(new BigDecimal(DateUtil.getBetweenDays(vo.getPredictFinish(), vo.getPlanFinish())));
                saveList.add(vo);
            });
        }
        return saveList;
    }

    private Boolean dealSourceList(Map<String, ExecPlanDetailEntity> detailMap, TotalPlanDetailEntity source, List<TotalPlanDetailEntity> filterList) {
        // 特殊的，结构码重复造成执行计划被覆盖
        if(detailMap.containsKey(source.getStructCode())){
            ExecPlanDetailEntity detail = detailMap.get(source.getStructCode());
            if(!detail.getSourceId().equals(source.getProgressId())){
                // 同级不能重复，如果重复取序号最大值加1
                Set<String> codeSet = detailMap.values().stream().filter(x->x.getParentId().equals(detail.getParentId())).
                        map(ExecPlanDetailEntity::getCode).filter(Objects::nonNull).collect(Collectors.toSet());
                Set<String> sourceSet = filterList.stream().filter(x->x.getParentId().equals(source.getParentId())).
                        map(TotalPlanDetailEntity::getCode).filter(Objects::nonNull).collect(Collectors.toSet());
                codeSet.addAll(sourceSet);
                if(codeSet.contains(source.getCode())){
                    Integer maxCode = codeSet.stream().filter(x->commonHelper.isInteger(x)).map(x->Integer.valueOf(x)).max(Integer::compare).orElse(0);
                    Integer code = maxCode + 1;
                    source.setTid(code);
                    source.setCode(String.valueOf(code));
                    // 结构码
                    String parentCode = detailMap.values().stream().filter(x->x.getId().equals(detail.getParentId())).
                            map(ExecPlanDetailEntity::getStructCode).findAny().orElse(null);
                    source.setStructCode(parentCode != null ? parentCode + "@@" + source.getCode() : source.getCode());
                    UpdateWrapper update = new UpdateWrapper();
                    update.set("tid", source.getTid());
                    update.set("code", source.getCode());
                    update.set("struct_code", source.getStructCode());
                    update.eq("id", source.getSourceBid());
                    detailService.update(update);
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * 处理主表
     * @param data
     * @param entity
     */
    private void transferHVO(ExecPlanEntity data, ExecPlanEntity entity) {
        if(data == null){
//                entity.setId(IdWorker.getId());
            entity.setId(null);
            CommonResponse<String> billCode = billCodeApi.getCodeBatchByRuleCode(BILL_CODE, InvocationInfoProxy.getTenantid());
            if (billCode.isSuccess()) {
                entity.setBillCode(billCode.getData());
            } else {
                throw new BusinessException("网络异常， 编码生成失败， 请稍后再试");
            }
            entity.setVersion(null);
            entity.setCreateUserCode(null);
            entity.setCreateTime(null);
            entity.setFillVersion(0);
        } else {
            entity.setId(data.getId());
            entity.setBillCode(data.getBillCode());
            entity.setVersion(data.getVersion());
            entity.setCreateUserCode(data.getCreateUserCode());
            entity.setCreateTime(data.getCreateTime());
            entity.setFillVersion(data.getFillVersion());
        }
        entity.setUpdateUserCode(null);
        entity.setUpdateTime(null);
    }

    /**
     * 计划推送回滚实现逻辑, 总计划弃审前调用
     * 1 校验计划是否被使用
     * 2 若使用,不能回滚
     * 若未使用则删除计划并逆向更新执行计划
     * @param id
     * @return
     */
    @Override
    public ExecPlanEntity handleRollback(Long id) {
        TotalPlanEntity source = service.selectById(id);
        if(source == null) {
            throw new BusinessException("未查询到总计划！");
        }
        QueryWrapper<TotalPlanDetailEntity> ew = new QueryWrapper<TotalPlanDetailEntity>()
                .eq("progress_id", id).orderByAsc("tid");
        List<TotalPlanDetailEntity> sourceList = detailService.list(ew);
        // 执行计划
        ExecPlanEntity data = execService.getOne(new QueryWrapper<ExecPlanEntity>()
                .eq("project_id", source.getProjectId()));
        if(data == null) {
            throw new BusinessException("未查询到项目下执行计划！");
        }
        List<ExecPlanDetailEntity> detailList = execDetailService.list(new QueryWrapper<ExecPlanDetailEntity>()
                .eq("progress_id", data.getId()).orderByAsc("tid"));
        // 处理删除
        List<Long> delIds = commonHelper.transferDelIds(sourceList, detailList, data.getProjectId(), PlanConstant.TOTAL_PLAN, null);
        execService.removeById(data.getId());
        if(CollectionUtils.isNotEmpty(delIds)){
            execDetailService.remove(new QueryWrapper<ExecPlanDetailEntity>().in("id", delIds));
        }
        return data;
    }

    @Override
    public ExecPlanEntity handleUpdate(Long id) {
        TotalPlanEntity source = service.selectById(id);
        if(source == null) {
            throw new BusinessException("未查询到总计划！");
        }
        QueryWrapper<TotalPlanDetailEntity> ew = new QueryWrapper<TotalPlanDetailEntity>().eq("progress_id", id).orderByAsc("tid");
        List<TotalPlanDetailEntity> sourceList = detailService.list(ew);
        // 执行计划
        ExecPlanEntity entity = execService.getOne(new QueryWrapper<ExecPlanEntity>() .eq("project_id", source.getProjectId()));
        Long execPlanId = entity != null ? entity.getId() : null;// 执行计划主键
        List<ExecPlanDetailEntity> detailList = execDetailService.list(new QueryWrapper<ExecPlanDetailEntity>().eq("progress_id", execPlanId).orderByAsc("tid"));
        // 加锁
        Boolean lock = PlanLockUtil.getLock(source.getProjectId());
        if (!lock) {
            throw new BusinessException("获取执行计划锁网络异常，请刷新重试！");
        }
        try {
            List<ExecPlanDetailEntity> saveList = new ArrayList<>();
            Map<String, ExecPlanDetailEntity> saveMap = new HashMap<>();
            Map<String, ExecPlanDetailEntity> detailMap = detailList.stream().filter(x->StringUtils.isNotEmpty(x.getStructCode()))
                    .collect(Collectors.toMap(ExecPlanDetailEntity::getStructCode, x->x));
            for(TotalPlanDetailEntity detail : sourceList){
                if(!detailMap.containsKey(detail.getStructCode())){
                    continue;
                }
                ExecPlanDetailEntity vo = detailMap.get(detail.getStructCode());
                if(vo.getEmployeeId() == null || !vo.getEmployeeId().equals(detail.getEmployeeId())){
                    vo.setEmployeeId(detail.getEmployeeId());
                    vo.setEmployeeName(detail.getEmployeeName());
                    saveList.add(vo);
                    saveMap.put(vo.getStructCode(), vo);
                }
                if(vo.getTaskLine() == null || !vo.getTaskLine().equals(detail.getTaskLine())){
                    vo.setTaskLine(detail.getTaskLine());
                    saveMap.put(vo.getStructCode(), vo);
                }
                if(vo.getNodeLevel() == null || !vo.getNodeLevel().equals(detail.getNodeLevel())){
                    vo.setNodeLevel(detail.getNodeLevel());
                    saveMap.put(vo.getStructCode(), vo);
                }
            }
            // 变更下级所有任务责任人
            saveList = saveList.stream().sorted(Comparator.comparing(x->x.getOutlineLevel(), Comparator.nullsFirst(Integer::compareTo).reversed())).collect(Collectors.toList());
            List<Map> tree = TreeHelper2.listMap2Tree2(BeanMapper.mapList(detailList, Map.class));
            for(ExecPlanDetailEntity save : saveList){
                if(save.getEmployeeId() != null){
                    List<String> childCodes = TreeHelper2.getAllChildList(tree, save.getStructCode()).stream().map(x->String.valueOf(x.get("structCode"))).collect(Collectors.toList());
                    detailList.stream().filter(x->childCodes.contains(x.getStructCode())).peek(x->{
                        if(saveMap.containsKey(x.getStructCode())){
                            x = saveMap.get(x.getStructCode());
                        }
                        x.setEmployeeId(save.getEmployeeId());
                        x.setEmployeeName(save.getEmployeeName());
                        saveMap.put(x.getStructCode(), x);
                    }).collect(Collectors.toList());
                }
            }

            if(CollectionUtils.isNotEmpty(saveMap.values())){
                execDetailService.saveOrUpdateBatch(new ArrayList<>(saveMap.values()));
            }
        } catch (Exception e) {
            logger.error("更新总计划id-{}给执行计划id-{} 异常，", id, execPlanId, e);
            throw new BusinessException(e.getMessage());
        } finally {
            if (lock) {
                PlanLockUtil.releaseLock(source.getProjectId());
            }
        }
        return entity;
    }
}
