package com.ejianc.business.labor.controller;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.labor.bean.AttendanceLogEntity;
import com.ejianc.business.labor.bean.TrainEntity;
import com.ejianc.business.labor.service.IAmapService;
import com.ejianc.business.labor.service.IAttendanceLogService;
import com.ejianc.business.labor.vo.AttendanceLogVO;
import com.ejianc.business.labor.vo.WorkRecordVO;
import com.ejianc.business.labor.vo.WorkerVO;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.foundation.support.api.IBillTypeApi;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.kit.collection.ListUtil;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.ExcelExport;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletResponse;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import static com.ejianc.framework.skeleton.template.BaseServiceImpl.changeToQueryWrapper;


/**
 * 考勤日志
 *
 * @author baipengyan
 * @version 1.0
 * @since JDK 1.8
 */
@RestController
@RequestMapping("attendanceLog")
public class AttendanceLogController implements Serializable {
	private static final long serialVersionUID = 5204994771029686422L;
	private static final String BILL_CODE = "Invoice_Open_Apply_Code";//此处需要根据实际修改
	private final Logger logger = LoggerFactory.getLogger(this.getClass());
	@Resource
	private IBillTypeApi billTypeApi;
	@Resource
	private IBillCodeApi billCodeApi;
	@Resource
	private IOrgApi iOrgApi;
	@Resource
	private IAttendanceLogService service;
	@Autowired
	private SessionManager sessionManager;
	@Resource
	private IAmapService amapService;


	/**
	 * 获取openid
	 *
	 * @param code 临时登录凭证
	 *
	 * @return {@link CommonResponse}<{@link JSONObject}>
	 */
	@GetMapping(value = "getOpenId")
	public CommonResponse<String> getOpenId(@RequestParam(value = "code") String code) {
		return CommonResponse.success("获取openid成功！", service.getOpenId(code));
	}


	/**
	 * 租户手机号绑定openid校验
	 *
	 * @param openid openid
	 *
	 * @return {@link CommonResponse}<{@link String}>
	 */
	@GetMapping(value = "checkBind")
	public CommonResponse<String> checkBind(@RequestParam(value = "openid") String openid) {
		return CommonResponse.success("租户手机号绑定openid校验成功！", service.checkBind(openid));
	}


	/**
	 * 发送验证码
	 *
	 * @param phone        手机号
	 * @param templateCode 短信模板编码
	 * @param signName     短信签名
	 *
	 * @return {@link CommonResponse}<{@link String}>
	 *
	 * @throws Exception 异常
	 */
	@GetMapping(value = "sendMessage")
	public CommonResponse<String> sendMessage(@RequestParam(value = "phone") String phone,
	                                          @RequestParam(value = "templateCode", required = false) String templateCode,
	                                          @RequestParam(value = "signName") String signName) throws Exception {
		return CommonResponse.success(service.sendMsg(phone, templateCode, signName));
	}


	/**
	 * 登录
	 *
	 * @param phone        手机号
	 * @param templateCode 短信模版
	 * @param validate     验证码
	 * @param openid       openid
	 *
	 * @return {@link CommonResponse}<{@link Void}>
	 */
	@GetMapping(value = "login")
	public CommonResponse<Void> login(@RequestParam(value = "phone") String phone,
	                                  @RequestParam(value = "templateCode") String templateCode,
	                                  @RequestParam(value = "validate") String validate,
	                                  @RequestParam(value = "openid") String openid) {
		service.login(phone, templateCode, validate, openid);
		return CommonResponse.success("登录成功！");
	}


	/**
	 * 获取用工记录
	 *
	 * @param projectId 项目id
	 * @param phone     电话号码
	 *
	 * @return {@link CommonResponse}<{@link WorkRecordVO}>
	 */
	@GetMapping(value = "fetchWorkRecord")
	public CommonResponse<WorkRecordVO> fetchWorkRecord(@RequestParam(value = "projectId") Long projectId,
	                                                        @RequestParam(value = "phone") String phone) {
		return CommonResponse.success("获取项目成功！", service.fetchWorkRecord(projectId, phone));
	}


	/**
	 * 获取项目
	 *
	 * @param phone 手机号
	 *
	 * @return {@link CommonResponse}<{@link List}<{@link JSONObject}>>
	 */
	@GetMapping(value = "fetchProject")
	public CommonResponse<List<JSONObject>> fetchProject(@RequestParam(value = "phone") String phone) {
		return CommonResponse.success("获取项目成功！", service.fetchProject(phone));
	}


	/**
	 * 校验人脸信息设置
	 *
	 * @param tenantId  租户id
	 * @param projectId 项目id
	 * @param phone     手机号
	 *
	 * @return {@link CommonResponse}<{@link String}>
	 */
	@GetMapping(value = "checkFaceDataSetting")
	public CommonResponse<String> checkFaceDataSetting(@RequestParam(value = "tenantId") Long tenantId,
	                                                   @RequestParam(value = "projectId") Long projectId,
	                                                   @RequestParam(value = "phone") String phone) {
		return service.checkFaceDataSetting(tenantId, projectId, phone);
	}


	/**
	 * 校验项目打卡权限
	 *
	 * @param tenantId  租户id
	 * @param projectId 项目id
	 * @param phone     手机号
	 *
	 * @return {@link CommonResponse}<{@link String}>
	 */
	@GetMapping(value = "checkPunchCardAuth")
	public CommonResponse<String> checkPunchCardAuth(@RequestParam(value = "tenantId") Long tenantId,
	                                                 @RequestParam(value = "projectId") Long projectId,
	                                                 @RequestParam(value = "phone") String phone) {
		service.checkPunchCardAuth(tenantId, projectId, phone);
		return CommonResponse.success("项目打卡权限校验成功！");
	}


	/**
	 * 校验考勤范围
	 *
	 * @param projectId   项目id
	 * @param destination 当前位置经纬度
	 *
	 * @return {@link CommonResponse}<{@link JSONObject}>
	 */
	@GetMapping(value = "checkDistance")
	public CommonResponse<JSONObject> checkDistance(@RequestParam(value = "projectId") Long projectId,
	                                                @RequestParam(value = "destination") String destination) {
		return service.checkDistance(projectId, destination);
	}


	/**
	 * 校验可用人脸质量
	 *
	 * @param jsonObject json对象
	 *
	 * @return {@link CommonResponse}<{@link String}>
	 */
	@PostMapping(value = "checkFaceQuality")
	public CommonResponse<String> checkFaceQuality(@RequestBody JSONObject jsonObject) {
		String imgBase64A = jsonObject.getString("imgBase64A");
		return service.checkFaceQuality(imgBase64A);
	}


	/**
	 * 校验人脸
	 *
	 * @param jsonObject json对象
	 *
	 * @return {@link CommonResponse}<{@link WorkerVO}>
	 */
	@PostMapping(value = "checkFace")
	public CommonResponse<WorkerVO> checkFace(@RequestBody JSONObject jsonObject) {
		Long tenantId = jsonObject.getLong("tenantId");
		Long projectId = jsonObject.getLong("projectId");
		String phone = jsonObject.getString("phone");
		String imgBase64A = jsonObject.getString("imgBase64A");
		return service.checkFace(tenantId, projectId, phone, imgBase64A);
	}


	/**
	 * 根据公司花名册id校验人脸
	 *
	 * @param jsonObject json对象
	 *
	 * @return {@link CommonResponse}<{@link WorkerVO}>
	 */
	@PostMapping(value = "checkFaceByWorkerId")
	public CommonResponse<WorkerVO> checkFaceByWorkerId(@RequestBody JSONObject jsonObject) {
		Long workerId = jsonObject.getLong("workerId");
		String imgBase64A = jsonObject.getString("imgBase64A");
		return service.checkFaceByWorkerId(workerId, imgBase64A);
	}


	/**
	 * 校验已存在的打卡记录
	 *
	 * @param tenantId      租户id
	 * @param projectId     项目id
	 * @param phone         手机号
	 * @param punchCardType 打卡类型：上班打卡、下班打卡
	 *
	 * @return {@link CommonResponse}<{@link List}<{@link AttendanceLogVO}>>
	 */
	@GetMapping(value = "checkExistedLog")
	public CommonResponse<List<AttendanceLogVO>> checkExistedLog(@RequestParam(value = "tenantId") Long tenantId,
	                                                             @RequestParam(value = "projectId") Long projectId,
	                                                             @RequestParam(value = "phone") String phone,
	                                                             @RequestParam(value = "punchCardType") String punchCardType) {
		return service.checkExistedLog(tenantId, projectId, phone, punchCardType);
	}


	/**
	 * 考勤打卡
	 *
	 * @param projectId      项目id
	 * @param workerId       公司花名册人员id
	 * @param punchCardType  打卡类型：上班打卡、下班打卡
	 * @param punchCardPlace 打卡地点
	 * @param remoteFlag     是否是异地打卡（1:是 0：否）
	 *
	 * @return {@link CommonResponse}<{@link AttendanceLogVO}>
	 */
	@GetMapping(value = "/punchCard")
	public CommonResponse<AttendanceLogVO> punchCard(@RequestParam(value = "projectId") Long projectId,
	                                                 @RequestParam(value = "workerId") Long workerId,
	                                                 @RequestParam(value = "punchCardType") String punchCardType,
	                                                 @RequestParam(value = "punchCardPlace") String punchCardPlace,
	                                                 @RequestParam(value = "remoteFlag") Integer remoteFlag) {
		return CommonResponse.success("考勤打卡成功！", service.punchCard(projectId, workerId, punchCardType, punchCardPlace, remoteFlag));
	}


	/**
	 * 查询详情
	 *
	 * @param id id
	 *
	 * @return {@link CommonResponse}<{@link AttendanceLogVO}>
	 */
	@GetMapping(value = "/queryDetail")
	public CommonResponse<AttendanceLogVO> queryDetail(Long id) {
		AttendanceLogEntity entity = service.selectById(id);
		AttendanceLogVO vo = BeanMapper.map(entity, AttendanceLogVO.class);
		return CommonResponse.success("查询详情数据成功！", vo);
	}


	/**
	 * @Description 导出原始考勤记录
	 * @param param
	 * @Return void
	 */
	@RequestMapping(value = "/recordExcelExport", method = RequestMethod.POST)
	@ResponseBody
	public void excelExport(@RequestBody QueryParam param, HttpServletResponse response) {
		List<String> fuzzyFields = param.getFuzzyFields();
		fuzzyFields.add("startWorkPlace");
		fuzzyFields.add("endWorkPlace");

		QueryWrapper<AttendanceLogEntity> queryWrapper = new QueryWrapper<>();

		/**
		 * 支持按月查询或者按范围查询
		 */
		if(param.getParams().get("punchCardDate") != null){
			String punchCardDate = param.getParams().get("punchCardDate").getValue().toString();
			if(punchCardDate.split(",").length == 1){
				param.getParams().remove("punchCardDate");

				queryWrapper = changeToQueryWrapper(param);
				queryWrapper.apply("date_format(punch_card_date,'%Y-%m') = {0}", punchCardDate);
			} else {
				queryWrapper = changeToQueryWrapper(param);
			}
		}
		List<AttendanceLogEntity> list = service.list(queryWrapper);
		Map<String, Object> beans = new HashMap<>();
		List<AttendanceLogVO> attendanceLogVOS = new ArrayList<>();
		if(ListUtil.isNotEmpty(list)){
			attendanceLogVOS = BeanMapper.mapList(list,AttendanceLogVO.class);
			for (AttendanceLogVO vo : attendanceLogVOS) {
				vo.setAttendanceType(vo.getCreateUserCode().equals("考勤机生成")?vo.getCreateUserCode():"公众号人脸识别");
				vo.setWorkPlace(vo.getStartWorkPlace() != null?vo.getStartWorkPlace():vo.getEndWorkPlace());
				vo.setPunchCardTime(vo.getStartWorkTime() != null?vo.getStartWorkTime():vo.getEndWorkTime());

			}
		}

		beans.put("records", attendanceLogVOS);
		ExcelExport.getInstance().export("Record-export.xlsx", beans, response);
	}

	/**
	 * 分页查询
	 *
	 * @param param 参数
	 *
	 * @return {@link CommonResponse}<{@link IPage}<{@link AttendanceLogVO}>>
	 */
	@PostMapping(value = "/queryPage")
	public CommonResponse<IPage<AttendanceLogVO>> queryPage(@RequestBody QueryParam param) {
		List<String> fuzzyFields = param.getFuzzyFields();
		fuzzyFields.add("startWorkPlace");
		fuzzyFields.add("endWorkPlace");

		QueryWrapper<AttendanceLogEntity> queryWrapper = new QueryWrapper<>();

		/**
		 * 支持按月查询或者按范围查询
		 */
		if(param.getParams().get("punchCardDate") != null){
			String punchCardDate = param.getParams().get("punchCardDate").getValue().toString();
			if(punchCardDate.split(",").length == 1){
				param.getParams().remove("punchCardDate");

				queryWrapper = changeToQueryWrapper(param);
				queryWrapper.apply("date_format(punch_card_date,'%Y-%m') = {0}", punchCardDate);
			} else {
				queryWrapper = changeToQueryWrapper(param);
			}
		}



		Page<AttendanceLogEntity> p = new Page<>(param.getPageIndex(), param.getPageSize());
		IPage<AttendanceLogEntity> page = service.page(p, queryWrapper);

		IPage<AttendanceLogVO> pageData = new Page<>(page.getCurrent(), page.getSize(), page.getTotal());
		pageData.setRecords(BeanMapper.mapList(page.getRecords(), AttendanceLogVO.class));

		return CommonResponse.success("查询列表数据成功！", pageData);
	}

	/**
	 * 内部员工：获取项目下所有在场的劳务人员
	 *
	 * @param param 参数
	 *
	 * @return {@link CommonResponse}<{@link IPage}<{@link WorkRecordVO}>>
	 */
	@PostMapping(value = "internalFetchWorkRecord")
	public CommonResponse<IPage<WorkRecordVO>> internalFetchWorkRecord(@RequestBody QueryParam param) {
		return CommonResponse.success("内部员工获取项目下所有在场的劳务人员成功！", service.internalFetchWorkRecord(param));
	}


	/**
	 * 内部员工：校验考勤范围
	 *
	 * @param projectId   项目id
	 * @param destination 当前位置经纬度
	 *
	 * @return {@link CommonResponse}<{@link JSONObject}>
	 */
	@GetMapping(value = "internalCheckDistance")
	public CommonResponse<JSONObject> internalCheckDistance(@RequestParam(value = "projectId") Long projectId,
	                                                        @RequestParam(value = "destination") String destination) {
		return service.internalCheckDistance(projectId, destination);
	}


	/**
	 * 内部员工：校验人脸
	 *
	 * @param jsonObject json对象
	 *
	 * @return {@link CommonResponse}<{@link WorkerVO}>
	 */
	@PostMapping(value = "internalCheckFace")
	public CommonResponse<WorkerVO> internalCheckFace(@RequestBody JSONObject jsonObject) {
		Long projectId = jsonObject.getLong("projectId");
		String imgBase64A = jsonObject.getString("imgBase64A");
		return service.internalCheckFace(projectId, imgBase64A);
	}


	/**
	 * 内部员工：校验已存在的打卡记录
	 *
	 * @param workerId      项目花名册人员id
	 * @param projectId     项目id
	 * @param punchCardType 打卡类型：上班打卡、下班打卡
	 *
	 * @return {@link CommonResponse}<{@link List}<{@link AttendanceLogVO}>>
	 */
	@GetMapping(value = "internalCheckExistedLog")
	public CommonResponse<List<AttendanceLogVO>> internalCheckExistedLog(@RequestParam(value = "workerId") Long workerId,
	                                                                     @RequestParam(value = "projectId") Long projectId,
	                                                                     @RequestParam(value = "punchCardType") String punchCardType) {
		return service.internalCheckExistedLog(workerId, projectId, punchCardType);
	}


	/**
	 * 内部员工：考勤打卡
	 *
	 * @param workerId       项目花名册人员id
	 * @param projectId      项目id
	 * @param punchCardType  打卡类型：上班打卡、下班打卡
	 * @param punchCardPlace 打卡地点
	 * @param remoteFlag     是否是异地打卡（1:是 0：否）
	 *
	 * @return {@link CommonResponse}<{@link AttendanceLogVO}>
	 */
	@GetMapping(value = "/internalPunchCard")
	public CommonResponse<AttendanceLogVO> internalPunchCard(@RequestParam(value = "workerId") Long workerId,
	                                                         @RequestParam(value = "projectId") Long projectId,
	                                                         @RequestParam(value = "punchCardType") String punchCardType,
	                                                         @RequestParam(value = "punchCardPlace") String punchCardPlace,
	                                                         @RequestParam(value = "remoteFlag") Integer remoteFlag) {
		return CommonResponse.success("考勤打卡成功！", service.internalPunchCard(workerId, projectId, punchCardType, punchCardPlace, remoteFlag));
	}


	/**
	 * 内部员工：分页查询
	 *
	 * @param param 参数
	 *
	 * @return {@link CommonResponse}<{@link IPage}<{@link AttendanceLogVO}>>
	 */
	@PostMapping(value = "/internalQueryPage")
	public CommonResponse<IPage<AttendanceLogVO>> internalQueryPage(@RequestBody QueryParam param) {
		List<String> fuzzyFields = param.getFuzzyFields();
		fuzzyFields.add("name");

//		// 租户隔离
//		param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));

		IPage<AttendanceLogEntity> page = service.queryPage(param);
		IPage<AttendanceLogVO> pageData = new Page<>(page.getCurrent(), page.getSize(), page.getTotal());
		pageData.setRecords(BeanMapper.mapList(page.getRecords(), AttendanceLogVO.class));

		return CommonResponse.success("查询列表数据成功！", pageData);
	}


	/**
	 * 班组长：获取项目下当前用户所在班组（通过班组长手机号与登录用户手机号校验）的在场用户
	 *
	 * @param param 参数
	 *
	 * @return {@link CommonResponse}<{@link IPage}<{@link WorkRecordVO}>>
	 */
	@PostMapping(value = "leaderFetchWorkRecord")
	public CommonResponse<IPage<WorkRecordVO>> leaderFetchWorkRecord(@RequestBody QueryParam param) {
		return CommonResponse.success("班组长获取项目下所有在场的劳务人员成功！", service.leaderFetchWorkRecord(param));
	}


	/**
	 * 班组长：校验考勤范围
	 *
	 * @param projectId   项目id
	 * @param destination 当前位置经纬度
	 *
	 * @return {@link CommonResponse}<{@link JSONObject}>
	 */
	@GetMapping(value = "leaderCheckDistance")
	public CommonResponse<JSONObject> leaderCheckDistance(@RequestParam(value = "projectId") Long projectId,
	                                                      @RequestParam(value = "destination") String destination) {
		return service.leaderCheckDistance(projectId, destination);
	}


	/**
	 * 班组长：校验人脸
	 *
	 * @param jsonObject json对象
	 *
	 * @return {@link CommonResponse}<{@link WorkerVO}>
	 */
	@PostMapping(value = "leaderCheckFace")
	public CommonResponse<WorkerVO> leaderCheckFace(@RequestBody JSONObject jsonObject) {
		Long projectId = jsonObject.getLong("projectId");
		String imgBase64A = jsonObject.getString("imgBase64A");
		String phone = jsonObject.getString("phone");
		return service.leaderCheckFace(projectId, imgBase64A,phone );
	}


	/**
	 * 班组长：校验已存在的打卡记录
	 *
	 * @param workerId      项目花名册人员id
	 * @param projectId     项目id
	 * @param punchCardType 打卡类型：上班打卡、下班打卡
	 *
	 * @return {@link CommonResponse}<{@link List}<{@link AttendanceLogVO}>>
	 */
	@GetMapping(value = "leaderCheckExistedLog")
	public CommonResponse<List<AttendanceLogVO>> leaderCheckExistedLog(@RequestParam(value = "workerId") Long workerId,
	                                                                   @RequestParam(value = "projectId") Long projectId,
	                                                                   @RequestParam(value = "punchCardType") String punchCardType) {
		return service.leaderCheckExistedLog(workerId, projectId, punchCardType);
	}


	/**
	 * 班组长：考勤打卡
	 *
	 * @param workerId       项目花名册人员id
	 * @param projectId      项目id
	 * @param punchCardType  打卡类型：上班打卡、下班打卡
	 * @param punchCardPlace 打卡地点
	 * @param remoteFlag     是否是异地打卡（1:是 0：否）
	 *
	 * @return {@link CommonResponse}<{@link AttendanceLogVO}>
	 */
	@GetMapping(value = "/leaderPunchCard")
	public CommonResponse<AttendanceLogVO> leaderPunchCard(@RequestParam(value = "workerId") Long workerId,
	                                                       @RequestParam(value = "projectId") Long projectId,
	                                                       @RequestParam(value = "punchCardType") String punchCardType,
	                                                       @RequestParam(value = "punchCardPlace") String punchCardPlace,
	                                                       @RequestParam(value = "remoteFlag") Integer remoteFlag) {
		return CommonResponse.success("考勤打卡成功！", service.leaderPunchCard(workerId, projectId, punchCardType, punchCardPlace, remoteFlag));
	}


	/**
	 * 班组长：分页查询
	 *
	 * @param param 参数
	 *
	 * @return {@link CommonResponse}<{@link IPage}<{@link AttendanceLogVO}>>
	 */
	@PostMapping(value = "/leaderQueryPage")
	public CommonResponse<IPage<AttendanceLogVO>> leaderQueryPage(@RequestBody QueryParam param) {
		List<String> fuzzyFields = param.getFuzzyFields();
		fuzzyFields.add("name");

//		// 租户隔离
//		param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));

		IPage<AttendanceLogEntity> page = service.queryPage(param);
		IPage<AttendanceLogVO> pageData = new Page<>(page.getCurrent(), page.getSize(), page.getTotal());
		pageData.setRecords(BeanMapper.mapList(page.getRecords(), AttendanceLogVO.class));

		return CommonResponse.success("查询列表数据成功！", pageData);
	}
}
